///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/adapt.h"
#include "rheolef/form.h"
#include "rheolef/field_component.h"
#include "rheolef/rheostream.h"
#include "rheolef/field_expr_v2_linear.h"

namespace rheolef {

// extern in adapt.cc
template<class T, class M>
field_basic<T,M>
hessian_criterion (
    const field_basic<T,M>&  uh0,
    const adapt_option_type& opts);

template<class T, class M>
field_basic<T,M>
proj (const field_basic<T,M>& uh, const std::string& approx = "P1");

// -----------------------------------------
// adapt
// -----------------------------------------
template<class T, class M>
geo_basic<T,M>
adapt_bamg (
    const field_basic<T,M>&  uh,
    const adapt_option_type& opts)
{
  using namespace std;
  typedef typename geo_basic<T,M>::size_type size_type;
  bool do_verbose  = iorheo::getverbose(clog);
  bool do_clean    = iorheo::getclean(clog);
  string command;
  // -----------------------------------------
  // 1a) sortie du maillage : nom-<i>.geo
  // -----------------------------------------
  const geo_basic<T,M>& omega = uh.get_geo();
  size_type i = omega.serial_number();
  string i_name = omega.name();
  string geo_name = i_name + ".geo";
  odiststream out;
  if (! dis_file_exists(geo_name) &&
      ! dis_file_exists(geo_name + ".gz")) {
    out.open (geo_name);
    out << omega;
    check_macro (out.good(), "adapt: file \"" << geo_name << "\"creation failed");
    out.close();
  }
  // ----------------------------------------
  // 1b) conversion geo : nom-<i>.bamg
  // ----------------------------------------
  string bamg_name = i_name + ".bamg";
  // -----------------------------------------
  // 2a) sortie du critere : nom-crit-<i>.field
  // -----------------------------------------
  string crit_name = i_name + "-crit.field";
  out.open (crit_name, "field");
  out << proj (uh);
  check_macro (out.good(), "adapt: file \"" << crit_name << "\"creation failed");
  out.close();
  // --------------------------------------------
  // 2b) conversion field : nom-crit-<i>.bb
  // --------------------------------------------
  string bb_name = i_name + "-crit.bb";
  command = "field -noverbose -bamg-bb " + crit_name + " > " + bb_name;
  if (do_verbose) derr << "! " << command << endl;
  check_macro (dis_system (command) == 0, "adapt: unix command failed");
  // ----------------------------------------
  // 3) run bamg in adapt mode:
  //     => nom-<i+1>.bamg
  // ----------------------------------------
  string bamgcad_file = omega.familyname() + ".bamgcad";
  check_macro (dis_file_exists(bamgcad_file), "adapt: missing \""<<bamgcad_file<<"\" file");
  size_type k = uh.get_space().degree();
  if (! uh.get_space().get_numbering().is_continuous()) k++; // uh is a grad of a solution
  Float error = opts.err;
  if (k > 1) {
    // when k > 1, decreses the interpolation error level for bamg, as computed when k=1
    error = pow (error, 2./(k+1));
  }
  string options =
       "-coef "     + ftos(opts.hcoef)
    + " -err  "     + ftos(error)
    + " -errg "     + ftos(opts.errg)
    + " -hmin "     + ftos(opts.hmin)
    + " -hmax "     + ftos(opts.hmax)
    + " -ratio "    + ftos(opts.ratio)
    + " -anisomax " + ftos(opts.anisomax)
    + " -nbv "      + itos(opts.n_vertices_max)
    + " -NbJacobi " + itos(opts.n_smooth_metric)
    + " -CutOff "   + ftos(opts.cutoff);
  if (opts.splitpbedge) {
    options += " -splitpbedge ";
  }
  if (opts.thetaquad != numeric_limits<Float>::max()) {
    options += " -thetaquad " + ftos(opts.thetaquad);
  }
  options += " " + opts.additional;
  string i1_name = omega.familyname() + "-" + itos(i+1);
  command = "bamg "
    + options
    + " -b " + bamg_name
    + " -Mbb " + bb_name
    + " -o " + i1_name + ".bamg 1>&2";
  if (do_verbose) derr << "! " << command << endl;
  check_macro (dis_system (command) == 0, "adapt: command failed");
  // ----------------------------------------
  // 4) conversion : nom-crit-<i>.bamg
  // ----------------------------------------
  string dmn_name = omega.familyname() + ".dmn";
  command = "bamg2geo -" + omega.coordinate_system_name()
            + " " + dmn_name + " " + i1_name + ".bamg | gzip -9 > "
            + i1_name + ".geo.gz";
  if (do_verbose) derr << "! " << command << endl;
  check_macro (dis_system (command) == 0, "adapt: command failed");
  // ----------------------------------------
  // 5) chargement  nom-<i+1>.geo
  // ----------------------------------------
  idiststream in (i1_name, "geo");
  geo_basic<T,M> new_omega;
  in >> new_omega;
  new_omega.set_name (omega.familyname());
  new_omega.set_serial_number (i+1);
  return new_omega;
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
#define _RHEOLEF_instanciation(T,M) 							\
template geo_basic<T,M> adapt_bamg (const field_basic<T,M>&, const adapt_option_type&);

_RHEOLEF_instanciation(Float,sequential)
#ifdef _RHEOLEF_HAVE_MPI
_RHEOLEF_instanciation(Float,distributed)
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
