%{
// This file is part of Rheolef.
//
// Copyright (C) 2000-2009 Pierre Saramito 
//
// Rheolef is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Rheolef is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Rheolef; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// lexer for qmg mesh files
// =========================================================================
static slist<slist<qmg::tetra>    >::iterator p_tetra_list;
static slist<slist<qmg::triangle> >::iterator p_triangle_list;
static slist<slist<qmg::edge>     >::iterator p_edge_list;
static slist<slist<qmg::vertex>   >::iterator p_vertex_list;
static unsigned int		              dom_dim = 0;
%}
%union  { 
        int             empty;
        int 	        int_value;
        size_type       uint_value;
        double          float_value;
        char*           string_value;
}
%token<empty>			MESH2D
%token<empty>			MESH3D
%token<int_value>		VOLUME_DOMAIN_NAMES
%token<int_value>		FACE_DOMAIN_NAMES
%token<int_value>		EDGE_DOMAIN_NAMES
%token<int_value>		VERTEX_DOMAIN_NAMES
%token<string_value>            IDENTIFIER
%token<int_value>               INT_NUMBER
%token<float_value>             FLOAT_NUMBER
%token<string_value>            STRING


%type<int_value>		uint
%type<int_value>		globalId
%type<int_value>		patchindex
%type<int_value>		n_domain
%type<int_value>		dimension_specifier

%type<float_value>		float
%type<float_value>		paramcoordU
%type<float_value>		paramcoordV

%type<empty>			vertexList2d
%type<empty>			vertexList3d
%type<empty>			edgeVlist
%type<empty>			edgeSlist
%type<empty>			surfaceVlist
%type<empty>			brepSurfaceList
%type<empty>			surfaceSlist
%type<empty>			chamberSlist
%type<empty>			asciiMesh
%type<empty>			asciiMesh2d
%type<empty>			asciiMesh3d
%type<empty>			value
%type<empty>			value_list
%type<empty>			all
%type<empty>			domain_name_list
%type<empty>			domain_name_list_list
%%
all : 	asciiMesh
	domain_name_list_list
	;

asciiMesh : asciiMesh2d
	|   asciiMesh3d
	;

asciiMesh2d : MESH2D
	'(' globalPropVal ')'
	'(' vertexList2d ')'
     	'(' brepVertexList ')'
	'(' brepEdgeList ')'
	'(' brepSurfaceList ')' 

	/* MESH2D : mesh_v2.01 2 2 */

			{ p_data -> _dim = 2; }
	;

asciiMesh3d : MESH3D
	'(' globalPropVal ')'
	'(' vertexList3d ')'
     	'(' brepVertexList ')'
	'(' brepEdgeList ')'
	'(' brepSurfaceList ')' 
	'(' brepChamberList ')' 

			{ p_data -> _dim = 3; }

	/* MESH3D : mesh_v2.01 3 3
     	 * The number of brep entity lists must be the intrinsic dimension plus 1. In
         * other words, if the mesh's intrinsic dimension is 1, then it must
      	 * possess a brepVertexList and a brepEdgeList but may not possess a
      	 * (optionals) brepSurfaceList nor a brepChamberList.
      	 */
	;

globalPropVal : /* empty! */

		{ trace_macro("globalPropVal");}

	| globalPropVal
	  IDENTIFIER value

	/* The property/value pairs are arbitrary strings. 
	 * If a property or value string contains spaces, 
	 * then it should be enclosed in an outer pair of parentheses.
     	 * Currently, the only global property value with significance to QMG 2.0
     	 * is geo_global_id.
	 */
	;

value	: IDENTIFIER		{$$=0;}
	| INT_NUMBER		{$$=0;}
	| '(' value_list ')'	{$$=0;}
	;
value_list: /* empty */		{$$=0;}
	| value_list value
	;

vertexList2d : /* empty */
	 		{ $$ =0;
			  trace_macro("vertexList: empty");
			}
	| vertexList2d	
	  uint float float
	 		{ $$ =0;
			  trace_macro("vertexList[2]: "<<$2<<" "<<$3<<" "<<$4);
			  p_data -> _x [$2] = point($3,$4);
			}
	;

vertexList3d : /* empty */
	 		{ $$ =0;
			  trace_macro("vertexList: empty");
			}
	| vertexList3d	
	  uint float float float
	 		{ $$ =0;
			  trace_macro("vertexList[3]: "<<$2<<" "<<$3<<" "<<$4<<" "<<$5);
			  p_data -> _x [$2] = point($3,$4,$5);
			}
     	/* This is a list of the n nodes in the mesh. Each entry consists of a
     	 * global ID, which must be a nonnegative integer, followed by either 2 or
     	 * 3 coordinates, which are real numbers. The number of coordinates is
     	 * equal to the embedded dimension (2 or 3). The global ID's do not have
     	 * to be in increasing order, but they must be distinct.]
	 */
	;
globalId : uint	{ $$ = $1; };
float : FLOAT_NUMBER | INT_NUMBER { $$ = $1; };
uint  : INT_NUMBER 
			{ 
			  if ($1 < 0) {
				warning_macro ("positive integer expected, "<<$1<<" founded"); 
			        qmg_error ("positive integer expected"); 
			  }
			  $$ = $1;
			}
	;
brepVertexList : /* empty */
	 		{ trace_macro("brepVertexList: empty");
			}
	| brepVertexList
	  '(' globalId ')' '(' ')'
	 		{ trace_macro("brepVertexList: "<<$3);
			  p_data -> _p.push_front(slist<qmg::vertex>());
			  p_vertex_list = p_data -> _p.begin();
			  p_vertex_list -> push_front(qmg::vertex($3));
			}

     	/* This is the list of mesh entities associated with the brep's vertices.
     	 * There is one globalId per brep vertex.
	 */
	;

brepEdgeList : /* empty */
	| brepEdgeList
	  '(' edgeVlist ')' 
	  '(' edgeSlist ')'

	/* This is the list of mesh entities associated with the brep's edges. In
     	 * particular, there is an edgeVlist and edgeSlist for each brep edge, so
     	 * m here denotes the number of brep edges.
	 */
	;

edgeVlist : /* empty */
			{ $$=0;
	 		  trace_macro("edgeVlist: empty"); 
			}
	| edgeVlist
	  uint	/* globalId */
	  uint	/* curveindex */
	  float		/* paramcoord in [0.1] */
			{ $$=0;
	 		  trace_macro("edgeVlist: "<<$2<<" "<<$3<<" "<<$4); 
			}
	/* This is the list of mesh nodes lying on a
     	 * particular brep edge. Each triple in this list specifies which mesh
     	 * node, which geometric curve within the brep edge, and the parametric
     	 * coordinate (a real number in [0,1]) of the mesh node.]
	 */
	;
edgeSlist : /* empty */
			{ $$=0;
	 		  trace_macro("edgeSlist: empty"); 
			  p_data -> _e.push_front(slist<qmg::edge>());
			  p_edge_list = p_data -> _e.begin();
			}
	| edgeSlist
	  globalId globalId
			{ $$=0;
	 		  trace_macro("edgeSlist: "<<$2<<" "<<$3); 
			  p_edge_list -> push_front(qmg::edge($2,$3));
			}

	/* This is a list of global node
     	 * ID's. They are arranged in pairs, for a total of r pairs. Each pair
     	 * indicates the endpoints of a mesh edge that lies on the brep edge. The
     	 * union of these mesh edges should equal the brep edge. Each global
     	 * vertex ID lying in this list must also occur in edgeVlist.]
	 */
	;

brepSurfaceList : /* empty */
			{$$=0;}
	| brepSurfaceList
	  '(' surfaceVlist ')' 
	  '(' surfaceSlist ')'
			{$$=0;}

	/* This is the list of mesh entities associated with the
     	 * brep's surfaces. In particular, there is a surfaceVlist and
     	 * surfaceSlist for each brep surface, so m here denotes the number of
     	 * brep surfaces.
	 */
	;

surfaceVlist : /* empty */
			{ $$=0;
	 		  trace_macro("surfaceVlist: empty"); 
			}
	| surfaceVlist
	  globalId patchindex paramcoordU paramcoordV
			{ $$=0;
	 		  trace_macro("surfaceVlist: "<<$2<<" "<<$3<<" "<<$4<<" "<<$5); 
			}
	;
     	/* This is the list of mesh nodes lying on a particular brep surface.
	 * If the embedded dimension is 2, then this list is required to be empty. 
     	 * If the embedded dimension is 3, then this list is divided into
     	 * four-tuples. Each four-tuple includes the global ID of the vertex, the
     	 * index of the geometric patch containing the mesh node, and the
     	 * parametric coordinates within the patch of the mesh node.
	 */

patchindex : uint;
paramcoordU : float;
paramcoordV : float;

surfaceSlist : /* empty */
			{ $$=0;
	 		  trace_macro("surfaceSlist: empty"); 
			  p_data -> _t.push_front(slist<qmg::triangle>());
			  p_triangle_list = p_data -> _t.begin();
			}
	| surfaceSlist
	  globalId globalId globalId
			{ $$=0;
	 		  trace_macro("surfaceSlist: "<<$2<<" "<<$3<<" "<<$4); 
			  p_triangle_list -> push_front(qmg::triangle($2,$3,$4));
			}
	
	/* This is a list of global
     	 * node ID's. They are arranged in triples, for a total of r triples. Each
     	 * triple indicates the endpoints of a mesh triangle that lies on the brep
     	 * surface. The union of these mesh triangles should equal the brep
     	 * surface. Each global vertex ID lying in this list must also occur in
     	 * surfaceVlist.
	 */
	;

brepChamberList : /* empty */
	| brepChamberList
	  '(' ')' '(' chamberSlist ')'

	/* This is the list of mesh entities associated with the brep's chambers (assuming
     	 * the intrinsic and embedded dimension are both 3). In particular, there
     	 * is a chamberSlist for each chamber, so m here denotes the number of
     	 * brep chambers.
	 */
	;

chamberSlist : /* empty */
			{ $$=0;
	 		  trace_macro("chamberSlist: empty"); 
			  p_data -> _T.push_front(slist<qmg::tetra>());
			  p_tetra_list = p_data -> _T.begin();
			}
	| chamberSlist
	  globalId globalId globalId globalId
			{ $$=0;
	 		  trace_macro("chamberSlist: "<<$2<<" "<<$3<<" "<<$4<<" "<<$5); 
			   p_tetra_list -> push_front(qmg::tetra($2,$3,$4,$5));
			}

	/* This is a list of global
     	 * node ID's. They are arranged in four-tuples, for a total of r
     	 * four-tuples. Each four-tuple indicates the endpoints of a mesh
     	 * tetrahedron that lies in the brep chamber. The union of these mesh
     	 * tetrahedra should equal the brep chamber.
	 */
	;

domain_name_list_list: /*empty */	{ $$ = 0; }
		| domain_name_list_list  domain_name_list
		;

domain_name_list:
	dimension_specifier		{ dom_dim = $1; }
	n_domain
	name_list
					{ $$ = 0; }
	;

dimension_specifier:
	  VOLUME_DOMAIN_NAMES
	| FACE_DOMAIN_NAMES
	| EDGE_DOMAIN_NAMES
	| VERTEX_DOMAIN_NAMES
	;

n_domain: INT_NUMBER
	;

name_list: /* empty */
	| name_list IDENTIFIER 
					{ p_data -> _name_list [dom_dim].push_front ($2); }
	;
%%

