///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/adapt-zz.h"
#include "rheolef/error-estimator-zz.h"
#include "rheolef/piola_algo_v1.h"
#include "rheolef/field-local-norm.h"
#include "rheolef/form.h"
using namespace std;
namespace rheolef {
// ---------------------------------------------------------
// part I. isotropic adaptation
// ---------------------------------------------------------
field h_local (const geo& omega) {
    space Xh (omega, "P0");
    form_diag m_P0 (Xh, "mass");
    size_t d = omega.dimension();
    // volume ratio between
    //         the isocela d-simplex with h length edge
    //     and the h^d cube
    // shape_coef(d) = volume(d-simplex)/h^d
    // shape_coef is largest => largest h_old => largest h_new => adapt loop does not reach precision...
    Float shape_coef = 1;
    switch (d) {
     case 1 : shape_coef = 1; break;
     case 2 : shape_coef = 2; break;
     default: shape_coef = ::sqrt(Float(3.0))*std::pow(6.0, 1./Float(3.0)); break; // TODO
    }
    field h_P0 = shape_coef*pow(field(m_P0), 1./d);
    return h_P0;
}
struct bound : unary_function<Float,Float> {
    bound (Float vmin1, Float vmax1) : vmin(vmin1), vmax(vmax1) {}
    Float operator() (const Float& h) { return ::min(vmax,::max(vmin,h)); }
    Float vmin, vmax;
};
field proj_and_smooth (field h, size_t n_smooth = 1) {
    space Xh (h.get_geo(), "P0");
    space Vh (h.get_geo(), "P1");
    form_diag inv_m_P1 = 1./form_diag(Vh,"mass");
    form_diag inv_m_P0 = form_diag(Xh,"inv_mass");
    form proj_P0_P1 = form(Xh,Vh,"mass");
    if (h.get_approx() == "P0") {
      h = inv_m_P1*(proj_P0_P1*h);
    }
    for (size_t i = 0; i < n_smooth; i++) {
      h = inv_m_P0*(proj_P0_P1.trans_mult(h));
      h = inv_m_P1*(proj_P0_P1*h);
    }
    return h;
}
// normalize eta_h with ||grad(uh)||_L2
field estim_normalized (const field& grad_uh, string method) {
    field eta_h = sqrt(estim2 (grad_uh, method));
    form m (grad_uh.get_space(), grad_uh.get_space(), "mass");
    Float norm_grad_uh = ::sqrt(m(grad_uh,grad_uh));
    if (1+norm_grad_uh == 1) norm_grad_uh = 1;
    return eta_h/norm_grad_uh;
}
//
//  eta_K ~= |grad(u-uh)| qui est constant sur l'element K
// pour l'adaptation isotrope, on utilise :
// si eta_K > err0 alors h_K diminue sinon augmente
//  h_new := (err0/eta)*h_old car l'erreur varie en O(h) pour P1 en norme H1
// on limite les variations entre h_new et h_old a` un facteur 2
//
field h_local_estim (const field& grad_uh, const adapt_option_type& options, string method, bool do_proj) {
    field eta_h = estim_normalized (grad_uh, method);
warning_macro ("eta_min            = " << eta_h.min());
warning_macro ("eta_max            = " << eta_h.max());
    size_t d         = grad_uh.dimension();
    size_t n_element = grad_uh.get_geo().size();
    size_t k         = grad_uh.get_space().degree() + 1;
    // err_coef is an adjustement coef (experimental)
    Float err_coef = 1;
    switch (d) {
     case 1 : err_coef = 1; break;
     case 2 : err_coef = 1; break;
     default: err_coef = 1; break;
    }
    Float eta_target = err_coef*options.err/::sqrt(1.0*n_element);
    field theta = pow(eta_h/eta_target, 1.0/k);
warning_macro ("theta_min            = " << theta.min());
warning_macro ("theta_max            = " << theta.max());
    Float ratio_old_to_new = 2.0;
    // on peut borner theta entre 0.5 et 2 ou bien entre ratio et 1/ratio
    theta = compose(bound(1./ratio_old_to_new, ratio_old_to_new), theta);
    field h_old = h_local (grad_uh.get_geo());
warning_macro ("hmin            = " << options.hmin);
    field h_new = compose(bound(options.hmin, options.hmax), options.hcoef*h_old/theta);
    if (do_proj) h_new = proj_and_smooth (h_new, options.n_smooth_metric);
    return h_new;
}
// ---------------------------------------------------------
// part II. anisotropic adaptation
// ---------------------------------------------------------

// evaluation from the current mesh:
// return the metric evaluation M such that
// the size in the direction v
// is h(v) = v'*M*v
// Rq. inverse that for bamg.
// an isotrope metric is: M(x) = h(x).I
// this function is provided for debug purpose
field h_local_aniso (const geo& omega)
{
  bool do_proj_p1 = true;
  Float fix_coef = ::sqrt(2.0)/1.144122805635369; // experimental...
  size_t d         = omega.dimension();
  space L0h_tens (omega, "P0", "tensor");
  field mh (L0h_tens);
  tensor DF_K, R_K, P_K;
  for (size_t K_idx = 0; K_idx < omega.size(); K_idx++) {
    const geo_element& K = omega.element (K_idx);
    set_piola_matrix (DF_K, omega, K);
    point H_K = DF_K.svd (R_K, P_K, d);
    H_K = fix_coef*H_K;
    tensor M_K = R_K*diag(H_K)*trans(R_K);
    mh.set_tensor_at (K_idx, M_K);
  }
  if (!do_proj_p1) {
    return mh;
  } else {
    space H1h_tens (omega, "P1", "tensor");
    field mh_P1 (H1h_tens);
    for (size_t i_comp = 0; i_comp < mh.n_component(); i_comp++) {
      mh_P1[i_comp] = proj_and_smooth (mh[i_comp], 0);
    }
    return mh_P1;
  }
}
field h_local_iso (const geo& omega)
{
  using namespace std;
  bool do_proj_p1 = true;
  Float fix_coef = ::sqrt(2.0)/1.144122805635369; // experimental...
  size_t d = omega.dimension();
  space L0h (omega, "P0");
  field h (L0h);
  for (size_t K_idx = 0; K_idx < omega.size(); K_idx++) {
    const geo_element& K = omega.element (K_idx);
    tensor DF_K, R_K, P_K;
    set_piola_matrix (DF_K, omega, K);
    point h_K = DF_K.svd (R_K, P_K, d);
    h_K = fix_coef*h_K;
    // since h_K[i] are sorted, the max is the first
    h.at(K_idx) = h_K [0];
  }
  if (!do_proj_p1) {
    return h;
  } else {
    space H1h (omega, "P1");
    field h_P1 = proj_and_smooth (h, 0);
    return h_P1;
  }
}
// estimation from the current solution:
// return the metric M such that
// the size in the direction v
// is h(v) = |v|/sqrt(v'*M*v)
// suitable for bamg.
// an isotrope metric is: M(x) = h(x)^{-2}.I
field h_local_estim_aniso (const field& grad_uh, const adapt_option_type& options, string method) {
warning_macro ("err = " << options.err);
warning_macro ("hmin = " << options.hmin);
warning_macro ("hmax = " << options.hmax);
warning_macro ("hcoef = " << options.hcoef);
warning_macro ("n_smooth = " << options.n_smooth_metric);
  field h_iso = h_local_estim (grad_uh, options, method, false);

  // TODO: what is the link between the directional H_K[j] and the
  //  directional error eta_K[j] ? eta_K(j) = C*{fix_coef*h_K(j)}**k ?
  // fix_coef : then the adaptation loop converge to the target error option.err
  size_t d = grad_uh.dimension();
  Float fix_coef = ::sqrt(2.0)/1.144122805635369; // Float fix_coef = 1.75;
  const Float ratio_old_to_new = 2;
  const Float ratio_aniso_max = ((d != 3) ? 1.0e6 : 100);
  size_t n_element = grad_uh.get_geo().size();
  size_t k         = grad_uh.get_space().degree() + 1;
  const geo& omega = grad_uh.get_geo();
  field eta_h_Pkd = estim_vector (grad_uh, method);
  space L0h_tens (omega, "P0", "tensor");
  field mh (L0h_tens);
  field ee_h (L0h_tens);
  for (size_t i = 0; i < d; i++) {
    for (size_t j = 0; j <= i; j++) {
      ee_h(i,j) = norm2_L2_local_scalar (eta_h_Pkd[i],eta_h_Pkd[j]);
    }
  }
#define USE_DEBUG
#ifdef USE_DEBUG
  field eh (L0h_tens);
#endif // USE_DEBUG

  // normalize the error estimate by |grad uh|_l2
  form m (grad_uh.get_space(), grad_uh.get_space(), "mass");
  Float norm_grad_uh = ::sqrt(m(grad_uh,grad_uh));
  if (1+norm_grad_uh == 1) norm_grad_uh = 1;
  Float eta_star = options.err*norm_grad_uh/::sqrt(1.0*d*n_element);
  for (size_t K_idx = 0; K_idx < n_element; K_idx++) {
    const geo_element& K = omega.element (K_idx);
    tensor G_K = ee_h.tensor_at (K_idx);
    tensor Q_K, Qt_K;
    point eta_K;
    const bool use_svd_when_2d = true;
    if (use_svd_when_2d && d == 2) {
      // TODO: eig when d=2 is bad...
      eta_K = sqrt(G_K.svd (Q_K, Qt_K, d));
    } else {
      eta_K = sqrt (G_K.eig (Q_K, d));
    }
    tensor DF_K, R_K, P_K;
    set_piola_matrix (DF_K, omega, K);
    point H_K = DF_K.svd (R_K, P_K, d);
    H_K = fix_coef*H_K;
    tensor M_K = R_K*diag(1.0/sqr(H_K))*trans(R_K);
    bound bnd (1./ratio_old_to_new, ratio_old_to_new);
    bound bnd_aniso (1./ratio_aniso_max, ratio_aniso_max);
    point H_K_star; // (h_star)^{-2}
    // h_star = alpha^{-2k}*h_old
    point q_K_0 = Q_K.col(0);
    Float h_K_0 = 1.0/::sqrt(dot(q_K_0, M_K*q_K_0));
    point stretch;
    stretch[0] = 1;
    for (size_t j = 1; j < d; j++) {
      if (eta_K[0] + 1 == 1) {
        stretch[j] = 1;
      } else {
        Float alpha_j = ::sqrt(eta_K[j]/eta_K[0]);
        stretch[j] = 1.0/bnd_aniso(::pow(alpha_j, 1.0/k));
        stretch[0] /= stretch[j];
        H_K_star[j] = stretch[j]*h_iso.at(K_idx);
      }
    }
    H_K_star[0] = stretch[0]*h_iso.at(K_idx);
    tensor M_K_star = Q_K*diag(1.0/sqr(H_K_star))*trans(Q_K);
    mh.set_tensor_at (K_idx, M_K_star);
#ifdef USE_DEBUG
#ifdef TO_CLEAN
    warning_macro ("Q_K = " << Q_K);
    warning_macro ("H_K_star = " << H_K_star);
    warning_macro ("1/sqr(H_K_star) = " << diag(1.0/sqr(H_K_star)));
    warning_macro ("M_K_star = " << M_K_star);
#endif // TO_CLEAN
    point mlog10_eta_K;
    for (size_t j = 0; j < d; j++) {
      if (eta_K[j] + 1 == 1) mlog10_eta_K[j] = 15;
      else mlog10_eta_K[j] = -::log10(eta_K[j]);
    }
    eh.set_tensor_at (K_idx, Q_K*diag(mlog10_eta_K)*trans(Q_K));
#endif // USE_DEBUG
  }
  space H1h_tens (omega, "P1", "tensor");
  field mh_P1 (H1h_tens);
  for (size_t i_comp = 0; i_comp < mh.n_component(); i_comp++) {
    mh_P1[i_comp] = proj_and_smooth (mh[i_comp], options.n_smooth_metric);
  }
#ifdef USE_DEBUG
  field eh_P1 (H1h_tens);
  for (size_t i_comp = 0; i_comp < eh.n_component(); i_comp++) {
    eh_P1[i_comp] = proj_and_smooth (eh[i_comp], 0);
  }
  string filename = omega.name() + "-err.mfield"; 
  ofstream err (filename.c_str());
  cerr << "! file " << filename << " created" << endl;
  err << catchmark ("ee") << eh_P1
      << catchmark ("eh") << eta_h_Pkd
      << catchmark ("h_iso") << h_iso
	;
#endif // USE_DEBUG
  return mh_P1;
}
}// namespace rheolef
