///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/input/MoveMode.h>
#include <core/viewport/input/XFormManager.h>
#include <core/data/DataSetManager.h>
#include <core/scene/animation/AnimManager.h>

namespace Core {

/******************************************************************************
* Is called when the xform operation begins.
******************************************************************************/
void MoveMode::startXForm()
{
	// Get the transformation for the first node in the selection set.
	if(DATASET_MANAGER.currentSelection()->empty()) return;
	SceneNode* masterNode = DATASET_MANAGER.currentSelection()->node(0);
	XFORM_MANAGER.getTransformationSystem(masterNode, translationSystem);

	// Use this master system as snapping system.
	viewport()->snapPoint(clickPoint, point1, translationSystem);
}

/******************************************************************************
* Is called during the operation.
******************************************************************************/
void MoveMode::doXForm()
{
    Point3 point2;
	viewport()->snapPoint(currentPoint, point2, translationSystem);

	// Get movement in world space.
	delta = translationSystem * (point2 - point1);

	// Apply transformation to selected nodes.
	applyXForm(DATASET_MANAGER.currentSelection()->nodes(), 1.0);
}

/******************************************************************************
* Applies the current transformation to the given nodes.
******************************************************************************/
void MoveMode::applyXForm(const QVector<SceneNode*>& nodeSet, FloatType multiplier)
{
	Q_FOREACH(SceneNode* node, nodeSet) {
		
		CHECK_OBJECT_POINTER(node->transformationController());
		
		// Get transformation system.
		AffineTransformation transformSystem;
		XFORM_MANAGER.getTransformationSystem(node, transformSystem);

		// Get parent's system.
		TimeInterval iv;
		const AffineTransformation& translationSys = node->parentNode()->getWorldTransform(ANIM_MANAGER.time(), iv);

		// Move node in parent's system.
		node->transformationController()->translate(ANIM_MANAGER.time(), delta * multiplier, translationSys.inverse());
	}
}

};
