"""Utilities related to the ARC infosystem."""

import ldap3
from arcnagios.infosys.ldaputils import LDAPEntry, ldap_entry_dn

# LDAP Helpers
# ============

def _get_str_opt(ent, attr):
    try:
        vs = ent[attr]
        if vs == []:
            return None
        if len(vs) == 1:
            return vs[0]
        raise ValueError('Expected single value for attribute %s in %s.'
                % (attr, ldap_entry_dn(ent)))
    except ldap3.core.exceptions.LDAPCursorError:
        return None


# EGIIS
# =====

EGIIS_REG_VALID = 0
EGIIS_REG_INVALID = 1
EGIIS_REG_PURGED = 2

_egiis_reg_status_of_string = {
    'VALID': EGIIS_REG_VALID,
    'INVALID': EGIIS_REG_INVALID,
    'PURGED': EGIIS_REG_PURGED,
}
_egiis_reg_status_to_string = ['VALID', 'INVALID', 'PURGED']

def egiis_reg_status_of_string(s):
    if not s in _egiis_reg_status_of_string:
        raise ValueError('Invalid Mds-Reg-status %s'%s)
    return _egiis_reg_status_of_string[s]

def egiis_reg_status_to_string(st):
    return _egiis_reg_status_to_string[st]

class MdsServiceLdap(LDAPEntry):
    """Instances of this class holds decoded representations of LDAP entries
    of class MdsServiceLdap."""

    structural_objectclass = 'MdsServiceLdap'
    strip_attribute_prefixes = ['Mds-Service-', 'Mds-']
    lowercase_all_attributes = True

    def __init__(self, server, ent):
        LDAPEntry.__init__(self, server, ent)
        # CHECKME: Mds-Service-protocol, Mds-Service-Ldap-*, and
        # Mds-Reg-status are multi-valued in the schema. Should they be?
        reg_status_str = _get_str_opt(ent, 'Mds-Reg-status')
        if reg_status_str is None:
            # FIXME: Mds-Reg-status is not required. What's the default?
            self.reg_status = None
        else:
            self.reg_status = egiis_reg_status_of_string(reg_status_str)


# ARIS
# ====

class NorduGridCluster(LDAPEntry):
    """Instances of this class holds decoded representations of LDAP entries
    of class nordugrid-cluster."""

    structural_objectclass = 'nordugrid-cluster'
    strip_attribute_prefixes = ['nordugrid-cluster-', 'Mds-']

class NorduGridQueue(LDAPEntry):
    """Instances of this class holds decoded representations of LDAP entries
    of class nordugrid-queue."""

    structural_objectclass = 'nordugrid-queue'
    strip_attribute_prefixes = ['nordugrid-queue-', 'Mds-']


# Glue Schemas
# ============
#
# Full list of structural object classes:
#   Glue-MDS.schema:
#       Mds
#   Glue-CORE.schema:
#       GlueService, GlueServiceData, GlueSite
#   Glue-SE.schema:
#       GlueSEAccessProtocol, GlueSEControlProtocol, GlueSL,
#       GlueSLLocalFileSystem, GlueSLRemoteFileSystem, GlueSLFile,
#       GlueSLDirectory, GlueVOInfo, GlueSA, GlueSE
#   Glue-CE.schema:
#       GlueCEJob, GlueCluster, GlueHost, GlueHostLocalFileSystem,
#       GlueHostRemoteFileSystem, GlueHostFile, GlueLocation, GlueVOView,
#       GlueSoftware, GlueSoftwareData, GlueCE, GlueSubCluster
#   Glue-CESEBind.schema:
#       GlueCESEBindGroup, GlueCESEBind

class GlueSite(LDAPEntry):
    structural_objectclass = 'GlueSite'

class GlueCluster(LDAPEntry):
    structural_objectclass = 'GlueCluster'

class GlueSubCluster(LDAPEntry):
    structural_objectclass = 'GlueSubCluster'

class GlueCE(LDAPEntry):
    structural_objectclass = 'GlueCE'

class GlueCESEBindGroup(LDAPEntry):
    structural_objectclass = 'GlueCESEBindGroup'

class GlueVOView(LDAPEntry):
    structural_objectclass = 'GlueVOView'

class GlueCESEBind(LDAPEntry):
    structural_objectclass = 'GlueCESEBind'

glue_class_map = dict(
    (c.structural_objectclass, c) for c in
    [GlueSite, GlueCluster, GlueSubCluster, GlueCE, GlueCESEBindGroup,
     GlueVOView, GlueCESEBind])
