/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var isProbability = require( '@stdlib/math/base/assert/is-probability' );
var isnan = require( '@stdlib/math/base/assert/is-nan' );
var exp = require( '@stdlib/math/base/special/exp' );
var ln = require( '@stdlib/math/base/special/ln' );


// MAIN //

/**
* Evaluates the moment-generating function (MGF) for a geometric distribution with success probability `p` at a value `t`.
*
* @param {number} t - input value
* @param {Probability} p - success probability
* @returns {number} evaluated MGF
*
* @example
* var y = mgf( 0.2, 0.5 );
* // returns ~1.569
*
* @example
* var y = mgf( 0.4, 0.5 );
* // returns ~2.936
*
* @example
* // Case: t >= -ln(1-p)
* var y = mgf( 0.8, 0.5 );
* // returns NaN
*
* @example
* var y = mgf( NaN, 0.0 );
* // returns NaN
*
* @example
* var y = mgf( 0.0, NaN );
* // returns NaN
*
* @example
* var y = mgf( -2.0, -1.0 );
* // returns NaN
*
* @example
* var y = mgf( 0.2, 2.0 );
* // returns NaN
*/
function mgf( t, p ) {
	var et;
	var q;
	if ( isnan( t ) || !isProbability( p ) ) {
		return NaN;
	}
	q = 1.0 - p;
	if ( t >= -ln( q ) ) {
		return NaN;
	}
	et = exp( t );
	return ( p * et ) / ( 1.0 - (q * et) );
}


// EXPORTS //

module.exports = mgf;
