/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iposition.h"

#include "ierror.h"
#include "iviewmodule.h"

//
//  iPosition functions
//
iPosition::iPosition(iViewModule *vm) : mViewModule(vm)
{
	IERROR_ASSERT(vm);
	mX[0] = mX[1] = mX[2] = 0.0;
	mBoxX[0] = mBoxX[1] = mBoxX[2] = 0.0;
}


iPosition::~iPosition()
{
}


void iPosition::UpdateBoxPosition() const
{
	int i;
	if(this->GetViewModule()->GetOpenGLCoordinates())
	{
		for(i=0; i<3; i++) mBoxX[i] = mX[i]; // so small, no need to use iTune
	}
	else
	{
		float boxSize = this->GetViewModule()->GetBoxSize();
		for(i=0; i<3; i++) mBoxX[i] = 0.5*(1.0+mX[i])*boxSize;
	}

}


void iPosition::UpdateOpenGLPosition()
{
	int i;
	if(this->GetViewModule()->GetOpenGLCoordinates())
	{
		for(i=0; i<3; i++) mX[i] = mBoxX[i];
	}
	else
	{
		float boxSize = this->GetViewModule()->GetBoxSize();
		for(i=0; i<3; i++) mX[i] = -1.0 + 2.0*mBoxX[i]/boxSize;
	}
}


void iPosition::PeriodicWrap()
{
	int i;
	//
	//  If x is in between -1 and 1, do nothing - this way x=1 remains 1, and not wrapped into x=-1
	//
	for(i=0; i<3; i++) if(mX[i]<-1.0 || mX[i]>1.0) mX[i] -= 2.0*floor(0.5*(1.0+mX[i]));
}


void iPosition::CutToBounds()
{
	int i;
	for(i=0; i<3; i++)
	{
		if(mX[i] < -1.0) mX[i] = -1.0;
		if(mX[i] >  1.0) mX[i] =  1.0;
	}
}

//
//  iDistance functions
//
iDistance::iDistance(iViewModule *vm, bool isSize) : mViewModule(vm), mIsSize(isSize)
{
	IERROR_ASSERT(vm)
	if(isSize) mX = 1.0; else mX = 0.0;
}


iDistance::~iDistance()
{
}


void iDistance::UpdateBoxDistance() const
{
	if(this->GetViewModule()->GetOpenGLCoordinates())
	{
		mBoxX = mX;
	}
	else
	{
		double offset = 1.0;
		if(mIsSize) offset = 0.0;
		mBoxX = 0.5*(offset+mX)*this->GetViewModule()->GetBoxSize();
	}

}


void iDistance::UpdateOpenGLDistance()
{
	if(this->GetViewModule()->GetOpenGLCoordinates())
	{
		mX = mBoxX;
	}
	else
	{
		double offset = 1.0;
		if(mIsSize) offset = 0.0;
		mX = -offset + 2.0*mBoxX/this->GetViewModule()->GetBoxSize();
	}
}


void iDistance::PeriodicWrap()
{
	if(!mIsSize) mX -= 2.0*floor(0.5*(1.0+mX));
}


void iDistance::CutToBounds()
{
	if(mX < -1.0) mX = -1.0;
	if(mX >  1.0) mX =  1.0;
}

