// -*- c-basic-offset: 4 -*-

/** @file ToolHelper.cpp
 *
 *  @author James Legg
 *  @author Darko Makreshanski
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public
 *  License along with this software. If not, see
 *  <http://www.gnu.org/licenses/>.
 *
 */ 


#ifdef _WIN32
#include "wx/msw/wrapwin.h"
#endif
#include "ToolHelper.h"
#include "Tool.h"
#include "GLPreviewFrame.h"
#include "GLViewer.h"
#include "MeshManager.h"

ToolHelper::ToolHelper(HuginBase::Panorama *pano_in,
                                     VisualizationState *visualization_state_in,
                                     GLPreviewFrame *frame_in)
{
    pano = pano_in;
    visualization_state = visualization_state_in;
    frame = frame_in;
    images_under_mouse_current = false;
    mouse_over_pano = false;
    mouse_screen_x = 0;
    mouse_screen_y = 0;
    mouse_pano_x = 0;
    mouse_pano_y = 0;
}

ToolHelper::~ToolHelper()
{
}

std::set<Tool *> ToolHelper::ActivateTool(Tool *tool)
{
    tools_deactivated.clear();
    tool->Activate();
    return tools_deactivated;
}

void ToolHelper::DeactivateTool(Tool *tool)
{
    tools_deactivated.insert(tool);
    // To deactivate it we need to give up all of its notifications.
    RemoveTool(tool, &mouse_button_notified_tools);
    RemoveTool(tool, &keypress_notified_tools);
    RemoveTool(tool, &mouse_move_notified_tools);
    RemoveTool(tool, &draw_under_notified_tools);
    RemoveTool(tool, &draw_over_notified_tools);
    RemoveTool(tool, &image_draw_begin_tools);
    RemoveTool(tool, &image_draw_end_tools);
    RemoveTool(tool, &images_under_mouse_notified_tools);
    RemoveTool(tool, &really_draw_over_notified_tools);
}

void ToolHelper::MouseMoved(int x, int y, wxMouseEvent & e)
{
    mouse_screen_x = x;
    mouse_screen_y = y;
    // now tell tools that want notification.
    for (auto& tool : mouse_move_notified_tools)
    {
        tool->MouseMoveEvent(mouse_screen_x, mouse_screen_y, e);
    }
    // If the mouse has moved, then we don't know what is underneath it anoymore
    InvalidateImagesUnderMouse();
}

void ToolHelper::InvalidateImagesUnderMouse()
{
    images_under_mouse_current = false;
    // if there are tools that want to know when the images under the mouse
    // pointer change, we better detect the images and notfiy them is applicable
    if (!images_under_mouse_notified_tools.empty())
    {
        // there are tools that want to know... so find out:
        std::set<unsigned int> old_images_under_mouse = images_under_mouse;
        UpdateImagesUnderMouse();
        if (old_images_under_mouse != images_under_mouse)
        {
            // The list has changed. Notifiy all tools that requested it.
            for (auto& tool : images_under_mouse_notified_tools)
            {
                tool->ImagesUnderMouseChangedEvent();
            }
        }
    }
}

void ToolHelper::UpdateImagesUnderMouse()
{
    // We want to find out which images cover the point underneath the mouse
    // pointer.
    images_under_mouse.clear();
    if (IsMouseOverPano())
    {
        images_under_mouse = GetImagesUnderPos(hugin_utils::FDiff2D(mouse_pano_x, mouse_pano_y));
    };
    images_under_mouse_current = true;
}

HuginBase::UIntSet ToolHelper::GetImagesUnderPos(const hugin_utils::FDiff2D& pos)
{
    HuginBase::UIntSet imgUnderPos;
    const unsigned int num_images = pano->getNrOfImages();
    HuginBase::UIntSet displayedImages = pano->getActiveImages();
    for (unsigned int image_index = 0; image_index < num_images; image_index++)
    {
        // don't try any images that are turned off
        if (set_contains(displayedImages, image_index))
        {
            // work out if the image covers the point under the mouse.
            HuginBase::PTools::Transform transform;
            transform.createTransform(*visualization_state->GetSrcImage(image_index),
                *visualization_state->GetOptions());
            double image_x, image_y;
            transform.transformImgCoord(image_x, image_y, pos.x, pos.y);
            if (visualization_state->getViewState()->GetSrcImage(image_index)->isInside(vigra::Point2D(
                int(image_x), int(image_y))))
            {
                // this image is under the position, add it to the set.
                imgUnderPos.insert(image_index);
            };
        };
    };
    return imgUnderPos;
}

void ToolHelper::MouseButtonEvent(wxMouseEvent &e)
{
    for (auto& tool : mouse_button_notified_tools)
    {
        tool->MouseButtonEvent(e);
    }
}

void ToolHelper::MouseWheelEvent(wxMouseEvent &e)
{
    for (auto& tool : mouse_wheel_notified_tools)
    {
        tool->MouseWheelEvent(e);
    }
}


void ToolHelper::KeypressEvent(int keycode, int modifiers, bool pressed)
{
    for (auto& tool : keypress_notified_tools)
    {
        tool->KeypressEvent(keycode, modifiers, pressed);
    }
}

void ToolHelper::BeforeDrawImages()
{
    // let all tools that want to draw under the images do so.
    for (auto& tool : draw_under_notified_tools)
    {
        tool->BeforeDrawImagesEvent();
    }
    // Since we are drawing a new frame, lets assume something has changed,
    // however we want to keep with no images under the mouse if the mouse is
    // not on the panorama.
    if (mouse_over_pano)
    {
        InvalidateImagesUnderMouse();
    }
}

void ToolHelper::AfterDrawImages()
{
    // let all tools that want to draw on top of the images do so.
    for (auto& tool : draw_over_notified_tools)
    {
        tool->AfterDrawImagesEvent();
    }
    // The overlays are done separetly to avoid errors with blending order.
    for (auto& tool : really_draw_over_notified_tools)
    {
        tool->ReallyAfterDrawImagesEvent();
    }
}

void ToolHelper::MarkDirty()
{
    // let all tools that want to draw on top of the images do so.
    for (auto& tool : m_tools_need_dirty_flag)
    {
        tool->MarkDirty();
    }
}

bool ToolHelper::BeforeDrawImageNumber(unsigned int image)
{
    if (image_draw_begin_tools.size() > image)
    {
        if (!image_draw_begin_tools[image].empty())
        {
            bool result = true;
            // BeforeDrawImageEvent can change image_draw_begin_tools[image]
            // so we create a copy before to process all dependent tools, also 
            // when the set is modified in the BeforeDrawImageEvent function
            const auto tools = image_draw_begin_tools[image];
            for (auto tool : tools)
            {
                result &= tool->BeforeDrawImageEvent(image);
            };
            return result;
        }
    }
    return true;
}

void ToolHelper::AfterDrawImageNumber(unsigned int image)
{
    if (image_draw_end_tools.size() > image)
    {
        if (!image_draw_end_tools[image].empty())
        {
            const auto tools = image_draw_end_tools[image];
            for (auto tool : tools)
            {
                tool->BeforeDrawImageEvent(image);
            }
        }
    }
}

void ToolHelper::MouseEnter(int x, int y, wxMouseEvent &e)
{
    // You might expect that this is redundant, because any MouseEnter will be
    // accompanied by a MouseMove, which will achieve the same ends.  However,
    // this is not so.  It's possible for the mouse to enter without moving.
    // Morevover, some environments will trigger a MouseLeave+MouseEnter just
    // before a MouseButton event; handling this is particularly critical for
    // such cases.
    mouse_over_pano = true;
    mouse_screen_x = x;
    mouse_screen_y = y;
    // If the mouse has moved, then we don't know what is underneath it anoymore
    InvalidateImagesUnderMouse();
}

void ToolHelper::MouseLeave()
{
    // if the mouse leaves the preview, there are no images under the mouse
    // pointer anymore.
    mouse_over_pano = false;
    images_under_mouse.clear();
    images_under_mouse_current = true;
    if (!images_under_mouse_notified_tools.empty())
    {
        // notify tools that the set has changed.
        for (auto& tool : images_under_mouse_notified_tools)
        {
            tool->ImagesUnderMouseChangedEvent();
        }        
    }
}


std::set<unsigned int> ToolHelper::GetImageNumbersUnderMouse()
{
    if (!images_under_mouse_current)
    {
        UpdateImagesUnderMouse();
    }
    return images_under_mouse;
}

hugin_utils::FDiff2D ToolHelper::GetMouseScreenPosition()
{
    return hugin_utils::FDiff2D(mouse_screen_x, mouse_screen_y);
}

hugin_utils::FDiff2D ToolHelper::GetMousePanoPosition()
{
    return hugin_utils::FDiff2D(mouse_pano_x, mouse_pano_y);
}

VisualizationState *ToolHelper::GetVisualizationStatePtr()
{
    return visualization_state;
}

ViewState *ToolHelper::GetViewStatePtr()
{
    return visualization_state->getViewState();
}

HuginBase::Panorama *ToolHelper::GetPanoramaPtr()
{
    return pano;
}

void ToolHelper::NotifyMe(Event event, Tool *tool)
{
    switch (event)
    {
        case MOUSE_MOVE:
            AddTool(tool, &mouse_move_notified_tools);
            break;
        case MOUSE_PRESS:
            AddTool(tool, &mouse_button_notified_tools);
            break;
        case MOUSE_WHEEL:
            AddTool(tool, &mouse_wheel_notified_tools);
            break;
        case KEY_PRESS:
            AddTool(tool, &keypress_notified_tools);
            break;
        case DRAW_UNDER_IMAGES:
            AddTool(tool, &draw_under_notified_tools);
            break;
        case DRAW_OVER_IMAGES:
            AddTool(tool, &draw_over_notified_tools);
            break;
        case IMAGES_UNDER_MOUSE_CHANGE:
            AddTool(tool, &images_under_mouse_notified_tools);
            break;
        case REALLY_DRAW_OVER_IMAGES:
            AddTool(tool, &really_draw_over_notified_tools);
            break;
        case MARK_DIRTY:
            AddTool(tool, &m_tools_need_dirty_flag);
            break;
    }   
}

void ToolHelper::NotifyMeBeforeDrawing(unsigned int image_nr,
                                              Tool *tool)
{
    AddTool(tool, &image_draw_begin_tools, image_nr);
}

void ToolHelper::NotifyMeAfterDrawing(unsigned int image_nr,
                                             Tool *tool)
{
    AddTool(tool, &image_draw_end_tools, image_nr);
}

void ToolHelper::DoNotNotifyMe(Event event, Tool *tool)
{
    // TODO we should probably check that the tools have the notification they
    // are trying to give up, as misbehaving tools could break another tool.
    switch (event)
    {
        case MOUSE_MOVE:
            RemoveTool(tool, &mouse_move_notified_tools);
            break;
        case MOUSE_PRESS:
            RemoveTool(tool, &mouse_button_notified_tools);
            break;
        case MOUSE_WHEEL:
            RemoveTool(tool, &mouse_wheel_notified_tools);
            break;
        case KEY_PRESS:
            RemoveTool(tool, &keypress_notified_tools);
            break;
        case DRAW_UNDER_IMAGES:
            RemoveTool(tool, &draw_under_notified_tools);
            break;
        case DRAW_OVER_IMAGES:
            RemoveTool(tool, &draw_over_notified_tools);
            break;
        case IMAGES_UNDER_MOUSE_CHANGE:
            RemoveTool(tool, &images_under_mouse_notified_tools);
            break;
        case REALLY_DRAW_OVER_IMAGES:
            RemoveTool(tool, &really_draw_over_notified_tools);
            break;
        case MARK_DIRTY:
            RemoveTool(tool, &m_tools_need_dirty_flag);
            break;
    }   
}

void ToolHelper::DoNotNotifyMeBeforeDrawing(unsigned int image_nr,
                                              Tool *tool)
{
    RemoveTool(tool, &image_draw_begin_tools, image_nr);
}

void ToolHelper::DoNotNotifyMeAfterDrawing(unsigned int image_nr,
                                             Tool *tool)
{
    RemoveTool(tool, &image_draw_end_tools, image_nr);
}

void ToolHelper::SetStatusMessage(wxString message)
{
    // get the GLPreviewFrame to set its status bar's text.
    frame->SetStatusMessage(message);
}


// These functions remove tools from various things that keep pointers to them
// so that they can be notified. They are called when we don't want to notify
// them anymore.

void ToolHelper::RemoveTool(Tool *tool, Tool **single)
{
    if (*single == tool)
    {
        *single = 0;
    }
}

void ToolHelper::RemoveTool(Tool *tool,
                                   std::set<Tool *> *set)
{
    auto foundTool = set->find(tool);
    if (foundTool != set->end())
    {
        set->erase(foundTool);
    }
}

void ToolHelper::RemoveTool(Tool *tool,
                                  std::vector<std::set<Tool *> > *vector)
{
    // check every item for presence.
    for (unsigned int image = 0; image < vector->size(); image++)
    {
        (*vector)[image].erase(tool);
    }
}

void ToolHelper::RemoveTool(Tool *tool,
                                  std::vector<std::set<Tool *> > *vector,
                                  unsigned int index)
{
    if (vector->size() > index)
    {
        (*vector)[index].erase(tool);
    }
}

void ToolHelper::AddTool(Tool *tool, Tool **single)
{
    if (*single != 0 || *single == tool)
    {
        DeactivateTool(*single);
    }
    *single = tool;
}

void ToolHelper::AddTool(Tool *tool, std::set<Tool *> *set)
{
    set->insert(tool);
}

void ToolHelper::AddTool(Tool *tool,
                               std::vector<std::set<Tool *> > *vector,
                               unsigned int index)
{
    if (vector->size() < index + 1)
    {
        // increase the size of the vector to handle enough elements for index
        // to exist
        vector->resize(index + 1, std::set<Tool*>());
    }
//    else if ((*vector)[index] && (*vector)[index] != tool)
//    {
//        // if a different tool already was doing this, deactivate it.
//        DeactivateTool((*vector)[index]);
//    };
//    (*vector)[index] = tool;
    (*vector)[index].insert(tool);
}


void PreviewToolHelper::MouseMoved(int x, int y, wxMouseEvent &e)
{
    mouse_over_pano = true;
    mouse_screen_x = x;
    mouse_screen_y = y;
    // work out where the pointer is in the panorama.
    vigra::Rect2D visible = visualization_state->GetVisibleArea();
    mouse_pano_x = (double) x / visualization_state->GetScale() + (double) visible.left();
    mouse_pano_y = (double) y / visualization_state->GetScale() + (double) visible.top();
    // now tell tools that want notification.
    for (auto& tool : mouse_move_notified_tools)
    {
        tool->MouseMoveEvent(mouse_pano_x, mouse_pano_y, e);
    }
    // If the mouse has moved, then we don't know what is underneath it anoymore
    InvalidateImagesUnderMouse();
}

void PanosphereOverviewToolHelper::MouseMoved(int x, int y, wxMouseEvent & e)
{
    PanosphereOverviewVisualizationState * panostate = static_cast<PanosphereOverviewVisualizationState*>(visualization_state);

    double d = panostate->getR();
    double r = panostate->getSphereRadius();

    int tcanv_w, tcanv_h;
    panostate->GetViewer()->GetClientSize(&tcanv_w,&tcanv_h);

    double canv_w, canv_h;
    canv_w = tcanv_w;
    canv_h = tcanv_h;
    
    const double fov = panostate->getFOV();
    double fovScale = 1.0;

    double fovy, fovx;
    if (canv_w > canv_h)
    {
        fovy = DEG_TO_RAD(fov);
        fovx = 2 * atan(tan(fovy / 2.0) * canv_w / canv_h);
        fovScale = tan(fovy / 2.0) / (0.5 * canv_h);
    }
    else
    {
        fovx = DEG_TO_RAD(fov);
        fovy = 2 * atan(tan(fovx / 2.0) * canv_h / canv_w);
        fovScale = tan(fovx / 2.0) / (0.5 * canv_w);
    };

    if (panostate->isInsideView())
    {
        // we are looking from inside, mouse is always about the panosphere
        mouse_over_pano = true;
        const double xSrc = (x - canv_w / 2.0) * fovScale;
        const double ySrc = (canv_h / 2.0 - y) * fovScale;
        const double yaw = atan(xSrc) + panostate->getAngX() + M_PI / 2.0;
        const double pitch = M_PI/2.0 - atan2(ySrc, std::sqrt(1.0 + xSrc * xSrc)) - panostate->getAngY();
        //if (yaw < 0) yaw += 2 * M_PI;
        //if (yaw > 2 * M_PI) yaw -= 2 * M_PI;
        mouse_pano_x = yaw / (2 * M_PI) * panostate->GetOptions()->getWidth();
        mouse_pano_y = pitch / (M_PI)*panostate->GetOptions()->getHeight();
    }
    else
    {
        // we are looking from outside, more calculation to find position needed
        const double ax = tan(fovx / 2.0) * d * (x / (canv_w / 2.0) - 1);
        const double ay = tan(fovy / 2.0) * d * (y / (canv_h / 2.0) - 1);

        const double a_limit = r * d / sqrt(d * d - r * r);

        if (ax * ax + ay * ay < a_limit * a_limit)
        {
            mouse_over_pano = true;

            const double ta = (ax * ax + ay * ay) / (d * d) + 1;
            const double tb = -2 * (ax * ax + ay * ay) / d;
            const double tc = ax * ax + ay * ay - r * r;

            const double pz = (-tb + sqrt(tb * tb - 4 * ta * tc)) / (2 * ta);
            const double px = ax * (d - pz) / d;
            const double py = ay * (d - pz) / d;

            const double pl = sqrt(px * px + py * py + pz * pz);

            const double pang_yaw = -atan(px / pz);
            const double pang_pitch = asin(py / pl);

            const double ang_yaw = panostate->getAngX();
            const double ang_pitch = panostate->getAngY();

            const double x = sin(ang_yaw) * cos(ang_pitch);
            const double z = cos(ang_yaw) * cos(ang_pitch);
            const double y = sin(ang_pitch);

            const Vector3 vec(x, y, z);

            const Vector3 top(0, 1, 0);

            const Vector3 ptop = vec.Cross(top).Cross(vec).GetNormalized();

            Vector3 tres;
            tres.x = ptop.x * (ptop.x * vec.x + ptop.y * vec.y + ptop.z * vec.z) + cos(pang_yaw) * (ptop.x * (-ptop.y * vec.y - ptop.z * vec.z) + vec.x * (ptop.y * ptop.y + ptop.z * ptop.z)) + sin(pang_yaw) * (-ptop.z * vec.y + ptop.y * vec.z);
            tres.y = ptop.y * (ptop.x * vec.x + ptop.y * vec.y + ptop.z * vec.z) + cos(pang_yaw) * (ptop.y * (-ptop.x * vec.x - ptop.z * vec.z) + vec.y * (ptop.x * ptop.x + ptop.z * ptop.z)) + sin(pang_yaw) * (-ptop.z * vec.x + ptop.x * vec.z);
            tres.z = ptop.z * (ptop.x * vec.x + ptop.y * vec.y + ptop.z * vec.z) + cos(pang_yaw) * (ptop.z * (-ptop.x * vec.x - ptop.y * vec.y) + vec.z * (ptop.x * ptop.x + ptop.y * ptop.y)) + sin(pang_yaw) * (-ptop.y * vec.x + ptop.x * vec.y);

            const Vector3 pside = ptop.Cross(tres).GetNormalized();
            Vector3 res;

            res.x = pside.x * (pside.x * tres.x + pside.y * tres.y + pside.z * tres.z) + cos(pang_pitch) * (pside.x * (-pside.y * tres.y - pside.z * tres.z) + tres.x * (pside.y * pside.y + pside.z * pside.z)) + sin(pang_pitch) * (-pside.z * tres.y + pside.y * tres.z);
            res.y = pside.y * (pside.x * tres.x + pside.y * tres.y + pside.z * tres.z) + cos(pang_pitch) * (pside.y * (-pside.x * tres.x - pside.z * tres.z) + tres.y * (pside.x * pside.x + pside.z * pside.z)) + sin(-pang_pitch) * (-pside.z * tres.x + pside.x * tres.z);
            res.z = pside.z * (pside.x * tres.x + pside.y * tres.y + pside.z * tres.z) + cos(pang_pitch) * (pside.z * (-pside.x * tres.x - pside.y * tres.y) + tres.z * (pside.x * pside.x + pside.y * pside.y)) + sin(pang_pitch) * (-pside.y * tres.x + pside.x * tres.y);

            double yaw, pitch;
            pitch = asin(res.y);
            yaw = atan2(res.x, res.z);

            yaw += M_PI / 2.0;
            if (yaw < 0) yaw += 2 * M_PI;
            if (yaw > 2 * M_PI) yaw -= 2 * M_PI;

            pitch += M_PI / 2.0;
            pitch = M_PI - pitch;

            mouse_pano_x = yaw / (2 * M_PI) * panostate->GetOptions()->getWidth();
            mouse_pano_y = pitch / (M_PI)*panostate->GetOptions()->getHeight();

            ////        cerr << "mouse " << RAD_TO_DEG(yaw) << " " << RAD_TO_DEG(pitch) << " " << " ; " << RAD_TO_DEG(pang_yaw) << " " << RAD_TO_DEG(pang_pitch) << " ; " << RAD_TO_DEG(ang_yaw) << " " << RAD_TO_DEG(ang_pitch) << " ; " << px << " " << py << " " << pz << " ; " << ax << " " << ay);
            //        cerr << "mouse " << mouse_pano_x << " " << mouse_pano_y << " ; " << canv_w << " " << canv_h);
            //        double t_mouse_pano_x = yaw M
        }
        else
        {
            mouse_over_pano = false;
        };
    };
    
    ToolHelper::MouseMoved(x,y,e);
}

PanosphereOverviewToolHelper::PanosphereOverviewToolHelper(HuginBase::Panorama *pano,
                  VisualizationState *visualization_state,
                  GLPreviewFrame * frame) : OverviewToolHelper(pano, visualization_state, frame) {}

PanosphereOverviewToolHelper::~PanosphereOverviewToolHelper() {}

void PanosphereOverviewToolHelper::NotifyMe(PanosphereOverviewEvent event, PanosphereOverviewTool * tool) {
    switch (event) {
        case DRAW_OVER_IMAGES_BACK:
            AddTool(tool, &draw_over_notified_tools_back);
        break;
        case DRAW_OVER_IMAGES_FRONT:
            AddTool(tool, &draw_over_notified_tools_front);
        break;
        case DRAW_UNDER_IMAGES_BACK:
            AddTool(tool, &draw_under_notified_tools_back);
        break;
        case DRAW_UNDER_IMAGES_FRONT:
            AddTool(tool, &draw_under_notified_tools_front);
        break;
    }
}

void PanosphereOverviewToolHelper::DoNotNotifyMe(PanosphereOverviewEvent event, PanosphereOverviewTool * tool) {
    switch (event) {
        case DRAW_OVER_IMAGES_BACK:
            RemoveTool(tool, &draw_over_notified_tools_back);
        break;
        case DRAW_OVER_IMAGES_FRONT:
            RemoveTool(tool, &draw_over_notified_tools_front);
        break;
        case DRAW_UNDER_IMAGES_BACK:
            RemoveTool(tool, &draw_under_notified_tools_back);
        break;
        case DRAW_UNDER_IMAGES_FRONT:
            RemoveTool(tool, &draw_under_notified_tools_front);
        break;
    }
}

void PanosphereOverviewToolHelper::BeforeDrawImagesBack()
{
    for (auto& tool : draw_under_notified_tools_back)
    {
        static_cast<PanosphereOverviewTool*>(tool)->BeforeDrawImagesBackEvent();
    }
}

void PanosphereOverviewToolHelper::BeforeDrawImagesFront()
{
    for (auto& tool : draw_under_notified_tools_front)
    {
        static_cast<PanosphereOverviewTool*>(tool)->BeforeDrawImagesFrontEvent();
    }
}

void PanosphereOverviewToolHelper::AfterDrawImagesBack()
{
    for (auto& tool : draw_over_notified_tools_back)
    {
        static_cast<PanosphereOverviewTool*>(tool)->AfterDrawImagesBackEvent();
    }
}

void PanosphereOverviewToolHelper::AfterDrawImagesFront()
{
    for (auto& tool : draw_over_notified_tools_front)
    {
        static_cast<PanosphereOverviewTool*>(tool)->AfterDrawImagesFrontEvent();
    }
}

void PanosphereOverviewToolHelper::DeactivateTool(Tool *tool)
{
    ToolHelper::DeactivateTool(tool);

    RemoveTool(tool, &draw_under_notified_tools_back);
    RemoveTool(tool, &draw_under_notified_tools_front);
    RemoveTool(tool, &draw_over_notified_tools_back);
    RemoveTool(tool, &draw_over_notified_tools_front);
}


PlaneOverviewToolHelper::PlaneOverviewToolHelper(HuginBase::Panorama *pano,
                  VisualizationState *visualization_state,
                  GLPreviewFrame * frame) : OverviewToolHelper(pano, visualization_state, frame) {}

PlaneOverviewToolHelper::~PlaneOverviewToolHelper() {}

void PlaneOverviewToolHelper::MouseMoved(int x, int y, wxMouseEvent & e)
{

    PlaneOverviewVisualizationState * panostate = static_cast<PlaneOverviewVisualizationState*>(visualization_state);

    double d = panostate->getR();

    int tcanv_w, tcanv_h;
    panostate->GetViewer()->GetClientSize(&tcanv_w,&tcanv_h);

    double canv_w, canv_h;
    canv_w = tcanv_w;
    canv_h = tcanv_h;
    
    double fov = panostate->getFOV();

    double fovy, fovx;
    if (canv_w > canv_h) {
        fovy = DEG_TO_RAD(fov);
        fovx = 2 * atan( tan(fovy / 2.0) * canv_w / canv_h);
    } else {
        fovx = DEG_TO_RAD(fov);
        fovy = 2 * atan( tan(fovx / 2.0) * canv_h / canv_w);
    }

    double vis_w, vis_h;
    vis_w = 2.0 * tan ( fovx / 2.0 ) * d;
    vis_h = 2.0 * tan ( fovy / 2.0 ) * d;

    //position of the mouse on the z=0 plane
    double prim_x, prim_y;
    prim_x = (double) x / canv_w * vis_w - vis_w / 2.0 + panostate->getX();
    prim_y = ((double) y / canv_h * vis_h - vis_h / 2.0 - panostate->getY());

//    std::cout << "mouse ov" << plane_x << " " << plane_y << std::endl;
    plane_x = prim_x;
    plane_y = prim_y;

    double width, height;
    HuginBase::PanoramaOptions * opts = panostate->GetOptions();
    width = opts->getWidth();
    height = opts->getHeight();

    mouse_pano_x = prim_x / MeshManager::PlaneOverviewMeshInfo::scale * width + width / 2.0;
    mouse_pano_y = prim_y / MeshManager::PlaneOverviewMeshInfo::scale * width + height / 2.0;

//    std::cout << "plane mouse " << mouse_pano_x << " " << mouse_pano_y << " ; " << prim_x << " " << prim_y << " ; " << width << " " << height << std::endl;
//    cerr << "plane mouse " << mouse_pano_x << " " << mouse_pano_y);

    mouse_over_pano = true;
    
    ToolHelper::MouseMoved(x,y,e);
}



