/*
** (c) 1996-2000 The Regents of the University of California (through
** E.O. Lawrence Berkeley National Laboratory), subject to approval by
** the U.S. Department of Energy.  Your use of this software is under
** license -- the license agreement is attached and included in the
** directory as license.txt or you may contact Berkeley Lab's Technology
** Transfer Department at TTD@lbl.gov.  NOTICE OF U.S. GOVERNMENT RIGHTS.
** The Software was developed under funding from the U.S. Government
** which consequently retains certain rights as follows: the
** U.S. Government has been granted for itself and others acting on its
** behalf a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, and perform publicly
** and display publicly.  Beginning five (5) years after the date
** permission to assert copyright is obtained from the U.S. Department of
** Energy, and subject to any subsequent five (5) year renewals, the
** U.S. Government is granted for itself and others acting on its behalf
** a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, distribute copies to
** the public, perform publicly and display publicly, and to permit
** others to do so.
*/


//
// This is another version of AmrDerive.cpp that calculates integrals of
// quantities and writes out scalars instead of plotfiles.
//
//  currently hardcoded for sum_rhoYO2   (quantity 6)
//                      and sum_rhoYC2H2 (quantity 9)
//
//  i'll update it to allow user input of quantity number.  v.
//

#include <new>
#include <iostream>
#include <cstdlib>
#include <cstring>
using std::set_new_handler;

#include <unistd.h>

#include "REAL.H"
#include "Box.H"
#include "FArrayBox.H"
#include "ParmParse.H"
#include "ParallelDescriptor.H"
#include "DataServices.H"
#include "Utility.H"
#include "VisMF.H"
#include "Derived.H"

static
void
PrintUsage (char* progName)
{
    cout << "\nUsage:\n"
         << progName
         << "\n\tinfile = inputFileName"
         << "\n\t[-help]"
         << "\n\n";
    exit(1);
}

static
Real
SumMass (AmrData& amrData, const int compNumber)
{
    Real sum = 0;
    //
    // The component number of density in MultiFab.
    //
    const int ncomp = compNumber;

    int finest_level = amrData.FinestLevel();

    FArrayBox fab;

    for (int iLevel = 0; iLevel <= finest_level; ++iLevel)
    {
        const Real* dx = amrData.DxLevel()[iLevel].dataPtr();

        MultiFabIterator ntmfi(amrData.GetGrids(iLevel, ncomp));

        for ( ; ntmfi.isValid(); ++ntmfi)
        {
            //
            // Make copy of FAB so that we can change it.
            //
            fab.resize(ntmfi.fabbox(), 1);

            //fab.copy(ntmfi(), ncomp, 0, 1);
            fab.copy(ntmfi(), 0, 0, 1);

            if (iLevel < finest_level)
            {
                const BoxArray& f_box = amrData.boxArray(iLevel+1);

                for (int j = 0; j < f_box.length(); j++)
                {
                    Box c_box = ::coarsen(f_box[j],amrData.RefRatio()[iLevel]);
                    c_box &= ntmfi.validbox();
                    if (c_box.ok())
                        fab.setVal(0.0, c_box, 0);
                }
            }
            Real s;
            const Real* dat = fab.dataPtr();
            const int* dlo  = fab.loVect();
            const int* dhi  = fab.hiVect();
            const int* lo   = ntmfi.validbox().loVect();
            const int* hi   = ntmfi.validbox().hiVect();
            int nz          = hi[1]-lo[1]+1;
            Array<Real> tmp(nz);
            FORT_SUMMASS(dat,ARLIM(dlo),ARLIM(dhi),ARLIM(lo),ARLIM(hi),
                         dx,&s,tmp.dataPtr());
            sum += s;
        }
    }

    ParallelDescriptor::ReduceRealSum(sum);

    return sum;
}


int
main (int   argc,
      char* argv[])
{
    //
    // Make sure to catch new failures.
    //
    set_new_handler(Utility::OutOfMemory);

    if (argc == 1)
        PrintUsage(argv[0]);

    //int iLevel;
    ParallelDescriptor::StartParallel(&argc, &argv);

    ParmParse pp(argc-1,argv+1);

    if (pp.contains("help"))
        PrintUsage(argv[0]);

    FArrayBox::setFormat(FABio::FAB_IEEE_32);
    //
    // Set default info.
    //
    Real gam = 1.667;
    FORT_SETGAMMA(&gam);
    //
    // Scan the arguments.
    //
    aString iFile;
    pp.query("infile", iFile);
    if (iFile.isNull())
        BoxLib::Abort("You must specify `infile'");

    DataServices::SetBatchMode();
    FileType fileType(NEWPLT);
    
    DataServices dataServices(iFile, fileType);

    if ( ! dataServices.AmrDataOk())
    {
        //
        // This calls ParallelDescriptor::EndParallel() and exit()
        //
        DataServices::Dispatch(DataServices::ExitRequest, NULL);
    }
    AmrData& amrData = dataServices.AmrDataRef();

    BL_ASSERT(amrData.NComp() == NSTATECOMPS);

    Real sum_rhoYO2   = SumMass(amrData, 6);
    Real sum_rhoYC2H2 = SumMass(amrData, 9);

    if (ParallelDescriptor::IOProcessor())
    {
        const Array<Real>& size = amrData.ProbSize();

        Real volume = D_TERM(size[0],*size[1],*size[2]);

        BL_ASSERT(volume > 0);

        cout << "time = " << amrData.Time()
             << "  sum_rhoYO2 = " << sum_rhoYO2
             << "  sum_rhoYC2H2 = " << sum_rhoYC2H2
             << endl;
    }

    DataServices::Dispatch(DataServices::ExitRequest, NULL);
}
