#!/usr/bin/python

# Copyright (c) 2014 Hewlett-Packard Development Company, L.P.
# Copyright (c) 2013, Benno Joy <benno@ansible.com>
# Copyright (c) 2013, John Dewey <john@dewey.ws>
#
# This module is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This software is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this software.  If not, see <http://www.gnu.org/licenses/>.


try:
    import shade
    HAS_SHADE = True
except ImportError:
    HAS_SHADE = False


DOCUMENTATION = '''
---
module: os_keypair
short_description: Add/Delete a keypair from OpenStack
extends_documentation_fragment: openstack
version_added: "2.0"
description:
  - Add or Remove key pair from OpenStack
options:
  name:
    description:
      - Name that has to be given to the key pair
    required: true
    default: None
  public_key:
    description:
      - The public key that would be uploaded to nova and injected into VMs
        upon creation.
    required: false
    default: None
  public_key_file:
    description:
      - Path to local file containing ssh public key. Mutually exclusive
        with public_key.
    required: false
    default: None
  state:
    description:
      - Should the resource be present or absent.
    choices: [present, absent]
    default: present
requirements: []
'''

EXAMPLES = '''
# Creates a key pair with the running users public key
- os_keypair:
      cloud: mordred
      state: present
      name: ansible_key
      public_key_file: /home/me/.ssh/id_rsa.pub

# Creates a new key pair and the private key returned after the run.
- os_keypair:
      cloud: rax-dfw
      state: present
      name: ansible_key
'''

RETURN = '''
id:
    description: Unique UUID.
    returned: success
    type: string
name:
    description: Name given to the keypair.
    returned: success
    type: string
public_key:
    description: The public key value for the keypair.
    returned: success
    type: string
private_key:
    description: The private key value for the keypair.
    returned: Only when a keypair is generated for the user (e.g., when creating one
              and a public key is not specified).
    type: string
'''


def _system_state_change(module, keypair):
    state = module.params['state']
    if state == 'present' and not keypair:
        return True
    if state == 'absent' and keypair:
        return True
    return False


def main():
    argument_spec = openstack_full_argument_spec(
        name            = dict(required=True),
        public_key      = dict(default=None),
        public_key_file = dict(default=None),
        state           = dict(default='present',
                               choices=['absent', 'present']),
    )

    module_kwargs = openstack_module_kwargs(
        mutually_exclusive=[['public_key', 'public_key_file']])

    module = AnsibleModule(argument_spec,
                           supports_check_mode=True,
                           **module_kwargs)

    if not HAS_SHADE:
        module.fail_json(msg='shade is required for this module')

    state = module.params['state']
    name = module.params['name']
    public_key = module.params['public_key']

    if module.params['public_key_file']:
        public_key = open(module.params['public_key_file']).read()
        public_key = public_key.rstrip()

    try:
        cloud = shade.openstack_cloud(**module.params)
        keypair = cloud.get_keypair(name)

        if module.check_mode:
            module.exit_json(changed=_system_state_change(module, keypair))

        if state == 'present':
            if keypair and keypair['name'] == name:
                if public_key and (public_key != keypair['public_key']):
                    module.fail_json(
                        msg="Key name %s present but key hash not the same"
                            " as offered. Delete key first." % name
                    )
                else:
                    changed = False
            else:
                keypair = cloud.create_keypair(name, public_key)
                changed = True

            module.exit_json(changed=changed,
                             key=keypair,
                             id=keypair['id'])

        elif state == 'absent':
            if keypair:
                cloud.delete_keypair(name)
                module.exit_json(changed=True)
            module.exit_json(changed=False)

    except shade.OpenStackCloudException as e:
        module.fail_json(msg=e.message)

# this is magic, see lib/ansible/module_common.py
from ansible.module_utils.basic import *
from ansible.module_utils.openstack import *
if __name__ == '__main__':
    main()
