# Copyright 2015 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""FanNetwork objects."""

from __future__ import (
    absolute_import,
    print_function,
    unicode_literals,
    )

str = None

__metaclass__ = type
__all__ = [
    "FanNetwork",
    "FANNETWORK_NAME_VALIDATOR",
    "FANNETWORK_BRIDGE_VALIDATOR",
    ]

from django.core.validators import RegexValidator
from django.db.models import (
    CharField,
    NullBooleanField,
    PositiveIntegerField,
)
from maasserver import DefaultMeta
from maasserver.fields import IPv4CIDRField
from maasserver.models.cleansave import CleanSave
from maasserver.models.timestampedmodel import TimestampedModel


FANNETWORK_NAME_VALIDATOR = RegexValidator('^[ \w-]+$')
FANNETWORK_BRIDGE_VALIDATOR = RegexValidator('^[\w\-_]+$')


class FanNetwork(CleanSave, TimestampedModel):
    """A `FanNetwork`.

    :ivar name: The short-human-identifiable name for this fannetwork.
    :ivar objects: An instance of the class :class:`FanNetworkManager`.
    """

    class Meta(DefaultMeta):
        """Needed for South to recognize this model."""
        verbose_name = "Fan Network"
        verbose_name_plural = "Fan Networks"

    # required fields
    name = CharField(
        max_length=256, unique=True, editable=True,
        help_text="Name of the fan network",
        validators=[FANNETWORK_NAME_VALIDATOR])

    overlay = IPv4CIDRField(
        blank=False, unique=True, editable=True, null=False)

    underlay = IPv4CIDRField(
        blank=False, unique=True, editable=True, null=False)

    # optional fields
    dhcp = NullBooleanField(
        blank=True, unique=False, editable=True, null=True)

    host_reserve = PositiveIntegerField(
        default=1,
        blank=True, unique=False, editable=True, null=True)

    bridge = CharField(
        blank=True, editable=True, max_length=255, null=True,
        help_text="If specified, this bridge name is used on the hosts",
        validators=[FANNETWORK_BRIDGE_VALIDATOR])

    off = NullBooleanField(
        default=False, unique=False, editable=True, null=True, blank=True,
        help_text="Create the configuration, but mark it as 'off'")
