/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Spicebird code.
 *
 * The Initial Developer of the Original Code is
 * Synovel Software Technologies
 * Portions created by the Initial Developer are Copyright (C) 2008
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Prasad Sunkari <prasad@synovel.com> (Original Author)
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

#include "csTPChannel.h"
#include "csITelepathyCallbacks.h"
#include "nsComponentManagerUtils.h"
#include "nsArrayUtils.h"
#include "nsIMutableArray.h"

#include "telepathy-glib/dbus.h"
#include "glib.h"

NS_IMPL_ISUPPORTS5          \
  (                         \
    csTPChannel,            \
    csITPChannel,           \
    csITPChannelGroup,      \
    csITPChannelChatState,  \
    csITPChannelPassword,   \
    csITPProxy              \
  )

/*
 * This include is below the NS_IMPL_ISUPPORTS
 */
#include "csTelepathyMacros.h"

csTPChannel::csTPChannel()
{
  m_BusDaemon = tp_dbus_daemon_new(tp_get_bus());
  m_Channel = NULL;

  CS_TELEPATHY_INIT_PROXY
}

csTPChannel::~csTPChannel()
{
}

NS_IMETHODIMP csTPChannel::GetNativeChannel(TpChannel **native)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  *native = m_Channel;
  return NS_OK;
}

NS_IMETHODIMP csTPChannel::SetNativeChannel(TpChannel *native)
{
  if (m_Channel)
    return NS_ERROR_ALREADY_INITIALIZED;

  m_Channel = native;
  return NS_OK;
}

NS_IMETHODIMP csTPChannel::Init(const nsACString & aConnectionObjPath,
                                const nsACString & aObjPath,
                                const nsACString & aChannelType,
                                PRUint32 aHandleType, PRUint32 aHandleNumber) 
{
  if (m_Channel)
    return NS_ERROR_ALREADY_INITIALIZED;

  m_Connection = tp_connection_new(m_BusDaemon, NULL, 
                                   nsCString(aConnectionObjPath).get(), NULL);
  if (!m_Connection)
    return NS_ERROR_OUT_OF_MEMORY;

  m_Channel = tp_channel_new(m_Connection, nsCString(aObjPath).get(), 
                             nsCString(aChannelType).get(), 
                             TP_UNKNOWN_HANDLE_TYPE, 0, NULL);
  if (!m_Channel)
    return NS_ERROR_OUT_OF_MEMORY;

  return NS_OK;
}

NS_IMETHODIMP csTPChannel::GetTargetHandle(PRUint32 *aTargetHandle)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  g_object_get(m_Channel, "handle", aTargetHandle, NULL);
  return NS_OK;
}

NS_IMETHODIMP csTPChannel::GetTargetHandleType(PRUint32 *aTargetHandleType)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  g_object_get(m_Channel, "handle-type", aTargetHandleType, NULL);
  return NS_OK;
}

NS_IMETHODIMP csTPChannel::GetChannelType(nsACString & aChannelType)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;
  
  char *channelType;
  g_object_get(m_Channel, "channel-type", &channelType, NULL);

  aChannelType.Assign(channelType);
  g_free(channelType);
  return NS_OK;
}

static void
GotClosed(TpChannel *channel, const GError *error,
          gpointer user_data, GObject *unused)
{
  csITPEmptyCB *callback = (csITPEmptyCB *)user_data;

  CS_TELEPATHY_CALLBACK_CHECK_ERROR(, callback, error);

  callback->DoAction();
  NS_IF_RELEASE(callback);
}

NS_IMETHODIMP csTPChannel::Close(csITPEmptyCB *cb)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  NS_IF_ADDREF(cb);
  tp_cli_channel_call_close(m_Channel, -1, GotClosed,
                            cb, NULL, NULL);
  return NS_OK;
}

void csTPChannel::ChannelReady()
{
  if (m_ChannelReadyObservers) {
    PRUint32 length;
    m_ChannelReadyObservers->GetLength(&length);
    nsCOMPtr<csITPEmptyCB> observer;
 
    for (PRUint32 i = 0; i < length; i++) {
      observer = do_QueryElementAt(m_ChannelReadyObservers, i);
      observer->DoAction();
    }
  }
}

static void 
GotChannelReadySignal(TpChannel *proxy, 
                      const GError *error, gpointer user_data)
{
  csTPChannel *channel = (csTPChannel *)user_data;
  if (!channel)
    return;

  channel->ChannelReady();
}

NS_IMETHODIMP csTPChannel::AddChannelReadyObserver(csITPEmptyCB *observer)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  NS_ENSURE_ARG_POINTER(observer);

  if (!m_ChannelReadyObservers) {
    m_ChannelReadyObservers = do_CreateInstance(NS_ARRAY_CONTRACTID);
    tp_channel_call_when_ready(m_Channel, GotChannelReadySignal, this);
  }

  m_ChannelReadyObservers->AppendElement(observer, PR_FALSE);
  return NS_OK;
}
CS_TELEPATHY_REMOVE_OBSERVER(csTPChannel, RemoveChannelReadyObserver,
                             csITPEmptyCB, m_ChannelReadyObservers);

void csTPChannel::ChannelClosed()
{
  if (m_CloseObservers) {
    PRUint32 length;
    m_CloseObservers->GetLength(&length);
    nsCOMPtr<csITPEmptyCB> observer;
 
    for (PRUint32 i = 0; i < length; i++) {
      observer = do_QueryElementAt(m_CloseObservers, i);
      observer->DoAction();
    }
  }
}

static void GotClosedSignal(TpChannel *proxy, 
                            gpointer user_data, GObject *unused)
{
  csTPChannel *channel = (csTPChannel *)user_data;
  if (!channel)
    return;

  channel->ChannelClosed();
}

NS_IMETHODIMP csTPChannel::AddCloseObserver(csITPEmptyCB *observer)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  NS_ENSURE_ARG_POINTER(observer);

  if (!m_CloseObservers) {
    m_CloseObservers = do_CreateInstance(NS_ARRAY_CONTRACTID);
    tp_cli_channel_connect_to_closed(m_Channel, GotClosedSignal, 
                                     this, NULL, NULL, NULL);
  }

  m_CloseObservers->AppendElement(observer, PR_FALSE);
  return NS_OK;
}
CS_TELEPATHY_REMOVE_OBSERVER(csTPChannel, RemoveCloseObserver,
                             csITPEmptyCB, m_CloseObservers);

/*
 * csIChannelGroup Implementation
 */

static void
GotAddMembersResponse(TpChannel *proxy, const GError *error,
                      gpointer user_data, GObject *weak_object)
{
  csITPEmptyCB *callback = (csITPEmptyCB *)user_data;

  CS_TELEPATHY_CALLBACK_CHECK_ERROR(, callback, error);

  callback->DoAction();
  NS_IF_RELEASE(callback);
}

NS_IMETHODIMP csTPChannel::AddMembers(PRUint32 count, PRUint32 *aContacts, 
                                      const nsACString & aAddMsg, csITPEmptyCB *cb)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  GArray *array = g_array_new(false, false, sizeof(guint));
  for (PRUint32 i=0; i<count; i++) 
    g_array_append_val(array, aContacts[i]);

  NS_IF_ADDREF(cb);
  tp_cli_channel_interface_group_call_add_members(m_Channel, -1, array,
                                                  nsCString(aAddMsg).get(),
                                                  GotAddMembersResponse, cb,
                                                  NULL, NULL);

  return NS_OK;
}

static void
GotAllMembers(TpChannel *proxy, const GArray *members,
                   const GArray *localPending, const GArray *remotePending,
                   const GError *error, gpointer user_data, GObject *weak_object)
{
  csITPGroupMembersChangeObserver *callback = 
                                  (csITPGroupMembersChangeObserver *)user_data;

  CS_TELEPATHY_CALLBACK_CHECK_ERROR(, callback, error);

  unsigned int len = members->len;
  for (unsigned int i=0; i<len; i++)
    callback->DoAddMember(g_array_index(members, PRUint32, i));

  len = localPending->len;
  for (unsigned int i=0; i<len; i++)
    callback->DoAddLocalPendingMember(g_array_index(localPending, PRUint32, i));

  len = remotePending->len;
  for (unsigned int i=0; i<len; i++)
    callback->DoAddRemotePendingMember(g_array_index(remotePending, PRUint32, i));

  callback->OnItemsComplete();
  NS_IF_RELEASE(callback);
}

NS_IMETHODIMP csTPChannel::GetAllMembers(csITPGroupMembersChangeObserver *cb)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  NS_IF_ADDREF(cb);
  tp_cli_channel_interface_group_call_get_all_members(m_Channel, -1,
                                                      GotAllMembers, cb,
                                                      NULL, NULL);
  return NS_OK;
}

NS_IMETHODIMP csTPChannel::GetGroupFlags(PRUint32 *aGroupFlags)
{
  *aGroupFlags = m_GroupFlags;
  return NS_OK;
}

static void
GotHandleOwners(TpChannel *proxy, const GArray *handles, 
                const GError *error, gpointer user_data, GObject *weak_object)
{
  csITPIntegerListCB *callback = (csITPIntegerListCB*)user_data;

  CS_TELEPATHY_CALLBACK_CHECK_ERROR(, callback, error);

  unsigned int len = handles->len;
  for (unsigned int i=0; i<len; i++)
    callback->OnAddItem(g_array_index(handles, PRUint32, i));

  callback->OnItemsComplete();
  NS_IF_RELEASE(callback);
}

NS_IMETHODIMP csTPChannel::GetHandleOwners(PRUint32 count, PRUint32 *aHandles, 
                                           csITPIntegerListCB *cb)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  GArray *array = g_array_new(false, false, sizeof(guint));
  for (PRUint32 i=0; i<count; i++) 
    g_array_append_val(array, aHandles[i]);

  NS_IF_ADDREF(cb);
  tp_cli_channel_interface_group_call_get_handle_owners(m_Channel, -1, array,
                                                        GotHandleOwners, cb,
                                                        NULL, NULL);
  return NS_OK;
}

static void
GotLocalPendingMembers(TpChannel *proxy, const GArray *localPending,
                       const GError *error, gpointer user_data, 
                       GObject *weak_object)
{
  csITPGroupMembersChangeObserver *callback = 
                                  (csITPGroupMembersChangeObserver *)user_data;

  CS_TELEPATHY_CALLBACK_CHECK_ERROR(, callback, error);

  unsigned int len = localPending->len;
  for (unsigned int i=0; i<len; i++)
    callback->DoAddLocalPendingMember(g_array_index(localPending, PRUint32, i));

  callback->OnItemsComplete();
  NS_IF_RELEASE(callback);
}

NS_IMETHODIMP csTPChannel::GetLocalPendingMembers(csITPGroupMembersChangeObserver *cb)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  NS_IF_ADDREF(cb);
  tp_cli_channel_interface_group_call_get_local_pending_members(m_Channel, -1,
                                                                GotLocalPendingMembers, cb,
                                                                NULL, NULL);
  return NS_OK;
}

NS_IMETHODIMP csTPChannel::GetLocalPendingMembersWithInfo(csITPInterfaceListCB *cb)
{
    return NS_ERROR_NOT_IMPLEMENTED;
}

static void
GotMembers(TpChannel *proxy, const GArray *members,
           const GError *error, gpointer user_data, GObject *weak_object)
{
  csITPGroupMembersChangeObserver *callback = 
                                  (csITPGroupMembersChangeObserver *)user_data;

  CS_TELEPATHY_CALLBACK_CHECK_ERROR(, callback, error);

  unsigned int len = members->len;
  for (unsigned int i=0; i<len; i++)
    callback->DoAddMember(g_array_index(members, PRUint32, i));

  callback->OnItemsComplete();
  NS_IF_RELEASE(callback);
}

NS_IMETHODIMP csTPChannel::GetMembers(csITPGroupMembersChangeObserver *cb)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  NS_IF_ADDREF(cb);
  tp_cli_channel_interface_group_call_get_members(m_Channel, -1,
                                                  GotMembers, cb,
                                                  NULL, NULL);
  return NS_OK;
}

static void
GotRemotePendingMembers(TpChannel *proxy, const GArray *remotePending,
                        const GError *error, gpointer user_data, 
                        GObject *weak_object)
{
  csITPGroupMembersChangeObserver *callback = 
                                  (csITPGroupMembersChangeObserver *)user_data;

  CS_TELEPATHY_CALLBACK_CHECK_ERROR(, callback, error);

  unsigned int len = remotePending->len;
  for (unsigned int i=0; i<len; i++)
    callback->DoAddRemotePendingMember(g_array_index(remotePending, PRUint32, i));

  callback->OnItemsComplete();
  NS_IF_RELEASE(callback);
}

NS_IMETHODIMP csTPChannel::GetRemotePendingMembers(csITPGroupMembersChangeObserver *cb)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  NS_IF_ADDREF(cb);
  tp_cli_channel_interface_group_call_get_remote_pending_members(m_Channel, -1,
                                                                GotRemotePendingMembers, cb,
                                                                NULL, NULL);
  return NS_OK;
}

NS_IMETHODIMP csTPChannel::GetSelfHandle(PRUint32 *aSelfHandle)
{
  *aSelfHandle = m_SelfHandle;
  return NS_OK;
}

NS_IMETHODIMP csTPChannel::RemoveMembers(PRUint32 count, PRUint32 *aContacts,
                                         const nsACString & aRemoveMsg, 
                                         csITPEmptyCB *cb)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  GArray *array = g_array_new(false, false, sizeof(guint));
  for (PRUint32 i=0; i<count; i++) 
    g_array_append_val(array, aContacts[i]);

  NS_IF_ADDREF(cb);
  tp_cli_channel_interface_group_call_remove_members(m_Channel, -1, array,
                                                     nsCString(aRemoveMsg).get(),
                                                     GotAddMembersResponse, cb,
                                                     NULL, NULL);

  return NS_OK;
}

NS_IMETHODIMP csTPChannel::RemoveMembersWithReason(PRUint32 count, 
                                                   PRUint32 *aContacts, 
                                                   const nsACString & aRemoveMsg, 
                                                   PRUint32 aReason, csITPEmptyCB *cb)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  GArray *array = g_array_new(false, false, sizeof(guint));
  for (PRUint32 i=0; i<count; i++) 
    g_array_append_val(array, aContacts[i]);

  NS_IF_ADDREF(cb);
  tp_cli_channel_interface_group_call_remove_members_with_reason(m_Channel, 
                                      -1, array, nsCString(aRemoveMsg).get(),
                                      aReason, GotAddMembersResponse, cb, NULL, NULL);

  return NS_OK;
}

///////////// Group Members Change Signal

void 
csTPChannel::MembersChanged(const gchar *aMessage, const GArray *aAdded, 
                           const GArray *aRemoved, const GArray *aLocalPending, 
                           const GArray *aRemotePending, guint aActor, 
                           guint aReason)
{
  if (m_GroupMembersChangeObservers) {
    PRUint32 length;
    m_GroupMembersChangeObservers->GetLength(&length);
    nsCOMPtr<csITPGroupMembersChangeObserver> observer;
 
    for (PRUint32 i = 0; i < length; i++) {
      observer = do_QueryElementAt(m_GroupMembersChangeObservers, i);

      unsigned int len = aAdded->len;
      for (unsigned int i=0; i<len; i++)
        observer->DoAddMember(g_array_index(aAdded, PRUint32, i));
  
      len = aRemoved->len;
      for (unsigned int i=0; i<len; i++)
        observer->DoRemoveMember(g_array_index(aRemoved, PRUint32, i));
  
      len = aLocalPending->len;
      for (unsigned int i=0; i<len; i++)
        observer->DoAddLocalPendingMember(g_array_index(aLocalPending, PRUint32, i));
  
      len = aRemotePending->len;
      for (unsigned int i=0; i<len; i++)
        observer->DoAddRemotePendingMember(g_array_index(aRemotePending, PRUint32, i));
  
      observer->OnItemsComplete();
    }
  }
}

static void 
GotMembersChangeSignal(TpChannel *proxy, const gchar *aMessage,
                       const GArray *aAdded, const GArray *aRemoved,
                       const GArray *aLocalPending, const GArray *aRemotePending, 
                       guint aActor, guint aReason, 
                       gpointer user_data, GObject *unused)
{
  csTPChannel *channel = (csTPChannel *)user_data;
  if (!channel)
    return;

  channel->MembersChanged(aMessage, aAdded, aRemoved, aLocalPending,
                               aRemotePending, aActor, aReason);
}

NS_IMETHODIMP csTPChannel::AddMembersChangeObserver(csITPGroupMembersChangeObserver *observer)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  NS_ENSURE_ARG_POINTER(observer);

  if (!m_GroupMembersChangeObservers) {
    m_GroupMembersChangeObservers = do_CreateInstance(NS_ARRAY_CONTRACTID);
    tp_cli_channel_interface_group_connect_to_members_changed(m_Channel,
                                              GotMembersChangeSignal,
                                              this, NULL, NULL, NULL);
  }

  m_GroupMembersChangeObservers->AppendElement(observer, PR_FALSE);
  return NS_OK;
}
CS_TELEPATHY_REMOVE_OBSERVER(csTPChannel, RemoveMembersChangeObserver,
                             csITPGroupMembersChangeObserver,
                             m_GroupMembersChangeObservers);

//////////////// Group Flags Change Signal

void csTPChannel::GroupFlagsChanged(PRUint32 aAdded, PRUint32 aRemoved)
{
  m_GroupFlags |= aAdded;
  m_GroupFlags ^= aRemoved;

  if (m_GroupFlagsChangeObservers) {
    PRUint32 length;
    m_GroupFlagsChangeObservers->GetLength(&length);
    nsCOMPtr<csITPGroupFlagsChangeObserver> observer;
 
    for (PRUint32 i = 0; i < length; i++) {
      observer = do_QueryElementAt(m_GroupFlagsChangeObservers, i);
      observer->DoUpdateGroupFlags(aAdded, aRemoved);
    }
  }
}

static void
GotGroupFlagsChangeSignal(TpChannel *proxy, guint aFlagsAdded, guint aFlagsRemoved, 
                          gpointer user_data, GObject *unused)
{
  csTPChannel *channel = (csTPChannel *)user_data;
  if (!channel)
    return;

  channel->GroupFlagsChanged(aFlagsAdded, aFlagsRemoved);
}

NS_IMETHODIMP csTPChannel::AddGroupFlagsChangeObserver(csITPGroupFlagsChangeObserver *observer)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  NS_ENSURE_ARG_POINTER(observer);

  if (!m_GroupFlagsChangeObservers) {
    m_GroupFlagsChangeObservers = do_CreateInstance(NS_ARRAY_CONTRACTID);
    tp_cli_channel_interface_group_connect_to_group_flags_changed(m_Channel,
                                              GotGroupFlagsChangeSignal,
                                              this, NULL, NULL, NULL);
  }

  m_GroupFlagsChangeObservers->AppendElement(observer, PR_FALSE);
  return NS_OK;
}
CS_TELEPATHY_REMOVE_OBSERVER(csTPChannel, RemoveGroupFlagsChangeObserver,
                             csITPGroupFlagsChangeObserver,
                             m_GroupFlagsChangeObservers);

//////////////// Group Handle Owners Change Signal

static void 
HandleOwnersChangeHelperFunc(gpointer key, gpointer value, gpointer user_data)
{
  csITPGroupHandleOwnersChangeObserver *observer = 
            (csITPGroupHandleOwnersChangeObserver *)user_data;
  if (!observer)
    return;

  observer->DoUpdateHandleOwner(GPOINTER_TO_UINT(key), GPOINTER_TO_UINT(value));
}

void csTPChannel::HandleOwnersChanged(GHashTable *aAdded, const GArray *aRemoved)
{
  if (m_GroupHandleOwnersChangeObservers) {
    PRUint32 length;
    m_GroupHandleOwnersChangeObservers->GetLength(&length);
    nsCOMPtr<csITPGroupHandleOwnersChangeObserver> observer;
 
    for (PRUint32 i = 0; i < length; i++) {
      observer = do_QueryElementAt(m_GroupHandleOwnersChangeObservers, i);
  
      g_hash_table_foreach(aAdded, HandleOwnersChangeHelperFunc, observer);
  
      unsigned int len = aRemoved->len;
      for (unsigned int i=0; i<len; i++)
        observer->DoRemoveHandleOwner(g_array_index(aRemoved, PRUint32, i));
    }
  }
}

static void 
GotHandleOwnersChangeSignal(TpChannel *proxy, GHashTable* aAdded,
                            const GArray *aRemoved, gpointer user_data, 
                            GObject *unused)
{
  csTPChannel *channel = (csTPChannel *)user_data;
  if (!channel)
    return;

  channel->HandleOwnersChanged(aAdded, aRemoved);
}

NS_IMETHODIMP csTPChannel::AddHandleOwnersChangeObserver(csITPGroupHandleOwnersChangeObserver *observer)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  NS_ENSURE_ARG_POINTER(observer);

  if (!m_GroupHandleOwnersChangeObservers) {
    m_GroupHandleOwnersChangeObservers = do_CreateInstance(NS_ARRAY_CONTRACTID);
    tp_cli_channel_interface_group_connect_to_handle_owners_changed(m_Channel,
                                              GotHandleOwnersChangeSignal,
                                              this, NULL, NULL, NULL);
  }

  m_GroupHandleOwnersChangeObservers->AppendElement(observer, PR_FALSE);
  return NS_OK;
}
CS_TELEPATHY_REMOVE_OBSERVER(csTPChannel, RemoveHandleOwnersChangeObserver,
                             csITPGroupHandleOwnersChangeObserver,
                             m_GroupHandleOwnersChangeObservers);

//////////////// Group Self Handle Change Signal

static void
GotSelfHandleChangeSignal(TpChannel *proxy, guint aSelfHandle, 
                          gpointer user_data, GObject *weak_object)
{
  csTPChannel *channel = (csTPChannel *)user_data;
  if (!channel)
    return;

  channel->SelfHandleChanged(aSelfHandle);
}

void csTPChannel::SelfHandleChanged(PRUint32 aSelfHandle)
{
  m_SelfHandle = aSelfHandle;

  if (m_GroupSelfHandleChangeObservers) {
    PRUint32 length;
    m_GroupSelfHandleChangeObservers->GetLength(&length);
    nsCOMPtr<csITPGroupSelfHandleChangeObserver> observer;
 
    for (PRUint32 i = 0; i < length; i++) {
      observer = do_QueryElementAt(m_GroupSelfHandleChangeObservers, i);
      observer->DoUpdateSelfHandle(aSelfHandle);
    }
  }
}

NS_IMETHODIMP csTPChannel::AddSelfHandleChangeObserver(csITPGroupSelfHandleChangeObserver *observer)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  NS_ENSURE_ARG_POINTER(observer);

  if (!m_GroupSelfHandleChangeObservers) {
    m_GroupSelfHandleChangeObservers = do_CreateInstance(NS_ARRAY_CONTRACTID);
    tp_cli_channel_interface_group_connect_to_self_handle_changed(m_Channel,
                                              GotSelfHandleChangeSignal,
                                              this, NULL, NULL, NULL);
  }

  m_GroupSelfHandleChangeObservers->AppendElement(observer, PR_FALSE);
  return NS_OK;
}
CS_TELEPATHY_REMOVE_OBSERVER(csTPChannel, RemoveSelfHandleChangeObserver,
                             csITPGroupSelfHandleChangeObserver,
                             m_GroupSelfHandleChangeObservers);

/*
 * csITPChannelChatState Implementation
 */

void 
csTPChannel::ChatStateChanged(PRUint32 aContact, PRUint32 aChatState)
{
  if (m_ChatStateObservers) {
    PRUint32 length;
    m_ChatStateObservers->GetLength(&length);
    nsCOMPtr<csITPChatStateObserver> observer;
 
    for (PRUint32 i = 0; i < length; i++) {
      observer = do_QueryElementAt(m_ChatStateObservers, i);
      observer->DoUpdateChatState(aContact, aChatState);
    }
  }
}

static void GotChatStateChangeSignal(TpChannel *proxy, guint aContact, 
                                     guint aState, gpointer user_data, 
                                     GObject *unused)
{
  csTPChannel *channel = (csTPChannel *)user_data;
  if (!channel)
    return;

  channel->ChatStateChanged(aContact, aState);
}

NS_IMETHODIMP csTPChannel::AddChatStateObserver(csITPChatStateObserver *observer)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  NS_ENSURE_ARG_POINTER(observer);

  if (!m_ChatStateObservers) {
    m_ChatStateObservers = do_CreateInstance(NS_ARRAY_CONTRACTID);
    tp_cli_channel_interface_chat_state_connect_to_chat_state_changed(m_Channel,
                                                  GotChatStateChangeSignal,
                                                  this, NULL, NULL, NULL);
  }

  m_ChatStateObservers->AppendElement(observer, PR_FALSE);
  return NS_OK;
}
CS_TELEPATHY_REMOVE_OBSERVER(csTPChannel, RemoveChatStateObserver,
                             csITPChatStateObserver, m_ChatStateObservers);

NS_IMETHODIMP csTPChannel::SetChatState(PRUint32 aChatState, csITPEmptyCB *cb)
{
    return NS_ERROR_NOT_IMPLEMENTED;
}

/*
 * csITPChannelPassword Implementation
 */

NS_IMETHODIMP csTPChannel::GetPasswordFlags(csITPIntegerCB *cb)
{
    return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP csTPChannel::ProvidePassword(const nsACString & aPassword, 
                                           csITPBooleanCB *cb)
{
    return NS_ERROR_NOT_IMPLEMENTED;
}

void 
csTPChannel::PasswordFlagsChanged(PRUint32 aFlagsAdded, PRUint32 aFlagsRemoved)
{
  if (m_PasswordFlagsChangeObservers) {
    PRUint32 length;
    m_PasswordFlagsChangeObservers->GetLength(&length);
    nsCOMPtr<csITPChannelPasswordFlagsChangeObserver> observer;
 
    for (PRUint32 i = 0; i < length; i++) {
      observer = do_QueryElementAt(m_PasswordFlagsChangeObservers, i);
      observer->DoUpdatePasswordFlags(aFlagsAdded, aFlagsRemoved);
    }
  }
}

static void 
GotPasswordFlagsChangeSignal(TpChannel *proxy, guint aFlagsAdded,
                             guint aFlagsRemoved, gpointer user_data, 
                             GObject *unused)
{
  csTPChannel *channel = (csTPChannel *)user_data;
  if (!channel)
    return;

  channel->PasswordFlagsChanged(aFlagsAdded, aFlagsRemoved);
}

NS_IMETHODIMP csTPChannel::AddFlagsChangeObserver(csITPChannelPasswordFlagsChangeObserver *observer)
{
  if (!m_Channel)
    return NS_ERROR_NOT_INITIALIZED;

  NS_ENSURE_ARG_POINTER(observer);

  if (!m_PasswordFlagsChangeObservers) {
    m_PasswordFlagsChangeObservers = do_CreateInstance(NS_ARRAY_CONTRACTID);
    tp_cli_channel_interface_password_connect_to_password_flags_changed(m_Channel,
                                                 GotPasswordFlagsChangeSignal,
                                                 this, NULL, NULL, NULL);
  }

  m_PasswordFlagsChangeObservers->AppendElement(observer, PR_FALSE);
  return NS_OK;
}
CS_TELEPATHY_REMOVE_OBSERVER(csTPChannel, RemoveFlagsChangeObserver,
                             csITPChannelPasswordFlagsChangeObserver,
                             m_PasswordFlagsChangeObservers);

/*
 * csITPProxy Implementation
 */

CS_TELEPATHY_IMPL_PROXY(csTPChannel, m_Channel);
