/* ========================================================================
 * Copyright (c) 2005-2022 The OPC Foundation, Inc. All rights reserved.
 *
 * OPC Foundation MIT License 1.00
 * 
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * The complete license agreement can be found here:
 * http://opcfoundation.org/License/MIT/1.00/
 * ======================================================================*/

using System;
using System.Collections.Generic;
using System.Text;
using System.Xml;
using System.Runtime.Serialization;

namespace Opc.Ua
{
    #region Enumeration Enumeration
    #if (!OPCUA_EXCLUDE_Enumeration)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum Enumeration
    {
    }

    #region EnumerationCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfEnumeration", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "Enumeration")]
    #if !NET_STANDARD
    public partial class EnumerationCollection : List<Enumeration>, ICloneable
    #else
    public partial class EnumerationCollection : List<Enumeration>
    #endif
    {
        #region Constructors
        /// <remarks />
        public EnumerationCollection() {}

        /// <remarks />
        public EnumerationCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public EnumerationCollection(IEnumerable<Enumeration> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator EnumerationCollection(Enumeration[] values)
        {
            if (values != null)
            {
                return new EnumerationCollection(values);
            }

            return new EnumerationCollection();
        }

        /// <remarks />
        public static explicit operator Enumeration[](EnumerationCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (EnumerationCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EnumerationCollection clone = new EnumerationCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((Enumeration)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region NamingRuleType Enumeration
    #if (!OPCUA_EXCLUDE_NamingRuleType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum NamingRuleType
    {
        /// <remarks />
        [EnumMember(Value = "Mandatory_1")]
        Mandatory = 1,

        /// <remarks />
        [EnumMember(Value = "Optional_2")]
        Optional = 2,

        /// <remarks />
        [EnumMember(Value = "Constraint_3")]
        Constraint = 3,
    }

    #region NamingRuleTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfNamingRuleType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "NamingRuleType")]
    #if !NET_STANDARD
    public partial class NamingRuleTypeCollection : List<NamingRuleType>, ICloneable
    #else
    public partial class NamingRuleTypeCollection : List<NamingRuleType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public NamingRuleTypeCollection() {}

        /// <remarks />
        public NamingRuleTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public NamingRuleTypeCollection(IEnumerable<NamingRuleType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator NamingRuleTypeCollection(NamingRuleType[] values)
        {
            if (values != null)
            {
                return new NamingRuleTypeCollection(values);
            }

            return new NamingRuleTypeCollection();
        }

        /// <remarks />
        public static explicit operator NamingRuleType[](NamingRuleTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (NamingRuleTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            NamingRuleTypeCollection clone = new NamingRuleTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((NamingRuleType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region Union Class
    #if (!OPCUA_EXCLUDE_Union)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class Union : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public Union()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.Union; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.Union_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.Union_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.Union_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            Union value = encodeable as Union;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (Union)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            Union clone = (Union)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region UnionCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfUnion", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "Union")]
    #if !NET_STANDARD
    public partial class UnionCollection : List<Union>, ICloneable
    #else
    public partial class UnionCollection : List<Union>
    #endif
    {
        #region Constructors
        /// <remarks />
        public UnionCollection() {}

        /// <remarks />
        public UnionCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public UnionCollection(IEnumerable<Union> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator UnionCollection(Union[] values)
        {
            if (values != null)
            {
                return new UnionCollection(values);
            }

            return new UnionCollection();
        }

        /// <remarks />
        public static explicit operator Union[](UnionCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (UnionCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UnionCollection clone = new UnionCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((Union)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region KeyValuePair Class
    #if (!OPCUA_EXCLUDE_KeyValuePair)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class KeyValuePair : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public KeyValuePair()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_key = null;
            m_value = Variant.Null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Key", IsRequired = false, Order = 1)]
        public QualifiedName Key
        {
            get { return m_key;  }
            set { m_key = value; }
        }

        /// <remarks />
        [DataMember(Name = "Value", IsRequired = false, Order = 2)]
        public Variant Value
        {
            get { return m_value;  }
            set { m_value = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.KeyValuePair; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.KeyValuePair_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.KeyValuePair_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.KeyValuePair_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteQualifiedName("Key", Key);
            encoder.WriteVariant("Value", Value);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Key = decoder.ReadQualifiedName("Key");
            Value = decoder.ReadVariant("Value");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            KeyValuePair value = encodeable as KeyValuePair;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_key, value.m_key)) return false;
            if (!Utils.IsEqual(m_value, value.m_value)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (KeyValuePair)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            KeyValuePair clone = (KeyValuePair)base.MemberwiseClone();

            clone.m_key = (QualifiedName)Utils.Clone(this.m_key);
            clone.m_value = (Variant)Utils.Clone(this.m_value);

            return clone;
        }
        #endregion

        #region Private Fields
        private QualifiedName m_key;
        private Variant m_value;
        #endregion
    }

    #region KeyValuePairCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfKeyValuePair", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "KeyValuePair")]
    #if !NET_STANDARD
    public partial class KeyValuePairCollection : List<KeyValuePair>, ICloneable
    #else
    public partial class KeyValuePairCollection : List<KeyValuePair>
    #endif
    {
        #region Constructors
        /// <remarks />
        public KeyValuePairCollection() {}

        /// <remarks />
        public KeyValuePairCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public KeyValuePairCollection(IEnumerable<KeyValuePair> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator KeyValuePairCollection(KeyValuePair[] values)
        {
            if (values != null)
            {
                return new KeyValuePairCollection(values);
            }

            return new KeyValuePairCollection();
        }

        /// <remarks />
        public static explicit operator KeyValuePair[](KeyValuePairCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (KeyValuePairCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            KeyValuePairCollection clone = new KeyValuePairCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((KeyValuePair)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region AdditionalParametersType Class
    #if (!OPCUA_EXCLUDE_AdditionalParametersType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class AdditionalParametersType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public AdditionalParametersType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_parameters = new KeyValuePairCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Parameters", IsRequired = false, Order = 1)]
        public KeyValuePairCollection Parameters
        {
            get
            {
                return m_parameters;
            }

            set
            {
                m_parameters = value;

                if (value == null)
                {
                    m_parameters = new KeyValuePairCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.AdditionalParametersType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.AdditionalParametersType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.AdditionalParametersType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.AdditionalParametersType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeableArray("Parameters", Parameters.ToArray(), typeof(KeyValuePair));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Parameters = (KeyValuePairCollection)decoder.ReadEncodeableArray("Parameters", typeof(KeyValuePair));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            AdditionalParametersType value = encodeable as AdditionalParametersType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_parameters, value.m_parameters)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (AdditionalParametersType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            AdditionalParametersType clone = (AdditionalParametersType)base.MemberwiseClone();

            clone.m_parameters = (KeyValuePairCollection)Utils.Clone(this.m_parameters);

            return clone;
        }
        #endregion

        #region Private Fields
        private KeyValuePairCollection m_parameters;
        #endregion
    }
    #endif
    #endregion

    #region EphemeralKeyType Class
    #if (!OPCUA_EXCLUDE_EphemeralKeyType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class EphemeralKeyType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public EphemeralKeyType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_publicKey = null;
            m_signature = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "PublicKey", IsRequired = false, Order = 1)]
        public byte[] PublicKey
        {
            get { return m_publicKey;  }
            set { m_publicKey = value; }
        }

        /// <remarks />
        [DataMember(Name = "Signature", IsRequired = false, Order = 2)]
        public byte[] Signature
        {
            get { return m_signature;  }
            set { m_signature = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.EphemeralKeyType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.EphemeralKeyType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.EphemeralKeyType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.EphemeralKeyType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteByteString("PublicKey", PublicKey);
            encoder.WriteByteString("Signature", Signature);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            PublicKey = decoder.ReadByteString("PublicKey");
            Signature = decoder.ReadByteString("Signature");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            EphemeralKeyType value = encodeable as EphemeralKeyType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_publicKey, value.m_publicKey)) return false;
            if (!Utils.IsEqual(m_signature, value.m_signature)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (EphemeralKeyType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EphemeralKeyType clone = (EphemeralKeyType)base.MemberwiseClone();

            clone.m_publicKey = (byte[])Utils.Clone(this.m_publicKey);
            clone.m_signature = (byte[])Utils.Clone(this.m_signature);

            return clone;
        }
        #endregion

        #region Private Fields
        private byte[] m_publicKey;
        private byte[] m_signature;
        #endregion
    }
    #endif
    #endregion

    #region EndpointType Class
    #if (!OPCUA_EXCLUDE_EndpointType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class EndpointType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public EndpointType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_endpointUrl = null;
            m_securityMode = MessageSecurityMode.Invalid;
            m_securityPolicyUri = null;
            m_transportProfileUri = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "EndpointUrl", IsRequired = false, Order = 1)]
        public string EndpointUrl
        {
            get { return m_endpointUrl;  }
            set { m_endpointUrl = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityMode", IsRequired = false, Order = 2)]
        public MessageSecurityMode SecurityMode
        {
            get { return m_securityMode;  }
            set { m_securityMode = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityPolicyUri", IsRequired = false, Order = 3)]
        public string SecurityPolicyUri
        {
            get { return m_securityPolicyUri;  }
            set { m_securityPolicyUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "TransportProfileUri", IsRequired = false, Order = 4)]
        public string TransportProfileUri
        {
            get { return m_transportProfileUri;  }
            set { m_transportProfileUri = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.EndpointType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.EndpointType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.EndpointType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.EndpointType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("EndpointUrl", EndpointUrl);
            encoder.WriteEnumerated("SecurityMode", SecurityMode);
            encoder.WriteString("SecurityPolicyUri", SecurityPolicyUri);
            encoder.WriteString("TransportProfileUri", TransportProfileUri);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            EndpointUrl = decoder.ReadString("EndpointUrl");
            SecurityMode = (MessageSecurityMode)decoder.ReadEnumerated("SecurityMode", typeof(MessageSecurityMode));
            SecurityPolicyUri = decoder.ReadString("SecurityPolicyUri");
            TransportProfileUri = decoder.ReadString("TransportProfileUri");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            EndpointType value = encodeable as EndpointType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_endpointUrl, value.m_endpointUrl)) return false;
            if (!Utils.IsEqual(m_securityMode, value.m_securityMode)) return false;
            if (!Utils.IsEqual(m_securityPolicyUri, value.m_securityPolicyUri)) return false;
            if (!Utils.IsEqual(m_transportProfileUri, value.m_transportProfileUri)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (EndpointType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EndpointType clone = (EndpointType)base.MemberwiseClone();

            clone.m_endpointUrl = (string)Utils.Clone(this.m_endpointUrl);
            clone.m_securityMode = (MessageSecurityMode)Utils.Clone(this.m_securityMode);
            clone.m_securityPolicyUri = (string)Utils.Clone(this.m_securityPolicyUri);
            clone.m_transportProfileUri = (string)Utils.Clone(this.m_transportProfileUri);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_endpointUrl;
        private MessageSecurityMode m_securityMode;
        private string m_securityPolicyUri;
        private string m_transportProfileUri;
        #endregion
    }

    #region EndpointTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfEndpointType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "EndpointType")]
    #if !NET_STANDARD
    public partial class EndpointTypeCollection : List<EndpointType>, ICloneable
    #else
    public partial class EndpointTypeCollection : List<EndpointType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public EndpointTypeCollection() {}

        /// <remarks />
        public EndpointTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public EndpointTypeCollection(IEnumerable<EndpointType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator EndpointTypeCollection(EndpointType[] values)
        {
            if (values != null)
            {
                return new EndpointTypeCollection(values);
            }

            return new EndpointTypeCollection();
        }

        /// <remarks />
        public static explicit operator EndpointType[](EndpointTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (EndpointTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EndpointTypeCollection clone = new EndpointTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((EndpointType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region RationalNumber Class
    #if (!OPCUA_EXCLUDE_RationalNumber)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class RationalNumber : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public RationalNumber()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_numerator = (int)0;
            m_denominator = (uint)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Numerator", IsRequired = false, Order = 1)]
        public int Numerator
        {
            get { return m_numerator;  }
            set { m_numerator = value; }
        }

        /// <remarks />
        [DataMember(Name = "Denominator", IsRequired = false, Order = 2)]
        public uint Denominator
        {
            get { return m_denominator;  }
            set { m_denominator = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.RationalNumber; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.RationalNumber_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.RationalNumber_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.RationalNumber_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteInt32("Numerator", Numerator);
            encoder.WriteUInt32("Denominator", Denominator);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Numerator = decoder.ReadInt32("Numerator");
            Denominator = decoder.ReadUInt32("Denominator");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            RationalNumber value = encodeable as RationalNumber;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_numerator, value.m_numerator)) return false;
            if (!Utils.IsEqual(m_denominator, value.m_denominator)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (RationalNumber)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            RationalNumber clone = (RationalNumber)base.MemberwiseClone();

            clone.m_numerator = (int)Utils.Clone(this.m_numerator);
            clone.m_denominator = (uint)Utils.Clone(this.m_denominator);

            return clone;
        }
        #endregion

        #region Private Fields
        private int m_numerator;
        private uint m_denominator;
        #endregion
    }

    #region RationalNumberCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfRationalNumber", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "RationalNumber")]
    #if !NET_STANDARD
    public partial class RationalNumberCollection : List<RationalNumber>, ICloneable
    #else
    public partial class RationalNumberCollection : List<RationalNumber>
    #endif
    {
        #region Constructors
        /// <remarks />
        public RationalNumberCollection() {}

        /// <remarks />
        public RationalNumberCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public RationalNumberCollection(IEnumerable<RationalNumber> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator RationalNumberCollection(RationalNumber[] values)
        {
            if (values != null)
            {
                return new RationalNumberCollection(values);
            }

            return new RationalNumberCollection();
        }

        /// <remarks />
        public static explicit operator RationalNumber[](RationalNumberCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (RationalNumberCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            RationalNumberCollection clone = new RationalNumberCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((RationalNumber)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region Vector Class
    #if (!OPCUA_EXCLUDE_Vector)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class Vector : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public Vector()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.Vector; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.Vector_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.Vector_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.Vector_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            Vector value = encodeable as Vector;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (Vector)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            Vector clone = (Vector)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region VectorCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfVector", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "Vector")]
    #if !NET_STANDARD
    public partial class VectorCollection : List<Vector>, ICloneable
    #else
    public partial class VectorCollection : List<Vector>
    #endif
    {
        #region Constructors
        /// <remarks />
        public VectorCollection() {}

        /// <remarks />
        public VectorCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public VectorCollection(IEnumerable<Vector> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator VectorCollection(Vector[] values)
        {
            if (values != null)
            {
                return new VectorCollection(values);
            }

            return new VectorCollection();
        }

        /// <remarks />
        public static explicit operator Vector[](VectorCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (VectorCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            VectorCollection clone = new VectorCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((Vector)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region ThreeDVector Class
    #if (!OPCUA_EXCLUDE_ThreeDVector)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ThreeDVector : Vector
    {
        #region Constructors
        /// <remarks />
        public ThreeDVector()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_x = (double)0;
            m_y = (double)0;
            m_z = (double)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "X", IsRequired = false, Order = 1)]
        public double X
        {
            get { return m_x;  }
            set { m_x = value; }
        }

        /// <remarks />
        [DataMember(Name = "Y", IsRequired = false, Order = 2)]
        public double Y
        {
            get { return m_y;  }
            set { m_y = value; }
        }

        /// <remarks />
        [DataMember(Name = "Z", IsRequired = false, Order = 3)]
        public double Z
        {
            get { return m_z;  }
            set { m_z = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.ThreeDVector; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.ThreeDVector_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.ThreeDVector_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.ThreeDVector_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteDouble("X", X);
            encoder.WriteDouble("Y", Y);
            encoder.WriteDouble("Z", Z);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            X = decoder.ReadDouble("X");
            Y = decoder.ReadDouble("Y");
            Z = decoder.ReadDouble("Z");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ThreeDVector value = encodeable as ThreeDVector;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_x, value.m_x)) return false;
            if (!Utils.IsEqual(m_y, value.m_y)) return false;
            if (!Utils.IsEqual(m_z, value.m_z)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (ThreeDVector)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ThreeDVector clone = (ThreeDVector)base.MemberwiseClone();

            clone.m_x = (double)Utils.Clone(this.m_x);
            clone.m_y = (double)Utils.Clone(this.m_y);
            clone.m_z = (double)Utils.Clone(this.m_z);

            return clone;
        }
        #endregion

        #region Private Fields
        private double m_x;
        private double m_y;
        private double m_z;
        #endregion
    }

    #region ThreeDVectorCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfThreeDVector", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ThreeDVector")]
    #if !NET_STANDARD
    public partial class ThreeDVectorCollection : List<ThreeDVector>, ICloneable
    #else
    public partial class ThreeDVectorCollection : List<ThreeDVector>
    #endif
    {
        #region Constructors
        /// <remarks />
        public ThreeDVectorCollection() {}

        /// <remarks />
        public ThreeDVectorCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public ThreeDVectorCollection(IEnumerable<ThreeDVector> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator ThreeDVectorCollection(ThreeDVector[] values)
        {
            if (values != null)
            {
                return new ThreeDVectorCollection(values);
            }

            return new ThreeDVectorCollection();
        }

        /// <remarks />
        public static explicit operator ThreeDVector[](ThreeDVectorCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (ThreeDVectorCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ThreeDVectorCollection clone = new ThreeDVectorCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ThreeDVector)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region CartesianCoordinates Class
    #if (!OPCUA_EXCLUDE_CartesianCoordinates)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class CartesianCoordinates : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public CartesianCoordinates()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.CartesianCoordinates; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.CartesianCoordinates_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.CartesianCoordinates_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.CartesianCoordinates_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            CartesianCoordinates value = encodeable as CartesianCoordinates;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (CartesianCoordinates)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            CartesianCoordinates clone = (CartesianCoordinates)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region CartesianCoordinatesCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfCartesianCoordinates", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "CartesianCoordinates")]
    #if !NET_STANDARD
    public partial class CartesianCoordinatesCollection : List<CartesianCoordinates>, ICloneable
    #else
    public partial class CartesianCoordinatesCollection : List<CartesianCoordinates>
    #endif
    {
        #region Constructors
        /// <remarks />
        public CartesianCoordinatesCollection() {}

        /// <remarks />
        public CartesianCoordinatesCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public CartesianCoordinatesCollection(IEnumerable<CartesianCoordinates> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator CartesianCoordinatesCollection(CartesianCoordinates[] values)
        {
            if (values != null)
            {
                return new CartesianCoordinatesCollection(values);
            }

            return new CartesianCoordinatesCollection();
        }

        /// <remarks />
        public static explicit operator CartesianCoordinates[](CartesianCoordinatesCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (CartesianCoordinatesCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            CartesianCoordinatesCollection clone = new CartesianCoordinatesCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((CartesianCoordinates)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region ThreeDCartesianCoordinates Class
    #if (!OPCUA_EXCLUDE_ThreeDCartesianCoordinates)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ThreeDCartesianCoordinates : CartesianCoordinates
    {
        #region Constructors
        /// <remarks />
        public ThreeDCartesianCoordinates()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_x = (double)0;
            m_y = (double)0;
            m_z = (double)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "X", IsRequired = false, Order = 1)]
        public double X
        {
            get { return m_x;  }
            set { m_x = value; }
        }

        /// <remarks />
        [DataMember(Name = "Y", IsRequired = false, Order = 2)]
        public double Y
        {
            get { return m_y;  }
            set { m_y = value; }
        }

        /// <remarks />
        [DataMember(Name = "Z", IsRequired = false, Order = 3)]
        public double Z
        {
            get { return m_z;  }
            set { m_z = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.ThreeDCartesianCoordinates; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.ThreeDCartesianCoordinates_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.ThreeDCartesianCoordinates_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.ThreeDCartesianCoordinates_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteDouble("X", X);
            encoder.WriteDouble("Y", Y);
            encoder.WriteDouble("Z", Z);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            X = decoder.ReadDouble("X");
            Y = decoder.ReadDouble("Y");
            Z = decoder.ReadDouble("Z");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ThreeDCartesianCoordinates value = encodeable as ThreeDCartesianCoordinates;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_x, value.m_x)) return false;
            if (!Utils.IsEqual(m_y, value.m_y)) return false;
            if (!Utils.IsEqual(m_z, value.m_z)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (ThreeDCartesianCoordinates)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ThreeDCartesianCoordinates clone = (ThreeDCartesianCoordinates)base.MemberwiseClone();

            clone.m_x = (double)Utils.Clone(this.m_x);
            clone.m_y = (double)Utils.Clone(this.m_y);
            clone.m_z = (double)Utils.Clone(this.m_z);

            return clone;
        }
        #endregion

        #region Private Fields
        private double m_x;
        private double m_y;
        private double m_z;
        #endregion
    }

    #region ThreeDCartesianCoordinatesCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfThreeDCartesianCoordinates", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ThreeDCartesianCoordinates")]
    #if !NET_STANDARD
    public partial class ThreeDCartesianCoordinatesCollection : List<ThreeDCartesianCoordinates>, ICloneable
    #else
    public partial class ThreeDCartesianCoordinatesCollection : List<ThreeDCartesianCoordinates>
    #endif
    {
        #region Constructors
        /// <remarks />
        public ThreeDCartesianCoordinatesCollection() {}

        /// <remarks />
        public ThreeDCartesianCoordinatesCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public ThreeDCartesianCoordinatesCollection(IEnumerable<ThreeDCartesianCoordinates> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator ThreeDCartesianCoordinatesCollection(ThreeDCartesianCoordinates[] values)
        {
            if (values != null)
            {
                return new ThreeDCartesianCoordinatesCollection(values);
            }

            return new ThreeDCartesianCoordinatesCollection();
        }

        /// <remarks />
        public static explicit operator ThreeDCartesianCoordinates[](ThreeDCartesianCoordinatesCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (ThreeDCartesianCoordinatesCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ThreeDCartesianCoordinatesCollection clone = new ThreeDCartesianCoordinatesCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ThreeDCartesianCoordinates)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region Orientation Class
    #if (!OPCUA_EXCLUDE_Orientation)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class Orientation : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public Orientation()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.Orientation; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.Orientation_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.Orientation_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.Orientation_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            Orientation value = encodeable as Orientation;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (Orientation)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            Orientation clone = (Orientation)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region OrientationCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfOrientation", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "Orientation")]
    #if !NET_STANDARD
    public partial class OrientationCollection : List<Orientation>, ICloneable
    #else
    public partial class OrientationCollection : List<Orientation>
    #endif
    {
        #region Constructors
        /// <remarks />
        public OrientationCollection() {}

        /// <remarks />
        public OrientationCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public OrientationCollection(IEnumerable<Orientation> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator OrientationCollection(Orientation[] values)
        {
            if (values != null)
            {
                return new OrientationCollection(values);
            }

            return new OrientationCollection();
        }

        /// <remarks />
        public static explicit operator Orientation[](OrientationCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (OrientationCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            OrientationCollection clone = new OrientationCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((Orientation)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region ThreeDOrientation Class
    #if (!OPCUA_EXCLUDE_ThreeDOrientation)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ThreeDOrientation : Orientation
    {
        #region Constructors
        /// <remarks />
        public ThreeDOrientation()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_a = (double)0;
            m_b = (double)0;
            m_c = (double)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "A", IsRequired = false, Order = 1)]
        public double A
        {
            get { return m_a;  }
            set { m_a = value; }
        }

        /// <remarks />
        [DataMember(Name = "B", IsRequired = false, Order = 2)]
        public double B
        {
            get { return m_b;  }
            set { m_b = value; }
        }

        /// <remarks />
        [DataMember(Name = "C", IsRequired = false, Order = 3)]
        public double C
        {
            get { return m_c;  }
            set { m_c = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.ThreeDOrientation; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.ThreeDOrientation_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.ThreeDOrientation_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.ThreeDOrientation_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteDouble("A", A);
            encoder.WriteDouble("B", B);
            encoder.WriteDouble("C", C);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            A = decoder.ReadDouble("A");
            B = decoder.ReadDouble("B");
            C = decoder.ReadDouble("C");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ThreeDOrientation value = encodeable as ThreeDOrientation;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_a, value.m_a)) return false;
            if (!Utils.IsEqual(m_b, value.m_b)) return false;
            if (!Utils.IsEqual(m_c, value.m_c)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (ThreeDOrientation)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ThreeDOrientation clone = (ThreeDOrientation)base.MemberwiseClone();

            clone.m_a = (double)Utils.Clone(this.m_a);
            clone.m_b = (double)Utils.Clone(this.m_b);
            clone.m_c = (double)Utils.Clone(this.m_c);

            return clone;
        }
        #endregion

        #region Private Fields
        private double m_a;
        private double m_b;
        private double m_c;
        #endregion
    }

    #region ThreeDOrientationCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfThreeDOrientation", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ThreeDOrientation")]
    #if !NET_STANDARD
    public partial class ThreeDOrientationCollection : List<ThreeDOrientation>, ICloneable
    #else
    public partial class ThreeDOrientationCollection : List<ThreeDOrientation>
    #endif
    {
        #region Constructors
        /// <remarks />
        public ThreeDOrientationCollection() {}

        /// <remarks />
        public ThreeDOrientationCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public ThreeDOrientationCollection(IEnumerable<ThreeDOrientation> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator ThreeDOrientationCollection(ThreeDOrientation[] values)
        {
            if (values != null)
            {
                return new ThreeDOrientationCollection(values);
            }

            return new ThreeDOrientationCollection();
        }

        /// <remarks />
        public static explicit operator ThreeDOrientation[](ThreeDOrientationCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (ThreeDOrientationCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ThreeDOrientationCollection clone = new ThreeDOrientationCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ThreeDOrientation)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region Frame Class
    #if (!OPCUA_EXCLUDE_Frame)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class Frame : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public Frame()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.Frame; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.Frame_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.Frame_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.Frame_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            Frame value = encodeable as Frame;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (Frame)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            Frame clone = (Frame)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region FrameCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfFrame", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "Frame")]
    #if !NET_STANDARD
    public partial class FrameCollection : List<Frame>, ICloneable
    #else
    public partial class FrameCollection : List<Frame>
    #endif
    {
        #region Constructors
        /// <remarks />
        public FrameCollection() {}

        /// <remarks />
        public FrameCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public FrameCollection(IEnumerable<Frame> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator FrameCollection(Frame[] values)
        {
            if (values != null)
            {
                return new FrameCollection(values);
            }

            return new FrameCollection();
        }

        /// <remarks />
        public static explicit operator Frame[](FrameCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (FrameCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            FrameCollection clone = new FrameCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((Frame)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region ThreeDFrame Class
    #if (!OPCUA_EXCLUDE_ThreeDFrame)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ThreeDFrame : Frame
    {
        #region Constructors
        /// <remarks />
        public ThreeDFrame()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_cartesianCoordinates = new ThreeDCartesianCoordinates();
            m_orientation = new ThreeDOrientation();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "CartesianCoordinates", IsRequired = false, Order = 1)]
        public ThreeDCartesianCoordinates CartesianCoordinates
        {
            get
            {
                return m_cartesianCoordinates;
            }

            set
            {
                m_cartesianCoordinates = value;

                if (value == null)
                {
                    m_cartesianCoordinates = new ThreeDCartesianCoordinates();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "Orientation", IsRequired = false, Order = 2)]
        public ThreeDOrientation Orientation
        {
            get
            {
                return m_orientation;
            }

            set
            {
                m_orientation = value;

                if (value == null)
                {
                    m_orientation = new ThreeDOrientation();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.ThreeDFrame; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.ThreeDFrame_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.ThreeDFrame_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.ThreeDFrame_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("CartesianCoordinates", CartesianCoordinates, typeof(ThreeDCartesianCoordinates));
            encoder.WriteEncodeable("Orientation", Orientation, typeof(ThreeDOrientation));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            CartesianCoordinates = (ThreeDCartesianCoordinates)decoder.ReadEncodeable("CartesianCoordinates", typeof(ThreeDCartesianCoordinates));
            Orientation = (ThreeDOrientation)decoder.ReadEncodeable("Orientation", typeof(ThreeDOrientation));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ThreeDFrame value = encodeable as ThreeDFrame;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_cartesianCoordinates, value.m_cartesianCoordinates)) return false;
            if (!Utils.IsEqual(m_orientation, value.m_orientation)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (ThreeDFrame)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ThreeDFrame clone = (ThreeDFrame)base.MemberwiseClone();

            clone.m_cartesianCoordinates = (ThreeDCartesianCoordinates)Utils.Clone(this.m_cartesianCoordinates);
            clone.m_orientation = (ThreeDOrientation)Utils.Clone(this.m_orientation);

            return clone;
        }
        #endregion

        #region Private Fields
        private ThreeDCartesianCoordinates m_cartesianCoordinates;
        private ThreeDOrientation m_orientation;
        #endregion
    }

    #region ThreeDFrameCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfThreeDFrame", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ThreeDFrame")]
    #if !NET_STANDARD
    public partial class ThreeDFrameCollection : List<ThreeDFrame>, ICloneable
    #else
    public partial class ThreeDFrameCollection : List<ThreeDFrame>
    #endif
    {
        #region Constructors
        /// <remarks />
        public ThreeDFrameCollection() {}

        /// <remarks />
        public ThreeDFrameCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public ThreeDFrameCollection(IEnumerable<ThreeDFrame> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator ThreeDFrameCollection(ThreeDFrame[] values)
        {
            if (values != null)
            {
                return new ThreeDFrameCollection(values);
            }

            return new ThreeDFrameCollection();
        }

        /// <remarks />
        public static explicit operator ThreeDFrame[](ThreeDFrameCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (ThreeDFrameCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ThreeDFrameCollection clone = new ThreeDFrameCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ThreeDFrame)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region OpenFileMode Enumeration
    #if (!OPCUA_EXCLUDE_OpenFileMode)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum OpenFileMode
    {
        /// <remarks />
        [EnumMember(Value = "Read_1")]
        Read = 1,

        /// <remarks />
        [EnumMember(Value = "Write_2")]
        Write = 2,

        /// <remarks />
        [EnumMember(Value = "EraseExisting_4")]
        EraseExisting = 4,

        /// <remarks />
        [EnumMember(Value = "Append_8")]
        Append = 8,
    }

    #region OpenFileModeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfOpenFileMode", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "OpenFileMode")]
    #if !NET_STANDARD
    public partial class OpenFileModeCollection : List<OpenFileMode>, ICloneable
    #else
    public partial class OpenFileModeCollection : List<OpenFileMode>
    #endif
    {
        #region Constructors
        /// <remarks />
        public OpenFileModeCollection() {}

        /// <remarks />
        public OpenFileModeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public OpenFileModeCollection(IEnumerable<OpenFileMode> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator OpenFileModeCollection(OpenFileMode[] values)
        {
            if (values != null)
            {
                return new OpenFileModeCollection(values);
            }

            return new OpenFileModeCollection();
        }

        /// <remarks />
        public static explicit operator OpenFileMode[](OpenFileModeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (OpenFileModeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            OpenFileModeCollection clone = new OpenFileModeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((OpenFileMode)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region IdentityCriteriaType Enumeration
    #if (!OPCUA_EXCLUDE_IdentityCriteriaType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum IdentityCriteriaType
    {
        /// <remarks />
        [EnumMember(Value = "UserName_1")]
        UserName = 1,

        /// <remarks />
        [EnumMember(Value = "Thumbprint_2")]
        Thumbprint = 2,

        /// <remarks />
        [EnumMember(Value = "Role_3")]
        Role = 3,

        /// <remarks />
        [EnumMember(Value = "GroupId_4")]
        GroupId = 4,

        /// <remarks />
        [EnumMember(Value = "Anonymous_5")]
        Anonymous = 5,

        /// <remarks />
        [EnumMember(Value = "AuthenticatedUser_6")]
        AuthenticatedUser = 6,

        /// <remarks />
        [EnumMember(Value = "Application_7")]
        Application = 7,

        /// <remarks />
        [EnumMember(Value = "X509Subject_8")]
        X509Subject = 8,
    }

    #region IdentityCriteriaTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfIdentityCriteriaType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "IdentityCriteriaType")]
    #if !NET_STANDARD
    public partial class IdentityCriteriaTypeCollection : List<IdentityCriteriaType>, ICloneable
    #else
    public partial class IdentityCriteriaTypeCollection : List<IdentityCriteriaType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public IdentityCriteriaTypeCollection() {}

        /// <remarks />
        public IdentityCriteriaTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public IdentityCriteriaTypeCollection(IEnumerable<IdentityCriteriaType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator IdentityCriteriaTypeCollection(IdentityCriteriaType[] values)
        {
            if (values != null)
            {
                return new IdentityCriteriaTypeCollection(values);
            }

            return new IdentityCriteriaTypeCollection();
        }

        /// <remarks />
        public static explicit operator IdentityCriteriaType[](IdentityCriteriaTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (IdentityCriteriaTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            IdentityCriteriaTypeCollection clone = new IdentityCriteriaTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((IdentityCriteriaType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region IdentityMappingRuleType Class
    #if (!OPCUA_EXCLUDE_IdentityMappingRuleType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class IdentityMappingRuleType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public IdentityMappingRuleType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_criteriaType = IdentityCriteriaType.UserName;
            m_criteria = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "CriteriaType", IsRequired = false, Order = 1)]
        public IdentityCriteriaType CriteriaType
        {
            get { return m_criteriaType;  }
            set { m_criteriaType = value; }
        }

        /// <remarks />
        [DataMember(Name = "Criteria", IsRequired = false, Order = 2)]
        public string Criteria
        {
            get { return m_criteria;  }
            set { m_criteria = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.IdentityMappingRuleType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.IdentityMappingRuleType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.IdentityMappingRuleType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.IdentityMappingRuleType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEnumerated("CriteriaType", CriteriaType);
            encoder.WriteString("Criteria", Criteria);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            CriteriaType = (IdentityCriteriaType)decoder.ReadEnumerated("CriteriaType", typeof(IdentityCriteriaType));
            Criteria = decoder.ReadString("Criteria");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            IdentityMappingRuleType value = encodeable as IdentityMappingRuleType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_criteriaType, value.m_criteriaType)) return false;
            if (!Utils.IsEqual(m_criteria, value.m_criteria)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (IdentityMappingRuleType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            IdentityMappingRuleType clone = (IdentityMappingRuleType)base.MemberwiseClone();

            clone.m_criteriaType = (IdentityCriteriaType)Utils.Clone(this.m_criteriaType);
            clone.m_criteria = (string)Utils.Clone(this.m_criteria);

            return clone;
        }
        #endregion

        #region Private Fields
        private IdentityCriteriaType m_criteriaType;
        private string m_criteria;
        #endregion
    }

    #region IdentityMappingRuleTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfIdentityMappingRuleType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "IdentityMappingRuleType")]
    #if !NET_STANDARD
    public partial class IdentityMappingRuleTypeCollection : List<IdentityMappingRuleType>, ICloneable
    #else
    public partial class IdentityMappingRuleTypeCollection : List<IdentityMappingRuleType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public IdentityMappingRuleTypeCollection() {}

        /// <remarks />
        public IdentityMappingRuleTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public IdentityMappingRuleTypeCollection(IEnumerable<IdentityMappingRuleType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator IdentityMappingRuleTypeCollection(IdentityMappingRuleType[] values)
        {
            if (values != null)
            {
                return new IdentityMappingRuleTypeCollection(values);
            }

            return new IdentityMappingRuleTypeCollection();
        }

        /// <remarks />
        public static explicit operator IdentityMappingRuleType[](IdentityMappingRuleTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (IdentityMappingRuleTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            IdentityMappingRuleTypeCollection clone = new IdentityMappingRuleTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((IdentityMappingRuleType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region CurrencyUnitType Class
    #if (!OPCUA_EXCLUDE_CurrencyUnitType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class CurrencyUnitType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public CurrencyUnitType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_numericCode = (short)0;
            m_exponent = (sbyte)0;
            m_alphabeticCode = null;
            m_currency = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "NumericCode", IsRequired = false, Order = 1)]
        public short NumericCode
        {
            get { return m_numericCode;  }
            set { m_numericCode = value; }
        }

        /// <remarks />
        [DataMember(Name = "Exponent", IsRequired = false, Order = 2)]
        public sbyte Exponent
        {
            get { return m_exponent;  }
            set { m_exponent = value; }
        }

        /// <remarks />
        [DataMember(Name = "AlphabeticCode", IsRequired = false, Order = 3)]
        public string AlphabeticCode
        {
            get { return m_alphabeticCode;  }
            set { m_alphabeticCode = value; }
        }

        /// <remarks />
        [DataMember(Name = "Currency", IsRequired = false, Order = 4)]
        public LocalizedText Currency
        {
            get { return m_currency;  }
            set { m_currency = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.CurrencyUnitType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.CurrencyUnitType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.CurrencyUnitType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.CurrencyUnitType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteInt16("NumericCode", NumericCode);
            encoder.WriteSByte("Exponent", Exponent);
            encoder.WriteString("AlphabeticCode", AlphabeticCode);
            encoder.WriteLocalizedText("Currency", Currency);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            NumericCode = decoder.ReadInt16("NumericCode");
            Exponent = decoder.ReadSByte("Exponent");
            AlphabeticCode = decoder.ReadString("AlphabeticCode");
            Currency = decoder.ReadLocalizedText("Currency");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            CurrencyUnitType value = encodeable as CurrencyUnitType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_numericCode, value.m_numericCode)) return false;
            if (!Utils.IsEqual(m_exponent, value.m_exponent)) return false;
            if (!Utils.IsEqual(m_alphabeticCode, value.m_alphabeticCode)) return false;
            if (!Utils.IsEqual(m_currency, value.m_currency)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (CurrencyUnitType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            CurrencyUnitType clone = (CurrencyUnitType)base.MemberwiseClone();

            clone.m_numericCode = (short)Utils.Clone(this.m_numericCode);
            clone.m_exponent = (sbyte)Utils.Clone(this.m_exponent);
            clone.m_alphabeticCode = (string)Utils.Clone(this.m_alphabeticCode);
            clone.m_currency = (LocalizedText)Utils.Clone(this.m_currency);

            return clone;
        }
        #endregion

        #region Private Fields
        private short m_numericCode;
        private sbyte m_exponent;
        private string m_alphabeticCode;
        private LocalizedText m_currency;
        #endregion
    }

    #region CurrencyUnitTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfCurrencyUnitType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "CurrencyUnitType")]
    #if !NET_STANDARD
    public partial class CurrencyUnitTypeCollection : List<CurrencyUnitType>, ICloneable
    #else
    public partial class CurrencyUnitTypeCollection : List<CurrencyUnitType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public CurrencyUnitTypeCollection() {}

        /// <remarks />
        public CurrencyUnitTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public CurrencyUnitTypeCollection(IEnumerable<CurrencyUnitType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator CurrencyUnitTypeCollection(CurrencyUnitType[] values)
        {
            if (values != null)
            {
                return new CurrencyUnitTypeCollection(values);
            }

            return new CurrencyUnitTypeCollection();
        }

        /// <remarks />
        public static explicit operator CurrencyUnitType[](CurrencyUnitTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (CurrencyUnitTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            CurrencyUnitTypeCollection clone = new CurrencyUnitTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((CurrencyUnitType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region AlarmMask Enumeration
    #if (!OPCUA_EXCLUDE_AlarmMask)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum AlarmMask : UInt16
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "Active_1")]
        Active = 1,

        /// <remarks />
        [EnumMember(Value = "Unacknowledged_2")]
        Unacknowledged = 2,

        /// <remarks />
        [EnumMember(Value = "Unconfirmed_4")]
        Unconfirmed = 4,
    }

    #region AlarmMaskCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfAlarmMask", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "AlarmMask")]
    #if !NET_STANDARD
    public partial class AlarmMaskCollection : List<AlarmMask>, ICloneable
    #else
    public partial class AlarmMaskCollection : List<AlarmMask>
    #endif
    {
        #region Constructors
        /// <remarks />
        public AlarmMaskCollection() {}

        /// <remarks />
        public AlarmMaskCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public AlarmMaskCollection(IEnumerable<AlarmMask> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator AlarmMaskCollection(AlarmMask[] values)
        {
            if (values != null)
            {
                return new AlarmMaskCollection(values);
            }

            return new AlarmMaskCollection();
        }

        /// <remarks />
        public static explicit operator AlarmMask[](AlarmMaskCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (AlarmMaskCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            AlarmMaskCollection clone = new AlarmMaskCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((AlarmMask)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region TrustListValidationOptions Enumeration
    #if (!OPCUA_EXCLUDE_TrustListValidationOptions)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum TrustListValidationOptions : UInt32
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "SuppressCertificateExpired_1")]
        SuppressCertificateExpired = 1,

        /// <remarks />
        [EnumMember(Value = "SuppressHostNameInvalid_2")]
        SuppressHostNameInvalid = 2,

        /// <remarks />
        [EnumMember(Value = "SuppressRevocationStatusUnknown_4")]
        SuppressRevocationStatusUnknown = 4,

        /// <remarks />
        [EnumMember(Value = "SuppressIssuerCertificateExpired_8")]
        SuppressIssuerCertificateExpired = 8,

        /// <remarks />
        [EnumMember(Value = "SuppressIssuerRevocationStatusUnknown_16")]
        SuppressIssuerRevocationStatusUnknown = 16,

        /// <remarks />
        [EnumMember(Value = "CheckRevocationStatusOnline_32")]
        CheckRevocationStatusOnline = 32,

        /// <remarks />
        [EnumMember(Value = "CheckRevocationStatusOffline_64")]
        CheckRevocationStatusOffline = 64,
    }
    #endif
    #endregion

    #region TrustListMasks Enumeration
    #if (!OPCUA_EXCLUDE_TrustListMasks)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum TrustListMasks
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "TrustedCertificates_1")]
        TrustedCertificates = 1,

        /// <remarks />
        [EnumMember(Value = "TrustedCrls_2")]
        TrustedCrls = 2,

        /// <remarks />
        [EnumMember(Value = "IssuerCertificates_4")]
        IssuerCertificates = 4,

        /// <remarks />
        [EnumMember(Value = "IssuerCrls_8")]
        IssuerCrls = 8,

        /// <remarks />
        [EnumMember(Value = "All_15")]
        All = 15,
    }
    #endif
    #endregion

    #region TrustListDataType Class
    #if (!OPCUA_EXCLUDE_TrustListDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class TrustListDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public TrustListDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_specifiedLists = (uint)0;
            m_trustedCertificates = new ByteStringCollection();
            m_trustedCrls = new ByteStringCollection();
            m_issuerCertificates = new ByteStringCollection();
            m_issuerCrls = new ByteStringCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "SpecifiedLists", IsRequired = false, Order = 1)]
        public uint SpecifiedLists
        {
            get { return m_specifiedLists;  }
            set { m_specifiedLists = value; }
        }

        /// <remarks />
        [DataMember(Name = "TrustedCertificates", IsRequired = false, Order = 2)]
        public ByteStringCollection TrustedCertificates
        {
            get
            {
                return m_trustedCertificates;
            }

            set
            {
                m_trustedCertificates = value;

                if (value == null)
                {
                    m_trustedCertificates = new ByteStringCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "TrustedCrls", IsRequired = false, Order = 3)]
        public ByteStringCollection TrustedCrls
        {
            get
            {
                return m_trustedCrls;
            }

            set
            {
                m_trustedCrls = value;

                if (value == null)
                {
                    m_trustedCrls = new ByteStringCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "IssuerCertificates", IsRequired = false, Order = 4)]
        public ByteStringCollection IssuerCertificates
        {
            get
            {
                return m_issuerCertificates;
            }

            set
            {
                m_issuerCertificates = value;

                if (value == null)
                {
                    m_issuerCertificates = new ByteStringCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "IssuerCrls", IsRequired = false, Order = 5)]
        public ByteStringCollection IssuerCrls
        {
            get
            {
                return m_issuerCrls;
            }

            set
            {
                m_issuerCrls = value;

                if (value == null)
                {
                    m_issuerCrls = new ByteStringCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.TrustListDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.TrustListDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.TrustListDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.TrustListDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt32("SpecifiedLists", SpecifiedLists);
            encoder.WriteByteStringArray("TrustedCertificates", TrustedCertificates);
            encoder.WriteByteStringArray("TrustedCrls", TrustedCrls);
            encoder.WriteByteStringArray("IssuerCertificates", IssuerCertificates);
            encoder.WriteByteStringArray("IssuerCrls", IssuerCrls);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            SpecifiedLists = decoder.ReadUInt32("SpecifiedLists");
            TrustedCertificates = decoder.ReadByteStringArray("TrustedCertificates");
            TrustedCrls = decoder.ReadByteStringArray("TrustedCrls");
            IssuerCertificates = decoder.ReadByteStringArray("IssuerCertificates");
            IssuerCrls = decoder.ReadByteStringArray("IssuerCrls");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            TrustListDataType value = encodeable as TrustListDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_specifiedLists, value.m_specifiedLists)) return false;
            if (!Utils.IsEqual(m_trustedCertificates, value.m_trustedCertificates)) return false;
            if (!Utils.IsEqual(m_trustedCrls, value.m_trustedCrls)) return false;
            if (!Utils.IsEqual(m_issuerCertificates, value.m_issuerCertificates)) return false;
            if (!Utils.IsEqual(m_issuerCrls, value.m_issuerCrls)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (TrustListDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TrustListDataType clone = (TrustListDataType)base.MemberwiseClone();

            clone.m_specifiedLists = (uint)Utils.Clone(this.m_specifiedLists);
            clone.m_trustedCertificates = (ByteStringCollection)Utils.Clone(this.m_trustedCertificates);
            clone.m_trustedCrls = (ByteStringCollection)Utils.Clone(this.m_trustedCrls);
            clone.m_issuerCertificates = (ByteStringCollection)Utils.Clone(this.m_issuerCertificates);
            clone.m_issuerCrls = (ByteStringCollection)Utils.Clone(this.m_issuerCrls);

            return clone;
        }
        #endregion

        #region Private Fields
        private uint m_specifiedLists;
        private ByteStringCollection m_trustedCertificates;
        private ByteStringCollection m_trustedCrls;
        private ByteStringCollection m_issuerCertificates;
        private ByteStringCollection m_issuerCrls;
        #endregion
    }

    #region TrustListDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfTrustListDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "TrustListDataType")]
    #if !NET_STANDARD
    public partial class TrustListDataTypeCollection : List<TrustListDataType>, ICloneable
    #else
    public partial class TrustListDataTypeCollection : List<TrustListDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public TrustListDataTypeCollection() {}

        /// <remarks />
        public TrustListDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public TrustListDataTypeCollection(IEnumerable<TrustListDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator TrustListDataTypeCollection(TrustListDataType[] values)
        {
            if (values != null)
            {
                return new TrustListDataTypeCollection(values);
            }

            return new TrustListDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator TrustListDataType[](TrustListDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (TrustListDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TrustListDataTypeCollection clone = new TrustListDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((TrustListDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region TransactionErrorType Class
    #if (!OPCUA_EXCLUDE_TransactionErrorType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class TransactionErrorType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public TransactionErrorType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_targetId = null;
            m_error = StatusCodes.Good;
            m_message = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "TargetId", IsRequired = false, Order = 1)]
        public NodeId TargetId
        {
            get { return m_targetId;  }
            set { m_targetId = value; }
        }

        /// <remarks />
        [DataMember(Name = "Error", IsRequired = false, Order = 2)]
        public StatusCode Error
        {
            get { return m_error;  }
            set { m_error = value; }
        }

        /// <remarks />
        [DataMember(Name = "Message", IsRequired = false, Order = 3)]
        public LocalizedText Message
        {
            get { return m_message;  }
            set { m_message = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.TransactionErrorType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.TransactionErrorType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.TransactionErrorType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.TransactionErrorType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteNodeId("TargetId", TargetId);
            encoder.WriteStatusCode("Error", Error);
            encoder.WriteLocalizedText("Message", Message);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            TargetId = decoder.ReadNodeId("TargetId");
            Error = decoder.ReadStatusCode("Error");
            Message = decoder.ReadLocalizedText("Message");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            TransactionErrorType value = encodeable as TransactionErrorType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_targetId, value.m_targetId)) return false;
            if (!Utils.IsEqual(m_error, value.m_error)) return false;
            if (!Utils.IsEqual(m_message, value.m_message)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (TransactionErrorType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TransactionErrorType clone = (TransactionErrorType)base.MemberwiseClone();

            clone.m_targetId = (NodeId)Utils.Clone(this.m_targetId);
            clone.m_error = (StatusCode)Utils.Clone(this.m_error);
            clone.m_message = (LocalizedText)Utils.Clone(this.m_message);

            return clone;
        }
        #endregion

        #region Private Fields
        private NodeId m_targetId;
        private StatusCode m_error;
        private LocalizedText m_message;
        #endregion
    }

    #region TransactionErrorTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfTransactionErrorType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "TransactionErrorType")]
    #if !NET_STANDARD
    public partial class TransactionErrorTypeCollection : List<TransactionErrorType>, ICloneable
    #else
    public partial class TransactionErrorTypeCollection : List<TransactionErrorType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public TransactionErrorTypeCollection() {}

        /// <remarks />
        public TransactionErrorTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public TransactionErrorTypeCollection(IEnumerable<TransactionErrorType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator TransactionErrorTypeCollection(TransactionErrorType[] values)
        {
            if (values != null)
            {
                return new TransactionErrorTypeCollection(values);
            }

            return new TransactionErrorTypeCollection();
        }

        /// <remarks />
        public static explicit operator TransactionErrorType[](TransactionErrorTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (TransactionErrorTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TransactionErrorTypeCollection clone = new TransactionErrorTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((TransactionErrorType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DecimalDataType Class
    #if (!OPCUA_EXCLUDE_DecimalDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DecimalDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public DecimalDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_scale = (short)0;
            m_value = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Scale", IsRequired = false, Order = 1)]
        public short Scale
        {
            get { return m_scale;  }
            set { m_scale = value; }
        }

        /// <remarks />
        [DataMember(Name = "Value", IsRequired = false, Order = 2)]
        public byte[] Value
        {
            get { return m_value;  }
            set { m_value = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.DecimalDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.DecimalDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.DecimalDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.DecimalDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteInt16("Scale", Scale);
            encoder.WriteByteString("Value", Value);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Scale = decoder.ReadInt16("Scale");
            Value = decoder.ReadByteString("Value");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DecimalDataType value = encodeable as DecimalDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_scale, value.m_scale)) return false;
            if (!Utils.IsEqual(m_value, value.m_value)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (DecimalDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DecimalDataType clone = (DecimalDataType)base.MemberwiseClone();

            clone.m_scale = (short)Utils.Clone(this.m_scale);
            clone.m_value = (byte[])Utils.Clone(this.m_value);

            return clone;
        }
        #endregion

        #region Private Fields
        private short m_scale;
        private byte[] m_value;
        #endregion
    }
    #endif
    #endregion

    #region DataTypeSchemaHeader Class
    #if (!OPCUA_EXCLUDE_DataTypeSchemaHeader)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DataTypeSchemaHeader : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public DataTypeSchemaHeader()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_namespaces = new StringCollection();
            m_structureDataTypes = new StructureDescriptionCollection();
            m_enumDataTypes = new EnumDescriptionCollection();
            m_simpleDataTypes = new SimpleTypeDescriptionCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Namespaces", IsRequired = false, Order = 1)]
        public StringCollection Namespaces
        {
            get
            {
                return m_namespaces;
            }

            set
            {
                m_namespaces = value;

                if (value == null)
                {
                    m_namespaces = new StringCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "StructureDataTypes", IsRequired = false, Order = 2)]
        public StructureDescriptionCollection StructureDataTypes
        {
            get
            {
                return m_structureDataTypes;
            }

            set
            {
                m_structureDataTypes = value;

                if (value == null)
                {
                    m_structureDataTypes = new StructureDescriptionCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "EnumDataTypes", IsRequired = false, Order = 3)]
        public EnumDescriptionCollection EnumDataTypes
        {
            get
            {
                return m_enumDataTypes;
            }

            set
            {
                m_enumDataTypes = value;

                if (value == null)
                {
                    m_enumDataTypes = new EnumDescriptionCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "SimpleDataTypes", IsRequired = false, Order = 4)]
        public SimpleTypeDescriptionCollection SimpleDataTypes
        {
            get
            {
                return m_simpleDataTypes;
            }

            set
            {
                m_simpleDataTypes = value;

                if (value == null)
                {
                    m_simpleDataTypes = new SimpleTypeDescriptionCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.DataTypeSchemaHeader; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.DataTypeSchemaHeader_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.DataTypeSchemaHeader_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.DataTypeSchemaHeader_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteStringArray("Namespaces", Namespaces);
            encoder.WriteEncodeableArray("StructureDataTypes", StructureDataTypes.ToArray(), typeof(StructureDescription));
            encoder.WriteEncodeableArray("EnumDataTypes", EnumDataTypes.ToArray(), typeof(EnumDescription));
            encoder.WriteEncodeableArray("SimpleDataTypes", SimpleDataTypes.ToArray(), typeof(SimpleTypeDescription));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Namespaces = decoder.ReadStringArray("Namespaces");
            StructureDataTypes = (StructureDescriptionCollection)decoder.ReadEncodeableArray("StructureDataTypes", typeof(StructureDescription));
            EnumDataTypes = (EnumDescriptionCollection)decoder.ReadEncodeableArray("EnumDataTypes", typeof(EnumDescription));
            SimpleDataTypes = (SimpleTypeDescriptionCollection)decoder.ReadEncodeableArray("SimpleDataTypes", typeof(SimpleTypeDescription));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DataTypeSchemaHeader value = encodeable as DataTypeSchemaHeader;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_namespaces, value.m_namespaces)) return false;
            if (!Utils.IsEqual(m_structureDataTypes, value.m_structureDataTypes)) return false;
            if (!Utils.IsEqual(m_enumDataTypes, value.m_enumDataTypes)) return false;
            if (!Utils.IsEqual(m_simpleDataTypes, value.m_simpleDataTypes)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (DataTypeSchemaHeader)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataTypeSchemaHeader clone = (DataTypeSchemaHeader)base.MemberwiseClone();

            clone.m_namespaces = (StringCollection)Utils.Clone(this.m_namespaces);
            clone.m_structureDataTypes = (StructureDescriptionCollection)Utils.Clone(this.m_structureDataTypes);
            clone.m_enumDataTypes = (EnumDescriptionCollection)Utils.Clone(this.m_enumDataTypes);
            clone.m_simpleDataTypes = (SimpleTypeDescriptionCollection)Utils.Clone(this.m_simpleDataTypes);

            return clone;
        }
        #endregion

        #region Private Fields
        private StringCollection m_namespaces;
        private StructureDescriptionCollection m_structureDataTypes;
        private EnumDescriptionCollection m_enumDataTypes;
        private SimpleTypeDescriptionCollection m_simpleDataTypes;
        #endregion
    }

    #region DataTypeSchemaHeaderCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDataTypeSchemaHeader", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DataTypeSchemaHeader")]
    #if !NET_STANDARD
    public partial class DataTypeSchemaHeaderCollection : List<DataTypeSchemaHeader>, ICloneable
    #else
    public partial class DataTypeSchemaHeaderCollection : List<DataTypeSchemaHeader>
    #endif
    {
        #region Constructors
        /// <remarks />
        public DataTypeSchemaHeaderCollection() {}

        /// <remarks />
        public DataTypeSchemaHeaderCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public DataTypeSchemaHeaderCollection(IEnumerable<DataTypeSchemaHeader> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator DataTypeSchemaHeaderCollection(DataTypeSchemaHeader[] values)
        {
            if (values != null)
            {
                return new DataTypeSchemaHeaderCollection(values);
            }

            return new DataTypeSchemaHeaderCollection();
        }

        /// <remarks />
        public static explicit operator DataTypeSchemaHeader[](DataTypeSchemaHeaderCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (DataTypeSchemaHeaderCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataTypeSchemaHeaderCollection clone = new DataTypeSchemaHeaderCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DataTypeSchemaHeader)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DataTypeDescription Class
    #if (!OPCUA_EXCLUDE_DataTypeDescription)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DataTypeDescription : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public DataTypeDescription()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_dataTypeId = null;
            m_name = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "DataTypeId", IsRequired = false, Order = 1)]
        public NodeId DataTypeId
        {
            get { return m_dataTypeId;  }
            set { m_dataTypeId = value; }
        }

        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 2)]
        public QualifiedName Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.DataTypeDescription; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.DataTypeDescription_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.DataTypeDescription_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.DataTypeDescription_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteNodeId("DataTypeId", DataTypeId);
            encoder.WriteQualifiedName("Name", Name);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            DataTypeId = decoder.ReadNodeId("DataTypeId");
            Name = decoder.ReadQualifiedName("Name");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DataTypeDescription value = encodeable as DataTypeDescription;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_dataTypeId, value.m_dataTypeId)) return false;
            if (!Utils.IsEqual(m_name, value.m_name)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (DataTypeDescription)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataTypeDescription clone = (DataTypeDescription)base.MemberwiseClone();

            clone.m_dataTypeId = (NodeId)Utils.Clone(this.m_dataTypeId);
            clone.m_name = (QualifiedName)Utils.Clone(this.m_name);

            return clone;
        }
        #endregion

        #region Private Fields
        private NodeId m_dataTypeId;
        private QualifiedName m_name;
        #endregion
    }

    #region DataTypeDescriptionCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDataTypeDescription", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DataTypeDescription")]
    #if !NET_STANDARD
    public partial class DataTypeDescriptionCollection : List<DataTypeDescription>, ICloneable
    #else
    public partial class DataTypeDescriptionCollection : List<DataTypeDescription>
    #endif
    {
        #region Constructors
        /// <remarks />
        public DataTypeDescriptionCollection() {}

        /// <remarks />
        public DataTypeDescriptionCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public DataTypeDescriptionCollection(IEnumerable<DataTypeDescription> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator DataTypeDescriptionCollection(DataTypeDescription[] values)
        {
            if (values != null)
            {
                return new DataTypeDescriptionCollection(values);
            }

            return new DataTypeDescriptionCollection();
        }

        /// <remarks />
        public static explicit operator DataTypeDescription[](DataTypeDescriptionCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (DataTypeDescriptionCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataTypeDescriptionCollection clone = new DataTypeDescriptionCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DataTypeDescription)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region StructureDescription Class
    #if (!OPCUA_EXCLUDE_StructureDescription)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class StructureDescription : DataTypeDescription
    {
        #region Constructors
        /// <remarks />
        public StructureDescription()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_structureDefinition = new StructureDefinition();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "StructureDefinition", IsRequired = false, Order = 1)]
        public StructureDefinition StructureDefinition
        {
            get
            {
                return m_structureDefinition;
            }

            set
            {
                m_structureDefinition = value;

                if (value == null)
                {
                    m_structureDefinition = new StructureDefinition();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.StructureDescription; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.StructureDescription_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.StructureDescription_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.StructureDescription_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("StructureDefinition", StructureDefinition, typeof(StructureDefinition));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            StructureDefinition = (StructureDefinition)decoder.ReadEncodeable("StructureDefinition", typeof(StructureDefinition));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            StructureDescription value = encodeable as StructureDescription;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_structureDefinition, value.m_structureDefinition)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (StructureDescription)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            StructureDescription clone = (StructureDescription)base.MemberwiseClone();

            clone.m_structureDefinition = (StructureDefinition)Utils.Clone(this.m_structureDefinition);

            return clone;
        }
        #endregion

        #region Private Fields
        private StructureDefinition m_structureDefinition;
        #endregion
    }

    #region StructureDescriptionCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfStructureDescription", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "StructureDescription")]
    #if !NET_STANDARD
    public partial class StructureDescriptionCollection : List<StructureDescription>, ICloneable
    #else
    public partial class StructureDescriptionCollection : List<StructureDescription>
    #endif
    {
        #region Constructors
        /// <remarks />
        public StructureDescriptionCollection() {}

        /// <remarks />
        public StructureDescriptionCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public StructureDescriptionCollection(IEnumerable<StructureDescription> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator StructureDescriptionCollection(StructureDescription[] values)
        {
            if (values != null)
            {
                return new StructureDescriptionCollection(values);
            }

            return new StructureDescriptionCollection();
        }

        /// <remarks />
        public static explicit operator StructureDescription[](StructureDescriptionCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (StructureDescriptionCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            StructureDescriptionCollection clone = new StructureDescriptionCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((StructureDescription)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region EnumDescription Class
    #if (!OPCUA_EXCLUDE_EnumDescription)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class EnumDescription : DataTypeDescription
    {
        #region Constructors
        /// <remarks />
        public EnumDescription()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_enumDefinition = new EnumDefinition();
            m_builtInType = (byte)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "EnumDefinition", IsRequired = false, Order = 1)]
        public EnumDefinition EnumDefinition
        {
            get
            {
                return m_enumDefinition;
            }

            set
            {
                m_enumDefinition = value;

                if (value == null)
                {
                    m_enumDefinition = new EnumDefinition();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "BuiltInType", IsRequired = false, Order = 2)]
        public byte BuiltInType
        {
            get { return m_builtInType;  }
            set { m_builtInType = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.EnumDescription; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.EnumDescription_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.EnumDescription_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.EnumDescription_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("EnumDefinition", EnumDefinition, typeof(EnumDefinition));
            encoder.WriteByte("BuiltInType", BuiltInType);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            EnumDefinition = (EnumDefinition)decoder.ReadEncodeable("EnumDefinition", typeof(EnumDefinition));
            BuiltInType = decoder.ReadByte("BuiltInType");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            EnumDescription value = encodeable as EnumDescription;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_enumDefinition, value.m_enumDefinition)) return false;
            if (!Utils.IsEqual(m_builtInType, value.m_builtInType)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (EnumDescription)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EnumDescription clone = (EnumDescription)base.MemberwiseClone();

            clone.m_enumDefinition = (EnumDefinition)Utils.Clone(this.m_enumDefinition);
            clone.m_builtInType = (byte)Utils.Clone(this.m_builtInType);

            return clone;
        }
        #endregion

        #region Private Fields
        private EnumDefinition m_enumDefinition;
        private byte m_builtInType;
        #endregion
    }

    #region EnumDescriptionCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfEnumDescription", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "EnumDescription")]
    #if !NET_STANDARD
    public partial class EnumDescriptionCollection : List<EnumDescription>, ICloneable
    #else
    public partial class EnumDescriptionCollection : List<EnumDescription>
    #endif
    {
        #region Constructors
        /// <remarks />
        public EnumDescriptionCollection() {}

        /// <remarks />
        public EnumDescriptionCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public EnumDescriptionCollection(IEnumerable<EnumDescription> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator EnumDescriptionCollection(EnumDescription[] values)
        {
            if (values != null)
            {
                return new EnumDescriptionCollection(values);
            }

            return new EnumDescriptionCollection();
        }

        /// <remarks />
        public static explicit operator EnumDescription[](EnumDescriptionCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (EnumDescriptionCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EnumDescriptionCollection clone = new EnumDescriptionCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((EnumDescription)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region SimpleTypeDescription Class
    #if (!OPCUA_EXCLUDE_SimpleTypeDescription)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class SimpleTypeDescription : DataTypeDescription
    {
        #region Constructors
        /// <remarks />
        public SimpleTypeDescription()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_baseDataType = null;
            m_builtInType = (byte)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "BaseDataType", IsRequired = false, Order = 1)]
        public NodeId BaseDataType
        {
            get { return m_baseDataType;  }
            set { m_baseDataType = value; }
        }

        /// <remarks />
        [DataMember(Name = "BuiltInType", IsRequired = false, Order = 2)]
        public byte BuiltInType
        {
            get { return m_builtInType;  }
            set { m_builtInType = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.SimpleTypeDescription; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.SimpleTypeDescription_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.SimpleTypeDescription_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.SimpleTypeDescription_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteNodeId("BaseDataType", BaseDataType);
            encoder.WriteByte("BuiltInType", BuiltInType);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            BaseDataType = decoder.ReadNodeId("BaseDataType");
            BuiltInType = decoder.ReadByte("BuiltInType");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            SimpleTypeDescription value = encodeable as SimpleTypeDescription;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_baseDataType, value.m_baseDataType)) return false;
            if (!Utils.IsEqual(m_builtInType, value.m_builtInType)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (SimpleTypeDescription)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            SimpleTypeDescription clone = (SimpleTypeDescription)base.MemberwiseClone();

            clone.m_baseDataType = (NodeId)Utils.Clone(this.m_baseDataType);
            clone.m_builtInType = (byte)Utils.Clone(this.m_builtInType);

            return clone;
        }
        #endregion

        #region Private Fields
        private NodeId m_baseDataType;
        private byte m_builtInType;
        #endregion
    }

    #region SimpleTypeDescriptionCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfSimpleTypeDescription", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "SimpleTypeDescription")]
    #if !NET_STANDARD
    public partial class SimpleTypeDescriptionCollection : List<SimpleTypeDescription>, ICloneable
    #else
    public partial class SimpleTypeDescriptionCollection : List<SimpleTypeDescription>
    #endif
    {
        #region Constructors
        /// <remarks />
        public SimpleTypeDescriptionCollection() {}

        /// <remarks />
        public SimpleTypeDescriptionCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public SimpleTypeDescriptionCollection(IEnumerable<SimpleTypeDescription> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator SimpleTypeDescriptionCollection(SimpleTypeDescription[] values)
        {
            if (values != null)
            {
                return new SimpleTypeDescriptionCollection(values);
            }

            return new SimpleTypeDescriptionCollection();
        }

        /// <remarks />
        public static explicit operator SimpleTypeDescription[](SimpleTypeDescriptionCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (SimpleTypeDescriptionCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            SimpleTypeDescriptionCollection clone = new SimpleTypeDescriptionCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((SimpleTypeDescription)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region UABinaryFileDataType Class
    #if (!OPCUA_EXCLUDE_UABinaryFileDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class UABinaryFileDataType : DataTypeSchemaHeader
    {
        #region Constructors
        /// <remarks />
        public UABinaryFileDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_schemaLocation = null;
            m_fileHeader = new KeyValuePairCollection();
            m_body = Variant.Null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "SchemaLocation", IsRequired = false, Order = 1)]
        public string SchemaLocation
        {
            get { return m_schemaLocation;  }
            set { m_schemaLocation = value; }
        }

        /// <remarks />
        [DataMember(Name = "FileHeader", IsRequired = false, Order = 2)]
        public KeyValuePairCollection FileHeader
        {
            get
            {
                return m_fileHeader;
            }

            set
            {
                m_fileHeader = value;

                if (value == null)
                {
                    m_fileHeader = new KeyValuePairCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "Body", IsRequired = false, Order = 3)]
        public Variant Body
        {
            get { return m_body;  }
            set { m_body = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.UABinaryFileDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.UABinaryFileDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.UABinaryFileDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.UABinaryFileDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("SchemaLocation", SchemaLocation);
            encoder.WriteEncodeableArray("FileHeader", FileHeader.ToArray(), typeof(KeyValuePair));
            encoder.WriteVariant("Body", Body);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            SchemaLocation = decoder.ReadString("SchemaLocation");
            FileHeader = (KeyValuePairCollection)decoder.ReadEncodeableArray("FileHeader", typeof(KeyValuePair));
            Body = decoder.ReadVariant("Body");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            UABinaryFileDataType value = encodeable as UABinaryFileDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_schemaLocation, value.m_schemaLocation)) return false;
            if (!Utils.IsEqual(m_fileHeader, value.m_fileHeader)) return false;
            if (!Utils.IsEqual(m_body, value.m_body)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (UABinaryFileDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UABinaryFileDataType clone = (UABinaryFileDataType)base.MemberwiseClone();

            clone.m_schemaLocation = (string)Utils.Clone(this.m_schemaLocation);
            clone.m_fileHeader = (KeyValuePairCollection)Utils.Clone(this.m_fileHeader);
            clone.m_body = (Variant)Utils.Clone(this.m_body);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_schemaLocation;
        private KeyValuePairCollection m_fileHeader;
        private Variant m_body;
        #endregion
    }

    #region UABinaryFileDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfUABinaryFileDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "UABinaryFileDataType")]
    #if !NET_STANDARD
    public partial class UABinaryFileDataTypeCollection : List<UABinaryFileDataType>, ICloneable
    #else
    public partial class UABinaryFileDataTypeCollection : List<UABinaryFileDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public UABinaryFileDataTypeCollection() {}

        /// <remarks />
        public UABinaryFileDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public UABinaryFileDataTypeCollection(IEnumerable<UABinaryFileDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator UABinaryFileDataTypeCollection(UABinaryFileDataType[] values)
        {
            if (values != null)
            {
                return new UABinaryFileDataTypeCollection(values);
            }

            return new UABinaryFileDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator UABinaryFileDataType[](UABinaryFileDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (UABinaryFileDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UABinaryFileDataTypeCollection clone = new UABinaryFileDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((UABinaryFileDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PortableQualifiedName Class
    #if (!OPCUA_EXCLUDE_PortableQualifiedName)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class PortableQualifiedName : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public PortableQualifiedName()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_namespaceUri = null;
            m_name = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "NamespaceUri", IsRequired = false, Order = 1)]
        public string NamespaceUri
        {
            get { return m_namespaceUri;  }
            set { m_namespaceUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 2)]
        public string Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.PortableQualifiedName; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.PortableQualifiedName_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.PortableQualifiedName_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.PortableQualifiedName_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("NamespaceUri", NamespaceUri);
            encoder.WriteString("Name", Name);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            NamespaceUri = decoder.ReadString("NamespaceUri");
            Name = decoder.ReadString("Name");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            PortableQualifiedName value = encodeable as PortableQualifiedName;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_namespaceUri, value.m_namespaceUri)) return false;
            if (!Utils.IsEqual(m_name, value.m_name)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (PortableQualifiedName)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PortableQualifiedName clone = (PortableQualifiedName)base.MemberwiseClone();

            clone.m_namespaceUri = (string)Utils.Clone(this.m_namespaceUri);
            clone.m_name = (string)Utils.Clone(this.m_name);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_namespaceUri;
        private string m_name;
        #endregion
    }

    #region PortableQualifiedNameCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPortableQualifiedName", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PortableQualifiedName")]
    #if !NET_STANDARD
    public partial class PortableQualifiedNameCollection : List<PortableQualifiedName>, ICloneable
    #else
    public partial class PortableQualifiedNameCollection : List<PortableQualifiedName>
    #endif
    {
        #region Constructors
        /// <remarks />
        public PortableQualifiedNameCollection() {}

        /// <remarks />
        public PortableQualifiedNameCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public PortableQualifiedNameCollection(IEnumerable<PortableQualifiedName> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator PortableQualifiedNameCollection(PortableQualifiedName[] values)
        {
            if (values != null)
            {
                return new PortableQualifiedNameCollection(values);
            }

            return new PortableQualifiedNameCollection();
        }

        /// <remarks />
        public static explicit operator PortableQualifiedName[](PortableQualifiedNameCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (PortableQualifiedNameCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PortableQualifiedNameCollection clone = new PortableQualifiedNameCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PortableQualifiedName)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PortableNodeId Class
    #if (!OPCUA_EXCLUDE_PortableNodeId)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class PortableNodeId : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public PortableNodeId()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_namespaceUri = null;
            m_identifier = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "NamespaceUri", IsRequired = false, Order = 1)]
        public string NamespaceUri
        {
            get { return m_namespaceUri;  }
            set { m_namespaceUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "Identifier", IsRequired = false, Order = 2)]
        public NodeId Identifier
        {
            get { return m_identifier;  }
            set { m_identifier = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.PortableNodeId; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.PortableNodeId_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.PortableNodeId_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.PortableNodeId_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("NamespaceUri", NamespaceUri);
            encoder.WriteNodeId("Identifier", Identifier);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            NamespaceUri = decoder.ReadString("NamespaceUri");
            Identifier = decoder.ReadNodeId("Identifier");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            PortableNodeId value = encodeable as PortableNodeId;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_namespaceUri, value.m_namespaceUri)) return false;
            if (!Utils.IsEqual(m_identifier, value.m_identifier)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (PortableNodeId)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PortableNodeId clone = (PortableNodeId)base.MemberwiseClone();

            clone.m_namespaceUri = (string)Utils.Clone(this.m_namespaceUri);
            clone.m_identifier = (NodeId)Utils.Clone(this.m_identifier);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_namespaceUri;
        private NodeId m_identifier;
        #endregion
    }

    #region PortableNodeIdCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPortableNodeId", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PortableNodeId")]
    #if !NET_STANDARD
    public partial class PortableNodeIdCollection : List<PortableNodeId>, ICloneable
    #else
    public partial class PortableNodeIdCollection : List<PortableNodeId>
    #endif
    {
        #region Constructors
        /// <remarks />
        public PortableNodeIdCollection() {}

        /// <remarks />
        public PortableNodeIdCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public PortableNodeIdCollection(IEnumerable<PortableNodeId> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator PortableNodeIdCollection(PortableNodeId[] values)
        {
            if (values != null)
            {
                return new PortableNodeIdCollection(values);
            }

            return new PortableNodeIdCollection();
        }

        /// <remarks />
        public static explicit operator PortableNodeId[](PortableNodeIdCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (PortableNodeIdCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PortableNodeIdCollection clone = new PortableNodeIdCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PortableNodeId)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region UnsignedRationalNumber Class
    #if (!OPCUA_EXCLUDE_UnsignedRationalNumber)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class UnsignedRationalNumber : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public UnsignedRationalNumber()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_numerator = (uint)0;
            m_denominator = (uint)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Numerator", IsRequired = false, Order = 1)]
        public uint Numerator
        {
            get { return m_numerator;  }
            set { m_numerator = value; }
        }

        /// <remarks />
        [DataMember(Name = "Denominator", IsRequired = false, Order = 2)]
        public uint Denominator
        {
            get { return m_denominator;  }
            set { m_denominator = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.UnsignedRationalNumber; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.UnsignedRationalNumber_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.UnsignedRationalNumber_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.UnsignedRationalNumber_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt32("Numerator", Numerator);
            encoder.WriteUInt32("Denominator", Denominator);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Numerator = decoder.ReadUInt32("Numerator");
            Denominator = decoder.ReadUInt32("Denominator");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            UnsignedRationalNumber value = encodeable as UnsignedRationalNumber;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_numerator, value.m_numerator)) return false;
            if (!Utils.IsEqual(m_denominator, value.m_denominator)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (UnsignedRationalNumber)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UnsignedRationalNumber clone = (UnsignedRationalNumber)base.MemberwiseClone();

            clone.m_numerator = (uint)Utils.Clone(this.m_numerator);
            clone.m_denominator = (uint)Utils.Clone(this.m_denominator);

            return clone;
        }
        #endregion

        #region Private Fields
        private uint m_numerator;
        private uint m_denominator;
        #endregion
    }

    #region UnsignedRationalNumberCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfUnsignedRationalNumber", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "UnsignedRationalNumber")]
    #if !NET_STANDARD
    public partial class UnsignedRationalNumberCollection : List<UnsignedRationalNumber>, ICloneable
    #else
    public partial class UnsignedRationalNumberCollection : List<UnsignedRationalNumber>
    #endif
    {
        #region Constructors
        /// <remarks />
        public UnsignedRationalNumberCollection() {}

        /// <remarks />
        public UnsignedRationalNumberCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public UnsignedRationalNumberCollection(IEnumerable<UnsignedRationalNumber> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator UnsignedRationalNumberCollection(UnsignedRationalNumber[] values)
        {
            if (values != null)
            {
                return new UnsignedRationalNumberCollection(values);
            }

            return new UnsignedRationalNumberCollection();
        }

        /// <remarks />
        public static explicit operator UnsignedRationalNumber[](UnsignedRationalNumberCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (UnsignedRationalNumberCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UnsignedRationalNumberCollection clone = new UnsignedRationalNumberCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((UnsignedRationalNumber)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PubSubState Enumeration
    #if (!OPCUA_EXCLUDE_PubSubState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum PubSubState
    {
        /// <remarks />
        [EnumMember(Value = "Disabled_0")]
        Disabled = 0,

        /// <remarks />
        [EnumMember(Value = "Paused_1")]
        Paused = 1,

        /// <remarks />
        [EnumMember(Value = "Operational_2")]
        Operational = 2,

        /// <remarks />
        [EnumMember(Value = "Error_3")]
        Error = 3,

        /// <remarks />
        [EnumMember(Value = "PreOperational_4")]
        PreOperational = 4,
    }

    #region PubSubStateCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPubSubState", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PubSubState")]
    #if !NET_STANDARD
    public partial class PubSubStateCollection : List<PubSubState>, ICloneable
    #else
    public partial class PubSubStateCollection : List<PubSubState>
    #endif
    {
        #region Constructors
        /// <remarks />
        public PubSubStateCollection() {}

        /// <remarks />
        public PubSubStateCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public PubSubStateCollection(IEnumerable<PubSubState> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator PubSubStateCollection(PubSubState[] values)
        {
            if (values != null)
            {
                return new PubSubStateCollection(values);
            }

            return new PubSubStateCollection();
        }

        /// <remarks />
        public static explicit operator PubSubState[](PubSubStateCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (PubSubStateCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PubSubStateCollection clone = new PubSubStateCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PubSubState)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DataSetMetaDataType Class
    #if (!OPCUA_EXCLUDE_DataSetMetaDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DataSetMetaDataType : DataTypeSchemaHeader
    {
        #region Constructors
        /// <remarks />
        public DataSetMetaDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_name = null;
            m_description = null;
            m_fields = new FieldMetaDataCollection();
            m_dataSetClassId = Uuid.Empty;
            m_configurationVersion = new ConfigurationVersionDataType();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 1)]
        public string Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }

        /// <remarks />
        [DataMember(Name = "Description", IsRequired = false, Order = 2)]
        public LocalizedText Description
        {
            get { return m_description;  }
            set { m_description = value; }
        }

        /// <remarks />
        [DataMember(Name = "Fields", IsRequired = false, Order = 3)]
        public FieldMetaDataCollection Fields
        {
            get
            {
                return m_fields;
            }

            set
            {
                m_fields = value;

                if (value == null)
                {
                    m_fields = new FieldMetaDataCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "DataSetClassId", IsRequired = false, Order = 4)]
        public Uuid DataSetClassId
        {
            get { return m_dataSetClassId;  }
            set { m_dataSetClassId = value; }
        }

        /// <remarks />
        [DataMember(Name = "ConfigurationVersion", IsRequired = false, Order = 5)]
        public ConfigurationVersionDataType ConfigurationVersion
        {
            get
            {
                return m_configurationVersion;
            }

            set
            {
                m_configurationVersion = value;

                if (value == null)
                {
                    m_configurationVersion = new ConfigurationVersionDataType();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.DataSetMetaDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.DataSetMetaDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.DataSetMetaDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.DataSetMetaDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("Name", Name);
            encoder.WriteLocalizedText("Description", Description);
            encoder.WriteEncodeableArray("Fields", Fields.ToArray(), typeof(FieldMetaData));
            encoder.WriteGuid("DataSetClassId", DataSetClassId);
            encoder.WriteEncodeable("ConfigurationVersion", ConfigurationVersion, typeof(ConfigurationVersionDataType));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Name = decoder.ReadString("Name");
            Description = decoder.ReadLocalizedText("Description");
            Fields = (FieldMetaDataCollection)decoder.ReadEncodeableArray("Fields", typeof(FieldMetaData));
            DataSetClassId = decoder.ReadGuid("DataSetClassId");
            ConfigurationVersion = (ConfigurationVersionDataType)decoder.ReadEncodeable("ConfigurationVersion", typeof(ConfigurationVersionDataType));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DataSetMetaDataType value = encodeable as DataSetMetaDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_name, value.m_name)) return false;
            if (!Utils.IsEqual(m_description, value.m_description)) return false;
            if (!Utils.IsEqual(m_fields, value.m_fields)) return false;
            if (!Utils.IsEqual(m_dataSetClassId, value.m_dataSetClassId)) return false;
            if (!Utils.IsEqual(m_configurationVersion, value.m_configurationVersion)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (DataSetMetaDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetMetaDataType clone = (DataSetMetaDataType)base.MemberwiseClone();

            clone.m_name = (string)Utils.Clone(this.m_name);
            clone.m_description = (LocalizedText)Utils.Clone(this.m_description);
            clone.m_fields = (FieldMetaDataCollection)Utils.Clone(this.m_fields);
            clone.m_dataSetClassId = (Uuid)Utils.Clone(this.m_dataSetClassId);
            clone.m_configurationVersion = (ConfigurationVersionDataType)Utils.Clone(this.m_configurationVersion);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_name;
        private LocalizedText m_description;
        private FieldMetaDataCollection m_fields;
        private Uuid m_dataSetClassId;
        private ConfigurationVersionDataType m_configurationVersion;
        #endregion
    }

    #region DataSetMetaDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDataSetMetaDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DataSetMetaDataType")]
    #if !NET_STANDARD
    public partial class DataSetMetaDataTypeCollection : List<DataSetMetaDataType>, ICloneable
    #else
    public partial class DataSetMetaDataTypeCollection : List<DataSetMetaDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public DataSetMetaDataTypeCollection() {}

        /// <remarks />
        public DataSetMetaDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public DataSetMetaDataTypeCollection(IEnumerable<DataSetMetaDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator DataSetMetaDataTypeCollection(DataSetMetaDataType[] values)
        {
            if (values != null)
            {
                return new DataSetMetaDataTypeCollection(values);
            }

            return new DataSetMetaDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator DataSetMetaDataType[](DataSetMetaDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (DataSetMetaDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetMetaDataTypeCollection clone = new DataSetMetaDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DataSetMetaDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region FieldMetaData Class
    #if (!OPCUA_EXCLUDE_FieldMetaData)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class FieldMetaData : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public FieldMetaData()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_name = null;
            m_description = null;
            m_fieldFlags = 0;
            m_builtInType = (byte)0;
            m_dataType = null;
            m_valueRank = (int)0;
            m_arrayDimensions = new UInt32Collection();
            m_maxStringLength = (uint)0;
            m_dataSetFieldId = Uuid.Empty;
            m_properties = new KeyValuePairCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 1)]
        public string Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }

        /// <remarks />
        [DataMember(Name = "Description", IsRequired = false, Order = 2)]
        public LocalizedText Description
        {
            get { return m_description;  }
            set { m_description = value; }
        }

        /// <remarks />
        [DataMember(Name = "FieldFlags", IsRequired = false, Order = 3)]
        public ushort FieldFlags
        {
            get { return m_fieldFlags;  }
            set { m_fieldFlags = value; }
        }

        /// <remarks />
        [DataMember(Name = "BuiltInType", IsRequired = false, Order = 4)]
        public byte BuiltInType
        {
            get { return m_builtInType;  }
            set { m_builtInType = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataType", IsRequired = false, Order = 5)]
        public NodeId DataType
        {
            get { return m_dataType;  }
            set { m_dataType = value; }
        }

        /// <remarks />
        [DataMember(Name = "ValueRank", IsRequired = false, Order = 6)]
        public int ValueRank
        {
            get { return m_valueRank;  }
            set { m_valueRank = value; }
        }

        /// <remarks />
        [DataMember(Name = "ArrayDimensions", IsRequired = false, Order = 7)]
        public UInt32Collection ArrayDimensions
        {
            get
            {
                return m_arrayDimensions;
            }

            set
            {
                m_arrayDimensions = value;

                if (value == null)
                {
                    m_arrayDimensions = new UInt32Collection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "MaxStringLength", IsRequired = false, Order = 8)]
        public uint MaxStringLength
        {
            get { return m_maxStringLength;  }
            set { m_maxStringLength = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetFieldId", IsRequired = false, Order = 9)]
        public Uuid DataSetFieldId
        {
            get { return m_dataSetFieldId;  }
            set { m_dataSetFieldId = value; }
        }

        /// <remarks />
        [DataMember(Name = "Properties", IsRequired = false, Order = 10)]
        public KeyValuePairCollection Properties
        {
            get
            {
                return m_properties;
            }

            set
            {
                m_properties = value;

                if (value == null)
                {
                    m_properties = new KeyValuePairCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.FieldMetaData; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.FieldMetaData_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.FieldMetaData_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.FieldMetaData_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("Name", Name);
            encoder.WriteLocalizedText("Description", Description);
            encoder.WriteUInt16("FieldFlags", FieldFlags);
            encoder.WriteByte("BuiltInType", BuiltInType);
            encoder.WriteNodeId("DataType", DataType);
            encoder.WriteInt32("ValueRank", ValueRank);
            encoder.WriteUInt32Array("ArrayDimensions", ArrayDimensions);
            encoder.WriteUInt32("MaxStringLength", MaxStringLength);
            encoder.WriteGuid("DataSetFieldId", DataSetFieldId);
            encoder.WriteEncodeableArray("Properties", Properties.ToArray(), typeof(KeyValuePair));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Name = decoder.ReadString("Name");
            Description = decoder.ReadLocalizedText("Description");
            FieldFlags = decoder.ReadUInt16("FieldFlags");
            BuiltInType = decoder.ReadByte("BuiltInType");
            DataType = decoder.ReadNodeId("DataType");
            ValueRank = decoder.ReadInt32("ValueRank");
            ArrayDimensions = decoder.ReadUInt32Array("ArrayDimensions");
            MaxStringLength = decoder.ReadUInt32("MaxStringLength");
            DataSetFieldId = decoder.ReadGuid("DataSetFieldId");
            Properties = (KeyValuePairCollection)decoder.ReadEncodeableArray("Properties", typeof(KeyValuePair));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            FieldMetaData value = encodeable as FieldMetaData;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_name, value.m_name)) return false;
            if (!Utils.IsEqual(m_description, value.m_description)) return false;
            if (!Utils.IsEqual(m_fieldFlags, value.m_fieldFlags)) return false;
            if (!Utils.IsEqual(m_builtInType, value.m_builtInType)) return false;
            if (!Utils.IsEqual(m_dataType, value.m_dataType)) return false;
            if (!Utils.IsEqual(m_valueRank, value.m_valueRank)) return false;
            if (!Utils.IsEqual(m_arrayDimensions, value.m_arrayDimensions)) return false;
            if (!Utils.IsEqual(m_maxStringLength, value.m_maxStringLength)) return false;
            if (!Utils.IsEqual(m_dataSetFieldId, value.m_dataSetFieldId)) return false;
            if (!Utils.IsEqual(m_properties, value.m_properties)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (FieldMetaData)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            FieldMetaData clone = (FieldMetaData)base.MemberwiseClone();

            clone.m_name = (string)Utils.Clone(this.m_name);
            clone.m_description = (LocalizedText)Utils.Clone(this.m_description);
            clone.m_fieldFlags = (ushort)Utils.Clone(this.m_fieldFlags);
            clone.m_builtInType = (byte)Utils.Clone(this.m_builtInType);
            clone.m_dataType = (NodeId)Utils.Clone(this.m_dataType);
            clone.m_valueRank = (int)Utils.Clone(this.m_valueRank);
            clone.m_arrayDimensions = (UInt32Collection)Utils.Clone(this.m_arrayDimensions);
            clone.m_maxStringLength = (uint)Utils.Clone(this.m_maxStringLength);
            clone.m_dataSetFieldId = (Uuid)Utils.Clone(this.m_dataSetFieldId);
            clone.m_properties = (KeyValuePairCollection)Utils.Clone(this.m_properties);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_name;
        private LocalizedText m_description;
        private ushort m_fieldFlags;
        private byte m_builtInType;
        private NodeId m_dataType;
        private int m_valueRank;
        private UInt32Collection m_arrayDimensions;
        private uint m_maxStringLength;
        private Uuid m_dataSetFieldId;
        private KeyValuePairCollection m_properties;
        #endregion
    }

    #region FieldMetaDataCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfFieldMetaData", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "FieldMetaData")]
    #if !NET_STANDARD
    public partial class FieldMetaDataCollection : List<FieldMetaData>, ICloneable
    #else
    public partial class FieldMetaDataCollection : List<FieldMetaData>
    #endif
    {
        #region Constructors
        /// <remarks />
        public FieldMetaDataCollection() {}

        /// <remarks />
        public FieldMetaDataCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public FieldMetaDataCollection(IEnumerable<FieldMetaData> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator FieldMetaDataCollection(FieldMetaData[] values)
        {
            if (values != null)
            {
                return new FieldMetaDataCollection(values);
            }

            return new FieldMetaDataCollection();
        }

        /// <remarks />
        public static explicit operator FieldMetaData[](FieldMetaDataCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (FieldMetaDataCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            FieldMetaDataCollection clone = new FieldMetaDataCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((FieldMetaData)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DataSetFieldFlags Enumeration
    #if (!OPCUA_EXCLUDE_DataSetFieldFlags)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum DataSetFieldFlags : UInt16
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "PromotedField_1")]
        PromotedField = 1,
    }
    #endif
    #endregion

    #region ConfigurationVersionDataType Class
    #if (!OPCUA_EXCLUDE_ConfigurationVersionDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ConfigurationVersionDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public ConfigurationVersionDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_majorVersion = (uint)0;
            m_minorVersion = (uint)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "MajorVersion", IsRequired = false, Order = 1)]
        public uint MajorVersion
        {
            get { return m_majorVersion;  }
            set { m_majorVersion = value; }
        }

        /// <remarks />
        [DataMember(Name = "MinorVersion", IsRequired = false, Order = 2)]
        public uint MinorVersion
        {
            get { return m_minorVersion;  }
            set { m_minorVersion = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.ConfigurationVersionDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.ConfigurationVersionDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.ConfigurationVersionDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.ConfigurationVersionDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt32("MajorVersion", MajorVersion);
            encoder.WriteUInt32("MinorVersion", MinorVersion);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            MajorVersion = decoder.ReadUInt32("MajorVersion");
            MinorVersion = decoder.ReadUInt32("MinorVersion");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ConfigurationVersionDataType value = encodeable as ConfigurationVersionDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_majorVersion, value.m_majorVersion)) return false;
            if (!Utils.IsEqual(m_minorVersion, value.m_minorVersion)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (ConfigurationVersionDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ConfigurationVersionDataType clone = (ConfigurationVersionDataType)base.MemberwiseClone();

            clone.m_majorVersion = (uint)Utils.Clone(this.m_majorVersion);
            clone.m_minorVersion = (uint)Utils.Clone(this.m_minorVersion);

            return clone;
        }
        #endregion

        #region Private Fields
        private uint m_majorVersion;
        private uint m_minorVersion;
        #endregion
    }

    #region ConfigurationVersionDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfConfigurationVersionDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ConfigurationVersionDataType")]
    #if !NET_STANDARD
    public partial class ConfigurationVersionDataTypeCollection : List<ConfigurationVersionDataType>, ICloneable
    #else
    public partial class ConfigurationVersionDataTypeCollection : List<ConfigurationVersionDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public ConfigurationVersionDataTypeCollection() {}

        /// <remarks />
        public ConfigurationVersionDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public ConfigurationVersionDataTypeCollection(IEnumerable<ConfigurationVersionDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator ConfigurationVersionDataTypeCollection(ConfigurationVersionDataType[] values)
        {
            if (values != null)
            {
                return new ConfigurationVersionDataTypeCollection(values);
            }

            return new ConfigurationVersionDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator ConfigurationVersionDataType[](ConfigurationVersionDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (ConfigurationVersionDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ConfigurationVersionDataTypeCollection clone = new ConfigurationVersionDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ConfigurationVersionDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PublishedDataSetDataType Class
    #if (!OPCUA_EXCLUDE_PublishedDataSetDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class PublishedDataSetDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public PublishedDataSetDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_name = null;
            m_dataSetFolder = new StringCollection();
            m_dataSetMetaData = new DataSetMetaDataType();
            m_extensionFields = new KeyValuePairCollection();
            m_dataSetSource = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 1)]
        public string Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetFolder", IsRequired = false, Order = 2)]
        public StringCollection DataSetFolder
        {
            get
            {
                return m_dataSetFolder;
            }

            set
            {
                m_dataSetFolder = value;

                if (value == null)
                {
                    m_dataSetFolder = new StringCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "DataSetMetaData", IsRequired = false, Order = 3)]
        public DataSetMetaDataType DataSetMetaData
        {
            get
            {
                return m_dataSetMetaData;
            }

            set
            {
                m_dataSetMetaData = value;

                if (value == null)
                {
                    m_dataSetMetaData = new DataSetMetaDataType();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "ExtensionFields", IsRequired = false, Order = 4)]
        public KeyValuePairCollection ExtensionFields
        {
            get
            {
                return m_extensionFields;
            }

            set
            {
                m_extensionFields = value;

                if (value == null)
                {
                    m_extensionFields = new KeyValuePairCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "DataSetSource", IsRequired = false, Order = 5)]
        public ExtensionObject DataSetSource
        {
            get { return m_dataSetSource;  }
            set { m_dataSetSource = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.PublishedDataSetDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.PublishedDataSetDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.PublishedDataSetDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.PublishedDataSetDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("Name", Name);
            encoder.WriteStringArray("DataSetFolder", DataSetFolder);
            encoder.WriteEncodeable("DataSetMetaData", DataSetMetaData, typeof(DataSetMetaDataType));
            encoder.WriteEncodeableArray("ExtensionFields", ExtensionFields.ToArray(), typeof(KeyValuePair));
            encoder.WriteExtensionObject("DataSetSource", DataSetSource);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Name = decoder.ReadString("Name");
            DataSetFolder = decoder.ReadStringArray("DataSetFolder");
            DataSetMetaData = (DataSetMetaDataType)decoder.ReadEncodeable("DataSetMetaData", typeof(DataSetMetaDataType));
            ExtensionFields = (KeyValuePairCollection)decoder.ReadEncodeableArray("ExtensionFields", typeof(KeyValuePair));
            DataSetSource = decoder.ReadExtensionObject("DataSetSource");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            PublishedDataSetDataType value = encodeable as PublishedDataSetDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_name, value.m_name)) return false;
            if (!Utils.IsEqual(m_dataSetFolder, value.m_dataSetFolder)) return false;
            if (!Utils.IsEqual(m_dataSetMetaData, value.m_dataSetMetaData)) return false;
            if (!Utils.IsEqual(m_extensionFields, value.m_extensionFields)) return false;
            if (!Utils.IsEqual(m_dataSetSource, value.m_dataSetSource)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (PublishedDataSetDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PublishedDataSetDataType clone = (PublishedDataSetDataType)base.MemberwiseClone();

            clone.m_name = (string)Utils.Clone(this.m_name);
            clone.m_dataSetFolder = (StringCollection)Utils.Clone(this.m_dataSetFolder);
            clone.m_dataSetMetaData = (DataSetMetaDataType)Utils.Clone(this.m_dataSetMetaData);
            clone.m_extensionFields = (KeyValuePairCollection)Utils.Clone(this.m_extensionFields);
            clone.m_dataSetSource = (ExtensionObject)Utils.Clone(this.m_dataSetSource);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_name;
        private StringCollection m_dataSetFolder;
        private DataSetMetaDataType m_dataSetMetaData;
        private KeyValuePairCollection m_extensionFields;
        private ExtensionObject m_dataSetSource;
        #endregion
    }

    #region PublishedDataSetDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPublishedDataSetDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PublishedDataSetDataType")]
    #if !NET_STANDARD
    public partial class PublishedDataSetDataTypeCollection : List<PublishedDataSetDataType>, ICloneable
    #else
    public partial class PublishedDataSetDataTypeCollection : List<PublishedDataSetDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public PublishedDataSetDataTypeCollection() {}

        /// <remarks />
        public PublishedDataSetDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public PublishedDataSetDataTypeCollection(IEnumerable<PublishedDataSetDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator PublishedDataSetDataTypeCollection(PublishedDataSetDataType[] values)
        {
            if (values != null)
            {
                return new PublishedDataSetDataTypeCollection(values);
            }

            return new PublishedDataSetDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator PublishedDataSetDataType[](PublishedDataSetDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (PublishedDataSetDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PublishedDataSetDataTypeCollection clone = new PublishedDataSetDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PublishedDataSetDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PublishedDataSetSourceDataType Class
    #if (!OPCUA_EXCLUDE_PublishedDataSetSourceDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class PublishedDataSetSourceDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public PublishedDataSetSourceDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.PublishedDataSetSourceDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.PublishedDataSetSourceDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.PublishedDataSetSourceDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.PublishedDataSetSourceDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            PublishedDataSetSourceDataType value = encodeable as PublishedDataSetSourceDataType;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (PublishedDataSetSourceDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PublishedDataSetSourceDataType clone = (PublishedDataSetSourceDataType)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region PublishedDataSetSourceDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPublishedDataSetSourceDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PublishedDataSetSourceDataType")]
    #if !NET_STANDARD
    public partial class PublishedDataSetSourceDataTypeCollection : List<PublishedDataSetSourceDataType>, ICloneable
    #else
    public partial class PublishedDataSetSourceDataTypeCollection : List<PublishedDataSetSourceDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public PublishedDataSetSourceDataTypeCollection() {}

        /// <remarks />
        public PublishedDataSetSourceDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public PublishedDataSetSourceDataTypeCollection(IEnumerable<PublishedDataSetSourceDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator PublishedDataSetSourceDataTypeCollection(PublishedDataSetSourceDataType[] values)
        {
            if (values != null)
            {
                return new PublishedDataSetSourceDataTypeCollection(values);
            }

            return new PublishedDataSetSourceDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator PublishedDataSetSourceDataType[](PublishedDataSetSourceDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (PublishedDataSetSourceDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PublishedDataSetSourceDataTypeCollection clone = new PublishedDataSetSourceDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PublishedDataSetSourceDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PublishedVariableDataType Class
    #if (!OPCUA_EXCLUDE_PublishedVariableDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class PublishedVariableDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public PublishedVariableDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_publishedVariable = null;
            m_attributeId = (uint)0;
            m_samplingIntervalHint = (double)0;
            m_deadbandType = (uint)0;
            m_deadbandValue = (double)0;
            m_indexRange = null;
            m_substituteValue = Variant.Null;
            m_metaDataProperties = new QualifiedNameCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "PublishedVariable", IsRequired = false, Order = 1)]
        public NodeId PublishedVariable
        {
            get { return m_publishedVariable;  }
            set { m_publishedVariable = value; }
        }

        /// <remarks />
        [DataMember(Name = "AttributeId", IsRequired = false, Order = 2)]
        public uint AttributeId
        {
            get { return m_attributeId;  }
            set { m_attributeId = value; }
        }

        /// <remarks />
        [DataMember(Name = "SamplingIntervalHint", IsRequired = false, Order = 3)]
        public double SamplingIntervalHint
        {
            get { return m_samplingIntervalHint;  }
            set { m_samplingIntervalHint = value; }
        }

        /// <remarks />
        [DataMember(Name = "DeadbandType", IsRequired = false, Order = 4)]
        public uint DeadbandType
        {
            get { return m_deadbandType;  }
            set { m_deadbandType = value; }
        }

        /// <remarks />
        [DataMember(Name = "DeadbandValue", IsRequired = false, Order = 5)]
        public double DeadbandValue
        {
            get { return m_deadbandValue;  }
            set { m_deadbandValue = value; }
        }

        /// <remarks />
        [DataMember(Name = "IndexRange", IsRequired = false, Order = 6)]
        public string IndexRange
        {
            get { return m_indexRange;  }
            set { m_indexRange = value; }
        }

        /// <remarks />
        [DataMember(Name = "SubstituteValue", IsRequired = false, Order = 7)]
        public Variant SubstituteValue
        {
            get { return m_substituteValue;  }
            set { m_substituteValue = value; }
        }

        /// <remarks />
        [DataMember(Name = "MetaDataProperties", IsRequired = false, Order = 8)]
        public QualifiedNameCollection MetaDataProperties
        {
            get
            {
                return m_metaDataProperties;
            }

            set
            {
                m_metaDataProperties = value;

                if (value == null)
                {
                    m_metaDataProperties = new QualifiedNameCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.PublishedVariableDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.PublishedVariableDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.PublishedVariableDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.PublishedVariableDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteNodeId("PublishedVariable", PublishedVariable);
            encoder.WriteUInt32("AttributeId", AttributeId);
            encoder.WriteDouble("SamplingIntervalHint", SamplingIntervalHint);
            encoder.WriteUInt32("DeadbandType", DeadbandType);
            encoder.WriteDouble("DeadbandValue", DeadbandValue);
            encoder.WriteString("IndexRange", IndexRange);
            encoder.WriteVariant("SubstituteValue", SubstituteValue);
            encoder.WriteQualifiedNameArray("MetaDataProperties", MetaDataProperties);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            PublishedVariable = decoder.ReadNodeId("PublishedVariable");
            AttributeId = decoder.ReadUInt32("AttributeId");
            SamplingIntervalHint = decoder.ReadDouble("SamplingIntervalHint");
            DeadbandType = decoder.ReadUInt32("DeadbandType");
            DeadbandValue = decoder.ReadDouble("DeadbandValue");
            IndexRange = decoder.ReadString("IndexRange");
            SubstituteValue = decoder.ReadVariant("SubstituteValue");
            MetaDataProperties = decoder.ReadQualifiedNameArray("MetaDataProperties");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            PublishedVariableDataType value = encodeable as PublishedVariableDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_publishedVariable, value.m_publishedVariable)) return false;
            if (!Utils.IsEqual(m_attributeId, value.m_attributeId)) return false;
            if (!Utils.IsEqual(m_samplingIntervalHint, value.m_samplingIntervalHint)) return false;
            if (!Utils.IsEqual(m_deadbandType, value.m_deadbandType)) return false;
            if (!Utils.IsEqual(m_deadbandValue, value.m_deadbandValue)) return false;
            if (!Utils.IsEqual(m_indexRange, value.m_indexRange)) return false;
            if (!Utils.IsEqual(m_substituteValue, value.m_substituteValue)) return false;
            if (!Utils.IsEqual(m_metaDataProperties, value.m_metaDataProperties)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (PublishedVariableDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PublishedVariableDataType clone = (PublishedVariableDataType)base.MemberwiseClone();

            clone.m_publishedVariable = (NodeId)Utils.Clone(this.m_publishedVariable);
            clone.m_attributeId = (uint)Utils.Clone(this.m_attributeId);
            clone.m_samplingIntervalHint = (double)Utils.Clone(this.m_samplingIntervalHint);
            clone.m_deadbandType = (uint)Utils.Clone(this.m_deadbandType);
            clone.m_deadbandValue = (double)Utils.Clone(this.m_deadbandValue);
            clone.m_indexRange = (string)Utils.Clone(this.m_indexRange);
            clone.m_substituteValue = (Variant)Utils.Clone(this.m_substituteValue);
            clone.m_metaDataProperties = (QualifiedNameCollection)Utils.Clone(this.m_metaDataProperties);

            return clone;
        }
        #endregion

        #region Private Fields
        private NodeId m_publishedVariable;
        private uint m_attributeId;
        private double m_samplingIntervalHint;
        private uint m_deadbandType;
        private double m_deadbandValue;
        private string m_indexRange;
        private Variant m_substituteValue;
        private QualifiedNameCollection m_metaDataProperties;
        #endregion
    }

    #region PublishedVariableDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPublishedVariableDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PublishedVariableDataType")]
    #if !NET_STANDARD
    public partial class PublishedVariableDataTypeCollection : List<PublishedVariableDataType>, ICloneable
    #else
    public partial class PublishedVariableDataTypeCollection : List<PublishedVariableDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public PublishedVariableDataTypeCollection() {}

        /// <remarks />
        public PublishedVariableDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public PublishedVariableDataTypeCollection(IEnumerable<PublishedVariableDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator PublishedVariableDataTypeCollection(PublishedVariableDataType[] values)
        {
            if (values != null)
            {
                return new PublishedVariableDataTypeCollection(values);
            }

            return new PublishedVariableDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator PublishedVariableDataType[](PublishedVariableDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (PublishedVariableDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PublishedVariableDataTypeCollection clone = new PublishedVariableDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PublishedVariableDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PublishedDataItemsDataType Class
    #if (!OPCUA_EXCLUDE_PublishedDataItemsDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class PublishedDataItemsDataType : PublishedDataSetSourceDataType
    {
        #region Constructors
        /// <remarks />
        public PublishedDataItemsDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_publishedData = new PublishedVariableDataTypeCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "PublishedData", IsRequired = false, Order = 1)]
        public PublishedVariableDataTypeCollection PublishedData
        {
            get
            {
                return m_publishedData;
            }

            set
            {
                m_publishedData = value;

                if (value == null)
                {
                    m_publishedData = new PublishedVariableDataTypeCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.PublishedDataItemsDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.PublishedDataItemsDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.PublishedDataItemsDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.PublishedDataItemsDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeableArray("PublishedData", PublishedData.ToArray(), typeof(PublishedVariableDataType));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            PublishedData = (PublishedVariableDataTypeCollection)decoder.ReadEncodeableArray("PublishedData", typeof(PublishedVariableDataType));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            PublishedDataItemsDataType value = encodeable as PublishedDataItemsDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_publishedData, value.m_publishedData)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (PublishedDataItemsDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PublishedDataItemsDataType clone = (PublishedDataItemsDataType)base.MemberwiseClone();

            clone.m_publishedData = (PublishedVariableDataTypeCollection)Utils.Clone(this.m_publishedData);

            return clone;
        }
        #endregion

        #region Private Fields
        private PublishedVariableDataTypeCollection m_publishedData;
        #endregion
    }

    #region PublishedDataItemsDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPublishedDataItemsDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PublishedDataItemsDataType")]
    #if !NET_STANDARD
    public partial class PublishedDataItemsDataTypeCollection : List<PublishedDataItemsDataType>, ICloneable
    #else
    public partial class PublishedDataItemsDataTypeCollection : List<PublishedDataItemsDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public PublishedDataItemsDataTypeCollection() {}

        /// <remarks />
        public PublishedDataItemsDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public PublishedDataItemsDataTypeCollection(IEnumerable<PublishedDataItemsDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator PublishedDataItemsDataTypeCollection(PublishedDataItemsDataType[] values)
        {
            if (values != null)
            {
                return new PublishedDataItemsDataTypeCollection(values);
            }

            return new PublishedDataItemsDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator PublishedDataItemsDataType[](PublishedDataItemsDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (PublishedDataItemsDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PublishedDataItemsDataTypeCollection clone = new PublishedDataItemsDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PublishedDataItemsDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PublishedEventsDataType Class
    #if (!OPCUA_EXCLUDE_PublishedEventsDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class PublishedEventsDataType : PublishedDataSetSourceDataType
    {
        #region Constructors
        /// <remarks />
        public PublishedEventsDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_eventNotifier = null;
            m_selectedFields = new SimpleAttributeOperandCollection();
            m_filter = new ContentFilter();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "EventNotifier", IsRequired = false, Order = 1)]
        public NodeId EventNotifier
        {
            get { return m_eventNotifier;  }
            set { m_eventNotifier = value; }
        }

        /// <remarks />
        [DataMember(Name = "SelectedFields", IsRequired = false, Order = 2)]
        public SimpleAttributeOperandCollection SelectedFields
        {
            get
            {
                return m_selectedFields;
            }

            set
            {
                m_selectedFields = value;

                if (value == null)
                {
                    m_selectedFields = new SimpleAttributeOperandCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "Filter", IsRequired = false, Order = 3)]
        public ContentFilter Filter
        {
            get
            {
                return m_filter;
            }

            set
            {
                m_filter = value;

                if (value == null)
                {
                    m_filter = new ContentFilter();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.PublishedEventsDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.PublishedEventsDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.PublishedEventsDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.PublishedEventsDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteNodeId("EventNotifier", EventNotifier);
            encoder.WriteEncodeableArray("SelectedFields", SelectedFields.ToArray(), typeof(SimpleAttributeOperand));
            encoder.WriteEncodeable("Filter", Filter, typeof(ContentFilter));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            EventNotifier = decoder.ReadNodeId("EventNotifier");
            SelectedFields = (SimpleAttributeOperandCollection)decoder.ReadEncodeableArray("SelectedFields", typeof(SimpleAttributeOperand));
            Filter = (ContentFilter)decoder.ReadEncodeable("Filter", typeof(ContentFilter));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            PublishedEventsDataType value = encodeable as PublishedEventsDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_eventNotifier, value.m_eventNotifier)) return false;
            if (!Utils.IsEqual(m_selectedFields, value.m_selectedFields)) return false;
            if (!Utils.IsEqual(m_filter, value.m_filter)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (PublishedEventsDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PublishedEventsDataType clone = (PublishedEventsDataType)base.MemberwiseClone();

            clone.m_eventNotifier = (NodeId)Utils.Clone(this.m_eventNotifier);
            clone.m_selectedFields = (SimpleAttributeOperandCollection)Utils.Clone(this.m_selectedFields);
            clone.m_filter = (ContentFilter)Utils.Clone(this.m_filter);

            return clone;
        }
        #endregion

        #region Private Fields
        private NodeId m_eventNotifier;
        private SimpleAttributeOperandCollection m_selectedFields;
        private ContentFilter m_filter;
        #endregion
    }

    #region PublishedEventsDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPublishedEventsDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PublishedEventsDataType")]
    #if !NET_STANDARD
    public partial class PublishedEventsDataTypeCollection : List<PublishedEventsDataType>, ICloneable
    #else
    public partial class PublishedEventsDataTypeCollection : List<PublishedEventsDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public PublishedEventsDataTypeCollection() {}

        /// <remarks />
        public PublishedEventsDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public PublishedEventsDataTypeCollection(IEnumerable<PublishedEventsDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator PublishedEventsDataTypeCollection(PublishedEventsDataType[] values)
        {
            if (values != null)
            {
                return new PublishedEventsDataTypeCollection(values);
            }

            return new PublishedEventsDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator PublishedEventsDataType[](PublishedEventsDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (PublishedEventsDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PublishedEventsDataTypeCollection clone = new PublishedEventsDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PublishedEventsDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PublishedDataSetCustomSourceDataType Class
    #if (!OPCUA_EXCLUDE_PublishedDataSetCustomSourceDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class PublishedDataSetCustomSourceDataType : PublishedDataSetSourceDataType
    {
        #region Constructors
        /// <remarks />
        public PublishedDataSetCustomSourceDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_cyclicDataSet = true;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "CyclicDataSet", IsRequired = false, Order = 1)]
        public bool CyclicDataSet
        {
            get { return m_cyclicDataSet;  }
            set { m_cyclicDataSet = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.PublishedDataSetCustomSourceDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.PublishedDataSetCustomSourceDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.PublishedDataSetCustomSourceDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.PublishedDataSetCustomSourceDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteBoolean("CyclicDataSet", CyclicDataSet);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            CyclicDataSet = decoder.ReadBoolean("CyclicDataSet");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            PublishedDataSetCustomSourceDataType value = encodeable as PublishedDataSetCustomSourceDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_cyclicDataSet, value.m_cyclicDataSet)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (PublishedDataSetCustomSourceDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PublishedDataSetCustomSourceDataType clone = (PublishedDataSetCustomSourceDataType)base.MemberwiseClone();

            clone.m_cyclicDataSet = (bool)Utils.Clone(this.m_cyclicDataSet);

            return clone;
        }
        #endregion

        #region Private Fields
        private bool m_cyclicDataSet;
        #endregion
    }

    #region PublishedDataSetCustomSourceDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPublishedDataSetCustomSourceDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PublishedDataSetCustomSourceDataType")]
    #if !NET_STANDARD
    public partial class PublishedDataSetCustomSourceDataTypeCollection : List<PublishedDataSetCustomSourceDataType>, ICloneable
    #else
    public partial class PublishedDataSetCustomSourceDataTypeCollection : List<PublishedDataSetCustomSourceDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public PublishedDataSetCustomSourceDataTypeCollection() {}

        /// <remarks />
        public PublishedDataSetCustomSourceDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public PublishedDataSetCustomSourceDataTypeCollection(IEnumerable<PublishedDataSetCustomSourceDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator PublishedDataSetCustomSourceDataTypeCollection(PublishedDataSetCustomSourceDataType[] values)
        {
            if (values != null)
            {
                return new PublishedDataSetCustomSourceDataTypeCollection(values);
            }

            return new PublishedDataSetCustomSourceDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator PublishedDataSetCustomSourceDataType[](PublishedDataSetCustomSourceDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (PublishedDataSetCustomSourceDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PublishedDataSetCustomSourceDataTypeCollection clone = new PublishedDataSetCustomSourceDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PublishedDataSetCustomSourceDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DataSetFieldContentMask Enumeration
    #if (!OPCUA_EXCLUDE_DataSetFieldContentMask)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum DataSetFieldContentMask : UInt32
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "StatusCode_1")]
        StatusCode = 1,

        /// <remarks />
        [EnumMember(Value = "SourceTimestamp_2")]
        SourceTimestamp = 2,

        /// <remarks />
        [EnumMember(Value = "ServerTimestamp_4")]
        ServerTimestamp = 4,

        /// <remarks />
        [EnumMember(Value = "SourcePicoSeconds_8")]
        SourcePicoSeconds = 8,

        /// <remarks />
        [EnumMember(Value = "ServerPicoSeconds_16")]
        ServerPicoSeconds = 16,

        /// <remarks />
        [EnumMember(Value = "RawData_32")]
        RawData = 32,
    }

    #region DataSetFieldContentMaskCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDataSetFieldContentMask", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DataSetFieldContentMask")]
    #if !NET_STANDARD
    public partial class DataSetFieldContentMaskCollection : List<DataSetFieldContentMask>, ICloneable
    #else
    public partial class DataSetFieldContentMaskCollection : List<DataSetFieldContentMask>
    #endif
    {
        #region Constructors
        /// <remarks />
        public DataSetFieldContentMaskCollection() {}

        /// <remarks />
        public DataSetFieldContentMaskCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public DataSetFieldContentMaskCollection(IEnumerable<DataSetFieldContentMask> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator DataSetFieldContentMaskCollection(DataSetFieldContentMask[] values)
        {
            if (values != null)
            {
                return new DataSetFieldContentMaskCollection(values);
            }

            return new DataSetFieldContentMaskCollection();
        }

        /// <remarks />
        public static explicit operator DataSetFieldContentMask[](DataSetFieldContentMaskCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (DataSetFieldContentMaskCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetFieldContentMaskCollection clone = new DataSetFieldContentMaskCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DataSetFieldContentMask)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DataSetWriterDataType Class
    #if (!OPCUA_EXCLUDE_DataSetWriterDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DataSetWriterDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public DataSetWriterDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_name = null;
            m_enabled = true;
            m_dataSetWriterId = (ushort)0;
            m_dataSetFieldContentMask = 0;
            m_keyFrameCount = (uint)0;
            m_dataSetName = null;
            m_dataSetWriterProperties = new KeyValuePairCollection();
            m_transportSettings = null;
            m_messageSettings = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 1)]
        public string Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }

        /// <remarks />
        [DataMember(Name = "Enabled", IsRequired = false, Order = 2)]
        public bool Enabled
        {
            get { return m_enabled;  }
            set { m_enabled = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetWriterId", IsRequired = false, Order = 3)]
        public ushort DataSetWriterId
        {
            get { return m_dataSetWriterId;  }
            set { m_dataSetWriterId = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetFieldContentMask", IsRequired = false, Order = 4)]
        public uint DataSetFieldContentMask
        {
            get { return m_dataSetFieldContentMask;  }
            set { m_dataSetFieldContentMask = value; }
        }

        /// <remarks />
        [DataMember(Name = "KeyFrameCount", IsRequired = false, Order = 5)]
        public uint KeyFrameCount
        {
            get { return m_keyFrameCount;  }
            set { m_keyFrameCount = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetName", IsRequired = false, Order = 6)]
        public string DataSetName
        {
            get { return m_dataSetName;  }
            set { m_dataSetName = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetWriterProperties", IsRequired = false, Order = 7)]
        public KeyValuePairCollection DataSetWriterProperties
        {
            get
            {
                return m_dataSetWriterProperties;
            }

            set
            {
                m_dataSetWriterProperties = value;

                if (value == null)
                {
                    m_dataSetWriterProperties = new KeyValuePairCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "TransportSettings", IsRequired = false, Order = 8)]
        public ExtensionObject TransportSettings
        {
            get { return m_transportSettings;  }
            set { m_transportSettings = value; }
        }

        /// <remarks />
        [DataMember(Name = "MessageSettings", IsRequired = false, Order = 9)]
        public ExtensionObject MessageSettings
        {
            get { return m_messageSettings;  }
            set { m_messageSettings = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.DataSetWriterDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.DataSetWriterDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.DataSetWriterDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.DataSetWriterDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("Name", Name);
            encoder.WriteBoolean("Enabled", Enabled);
            encoder.WriteUInt16("DataSetWriterId", DataSetWriterId);
            encoder.WriteUInt32("DataSetFieldContentMask", DataSetFieldContentMask);
            encoder.WriteUInt32("KeyFrameCount", KeyFrameCount);
            encoder.WriteString("DataSetName", DataSetName);
            encoder.WriteEncodeableArray("DataSetWriterProperties", DataSetWriterProperties.ToArray(), typeof(KeyValuePair));
            encoder.WriteExtensionObject("TransportSettings", TransportSettings);
            encoder.WriteExtensionObject("MessageSettings", MessageSettings);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Name = decoder.ReadString("Name");
            Enabled = decoder.ReadBoolean("Enabled");
            DataSetWriterId = decoder.ReadUInt16("DataSetWriterId");
            DataSetFieldContentMask = decoder.ReadUInt32("DataSetFieldContentMask");
            KeyFrameCount = decoder.ReadUInt32("KeyFrameCount");
            DataSetName = decoder.ReadString("DataSetName");
            DataSetWriterProperties = (KeyValuePairCollection)decoder.ReadEncodeableArray("DataSetWriterProperties", typeof(KeyValuePair));
            TransportSettings = decoder.ReadExtensionObject("TransportSettings");
            MessageSettings = decoder.ReadExtensionObject("MessageSettings");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DataSetWriterDataType value = encodeable as DataSetWriterDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_name, value.m_name)) return false;
            if (!Utils.IsEqual(m_enabled, value.m_enabled)) return false;
            if (!Utils.IsEqual(m_dataSetWriterId, value.m_dataSetWriterId)) return false;
            if (!Utils.IsEqual(m_dataSetFieldContentMask, value.m_dataSetFieldContentMask)) return false;
            if (!Utils.IsEqual(m_keyFrameCount, value.m_keyFrameCount)) return false;
            if (!Utils.IsEqual(m_dataSetName, value.m_dataSetName)) return false;
            if (!Utils.IsEqual(m_dataSetWriterProperties, value.m_dataSetWriterProperties)) return false;
            if (!Utils.IsEqual(m_transportSettings, value.m_transportSettings)) return false;
            if (!Utils.IsEqual(m_messageSettings, value.m_messageSettings)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (DataSetWriterDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetWriterDataType clone = (DataSetWriterDataType)base.MemberwiseClone();

            clone.m_name = (string)Utils.Clone(this.m_name);
            clone.m_enabled = (bool)Utils.Clone(this.m_enabled);
            clone.m_dataSetWriterId = (ushort)Utils.Clone(this.m_dataSetWriterId);
            clone.m_dataSetFieldContentMask = (uint)Utils.Clone(this.m_dataSetFieldContentMask);
            clone.m_keyFrameCount = (uint)Utils.Clone(this.m_keyFrameCount);
            clone.m_dataSetName = (string)Utils.Clone(this.m_dataSetName);
            clone.m_dataSetWriterProperties = (KeyValuePairCollection)Utils.Clone(this.m_dataSetWriterProperties);
            clone.m_transportSettings = (ExtensionObject)Utils.Clone(this.m_transportSettings);
            clone.m_messageSettings = (ExtensionObject)Utils.Clone(this.m_messageSettings);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_name;
        private bool m_enabled;
        private ushort m_dataSetWriterId;
        private uint m_dataSetFieldContentMask;
        private uint m_keyFrameCount;
        private string m_dataSetName;
        private KeyValuePairCollection m_dataSetWriterProperties;
        private ExtensionObject m_transportSettings;
        private ExtensionObject m_messageSettings;
        #endregion
    }

    #region DataSetWriterDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDataSetWriterDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DataSetWriterDataType")]
    #if !NET_STANDARD
    public partial class DataSetWriterDataTypeCollection : List<DataSetWriterDataType>, ICloneable
    #else
    public partial class DataSetWriterDataTypeCollection : List<DataSetWriterDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public DataSetWriterDataTypeCollection() {}

        /// <remarks />
        public DataSetWriterDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public DataSetWriterDataTypeCollection(IEnumerable<DataSetWriterDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator DataSetWriterDataTypeCollection(DataSetWriterDataType[] values)
        {
            if (values != null)
            {
                return new DataSetWriterDataTypeCollection(values);
            }

            return new DataSetWriterDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator DataSetWriterDataType[](DataSetWriterDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (DataSetWriterDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetWriterDataTypeCollection clone = new DataSetWriterDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DataSetWriterDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DataSetWriterTransportDataType Class
    #if (!OPCUA_EXCLUDE_DataSetWriterTransportDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DataSetWriterTransportDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public DataSetWriterTransportDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.DataSetWriterTransportDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.DataSetWriterTransportDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.DataSetWriterTransportDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.DataSetWriterTransportDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DataSetWriterTransportDataType value = encodeable as DataSetWriterTransportDataType;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (DataSetWriterTransportDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetWriterTransportDataType clone = (DataSetWriterTransportDataType)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region DataSetWriterTransportDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDataSetWriterTransportDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DataSetWriterTransportDataType")]
    #if !NET_STANDARD
    public partial class DataSetWriterTransportDataTypeCollection : List<DataSetWriterTransportDataType>, ICloneable
    #else
    public partial class DataSetWriterTransportDataTypeCollection : List<DataSetWriterTransportDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public DataSetWriterTransportDataTypeCollection() {}

        /// <remarks />
        public DataSetWriterTransportDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public DataSetWriterTransportDataTypeCollection(IEnumerable<DataSetWriterTransportDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator DataSetWriterTransportDataTypeCollection(DataSetWriterTransportDataType[] values)
        {
            if (values != null)
            {
                return new DataSetWriterTransportDataTypeCollection(values);
            }

            return new DataSetWriterTransportDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator DataSetWriterTransportDataType[](DataSetWriterTransportDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (DataSetWriterTransportDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetWriterTransportDataTypeCollection clone = new DataSetWriterTransportDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DataSetWriterTransportDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DataSetWriterMessageDataType Class
    #if (!OPCUA_EXCLUDE_DataSetWriterMessageDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DataSetWriterMessageDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public DataSetWriterMessageDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.DataSetWriterMessageDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.DataSetWriterMessageDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.DataSetWriterMessageDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.DataSetWriterMessageDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DataSetWriterMessageDataType value = encodeable as DataSetWriterMessageDataType;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (DataSetWriterMessageDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetWriterMessageDataType clone = (DataSetWriterMessageDataType)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region DataSetWriterMessageDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDataSetWriterMessageDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DataSetWriterMessageDataType")]
    #if !NET_STANDARD
    public partial class DataSetWriterMessageDataTypeCollection : List<DataSetWriterMessageDataType>, ICloneable
    #else
    public partial class DataSetWriterMessageDataTypeCollection : List<DataSetWriterMessageDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public DataSetWriterMessageDataTypeCollection() {}

        /// <remarks />
        public DataSetWriterMessageDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public DataSetWriterMessageDataTypeCollection(IEnumerable<DataSetWriterMessageDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator DataSetWriterMessageDataTypeCollection(DataSetWriterMessageDataType[] values)
        {
            if (values != null)
            {
                return new DataSetWriterMessageDataTypeCollection(values);
            }

            return new DataSetWriterMessageDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator DataSetWriterMessageDataType[](DataSetWriterMessageDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (DataSetWriterMessageDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetWriterMessageDataTypeCollection clone = new DataSetWriterMessageDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DataSetWriterMessageDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PubSubGroupDataType Class
    #if (!OPCUA_EXCLUDE_PubSubGroupDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class PubSubGroupDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public PubSubGroupDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_name = null;
            m_enabled = true;
            m_securityMode = MessageSecurityMode.Invalid;
            m_securityGroupId = null;
            m_securityKeyServices = new EndpointDescriptionCollection();
            m_maxNetworkMessageSize = (uint)0;
            m_groupProperties = new KeyValuePairCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 1)]
        public string Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }

        /// <remarks />
        [DataMember(Name = "Enabled", IsRequired = false, Order = 2)]
        public bool Enabled
        {
            get { return m_enabled;  }
            set { m_enabled = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityMode", IsRequired = false, Order = 3)]
        public MessageSecurityMode SecurityMode
        {
            get { return m_securityMode;  }
            set { m_securityMode = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityGroupId", IsRequired = false, Order = 4)]
        public string SecurityGroupId
        {
            get { return m_securityGroupId;  }
            set { m_securityGroupId = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityKeyServices", IsRequired = false, Order = 5)]
        public EndpointDescriptionCollection SecurityKeyServices
        {
            get
            {
                return m_securityKeyServices;
            }

            set
            {
                m_securityKeyServices = value;

                if (value == null)
                {
                    m_securityKeyServices = new EndpointDescriptionCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "MaxNetworkMessageSize", IsRequired = false, Order = 6)]
        public uint MaxNetworkMessageSize
        {
            get { return m_maxNetworkMessageSize;  }
            set { m_maxNetworkMessageSize = value; }
        }

        /// <remarks />
        [DataMember(Name = "GroupProperties", IsRequired = false, Order = 7)]
        public KeyValuePairCollection GroupProperties
        {
            get
            {
                return m_groupProperties;
            }

            set
            {
                m_groupProperties = value;

                if (value == null)
                {
                    m_groupProperties = new KeyValuePairCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.PubSubGroupDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.PubSubGroupDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.PubSubGroupDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.PubSubGroupDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("Name", Name);
            encoder.WriteBoolean("Enabled", Enabled);
            encoder.WriteEnumerated("SecurityMode", SecurityMode);
            encoder.WriteString("SecurityGroupId", SecurityGroupId);
            encoder.WriteEncodeableArray("SecurityKeyServices", SecurityKeyServices.ToArray(), typeof(EndpointDescription));
            encoder.WriteUInt32("MaxNetworkMessageSize", MaxNetworkMessageSize);
            encoder.WriteEncodeableArray("GroupProperties", GroupProperties.ToArray(), typeof(KeyValuePair));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Name = decoder.ReadString("Name");
            Enabled = decoder.ReadBoolean("Enabled");
            SecurityMode = (MessageSecurityMode)decoder.ReadEnumerated("SecurityMode", typeof(MessageSecurityMode));
            SecurityGroupId = decoder.ReadString("SecurityGroupId");
            SecurityKeyServices = (EndpointDescriptionCollection)decoder.ReadEncodeableArray("SecurityKeyServices", typeof(EndpointDescription));
            MaxNetworkMessageSize = decoder.ReadUInt32("MaxNetworkMessageSize");
            GroupProperties = (KeyValuePairCollection)decoder.ReadEncodeableArray("GroupProperties", typeof(KeyValuePair));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            PubSubGroupDataType value = encodeable as PubSubGroupDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_name, value.m_name)) return false;
            if (!Utils.IsEqual(m_enabled, value.m_enabled)) return false;
            if (!Utils.IsEqual(m_securityMode, value.m_securityMode)) return false;
            if (!Utils.IsEqual(m_securityGroupId, value.m_securityGroupId)) return false;
            if (!Utils.IsEqual(m_securityKeyServices, value.m_securityKeyServices)) return false;
            if (!Utils.IsEqual(m_maxNetworkMessageSize, value.m_maxNetworkMessageSize)) return false;
            if (!Utils.IsEqual(m_groupProperties, value.m_groupProperties)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (PubSubGroupDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PubSubGroupDataType clone = (PubSubGroupDataType)base.MemberwiseClone();

            clone.m_name = (string)Utils.Clone(this.m_name);
            clone.m_enabled = (bool)Utils.Clone(this.m_enabled);
            clone.m_securityMode = (MessageSecurityMode)Utils.Clone(this.m_securityMode);
            clone.m_securityGroupId = (string)Utils.Clone(this.m_securityGroupId);
            clone.m_securityKeyServices = (EndpointDescriptionCollection)Utils.Clone(this.m_securityKeyServices);
            clone.m_maxNetworkMessageSize = (uint)Utils.Clone(this.m_maxNetworkMessageSize);
            clone.m_groupProperties = (KeyValuePairCollection)Utils.Clone(this.m_groupProperties);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_name;
        private bool m_enabled;
        private MessageSecurityMode m_securityMode;
        private string m_securityGroupId;
        private EndpointDescriptionCollection m_securityKeyServices;
        private uint m_maxNetworkMessageSize;
        private KeyValuePairCollection m_groupProperties;
        #endregion
    }

    #region PubSubGroupDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPubSubGroupDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PubSubGroupDataType")]
    #if !NET_STANDARD
    public partial class PubSubGroupDataTypeCollection : List<PubSubGroupDataType>, ICloneable
    #else
    public partial class PubSubGroupDataTypeCollection : List<PubSubGroupDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public PubSubGroupDataTypeCollection() {}

        /// <remarks />
        public PubSubGroupDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public PubSubGroupDataTypeCollection(IEnumerable<PubSubGroupDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator PubSubGroupDataTypeCollection(PubSubGroupDataType[] values)
        {
            if (values != null)
            {
                return new PubSubGroupDataTypeCollection(values);
            }

            return new PubSubGroupDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator PubSubGroupDataType[](PubSubGroupDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (PubSubGroupDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PubSubGroupDataTypeCollection clone = new PubSubGroupDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PubSubGroupDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region WriterGroupDataType Class
    #if (!OPCUA_EXCLUDE_WriterGroupDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class WriterGroupDataType : PubSubGroupDataType
    {
        #region Constructors
        /// <remarks />
        public WriterGroupDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_writerGroupId = (ushort)0;
            m_publishingInterval = (double)0;
            m_keepAliveTime = (double)0;
            m_priority = (byte)0;
            m_localeIds = new StringCollection();
            m_headerLayoutUri = null;
            m_transportSettings = null;
            m_messageSettings = null;
            m_dataSetWriters = new DataSetWriterDataTypeCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "WriterGroupId", IsRequired = false, Order = 1)]
        public ushort WriterGroupId
        {
            get { return m_writerGroupId;  }
            set { m_writerGroupId = value; }
        }

        /// <remarks />
        [DataMember(Name = "PublishingInterval", IsRequired = false, Order = 2)]
        public double PublishingInterval
        {
            get { return m_publishingInterval;  }
            set { m_publishingInterval = value; }
        }

        /// <remarks />
        [DataMember(Name = "KeepAliveTime", IsRequired = false, Order = 3)]
        public double KeepAliveTime
        {
            get { return m_keepAliveTime;  }
            set { m_keepAliveTime = value; }
        }

        /// <remarks />
        [DataMember(Name = "Priority", IsRequired = false, Order = 4)]
        public byte Priority
        {
            get { return m_priority;  }
            set { m_priority = value; }
        }

        /// <remarks />
        [DataMember(Name = "LocaleIds", IsRequired = false, Order = 5)]
        public StringCollection LocaleIds
        {
            get
            {
                return m_localeIds;
            }

            set
            {
                m_localeIds = value;

                if (value == null)
                {
                    m_localeIds = new StringCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "HeaderLayoutUri", IsRequired = false, Order = 6)]
        public string HeaderLayoutUri
        {
            get { return m_headerLayoutUri;  }
            set { m_headerLayoutUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "TransportSettings", IsRequired = false, Order = 7)]
        public ExtensionObject TransportSettings
        {
            get { return m_transportSettings;  }
            set { m_transportSettings = value; }
        }

        /// <remarks />
        [DataMember(Name = "MessageSettings", IsRequired = false, Order = 8)]
        public ExtensionObject MessageSettings
        {
            get { return m_messageSettings;  }
            set { m_messageSettings = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetWriters", IsRequired = false, Order = 9)]
        public DataSetWriterDataTypeCollection DataSetWriters
        {
            get
            {
                return m_dataSetWriters;
            }

            set
            {
                m_dataSetWriters = value;

                if (value == null)
                {
                    m_dataSetWriters = new DataSetWriterDataTypeCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.WriterGroupDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.WriterGroupDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.WriterGroupDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.WriterGroupDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt16("WriterGroupId", WriterGroupId);
            encoder.WriteDouble("PublishingInterval", PublishingInterval);
            encoder.WriteDouble("KeepAliveTime", KeepAliveTime);
            encoder.WriteByte("Priority", Priority);
            encoder.WriteStringArray("LocaleIds", LocaleIds);
            encoder.WriteString("HeaderLayoutUri", HeaderLayoutUri);
            encoder.WriteExtensionObject("TransportSettings", TransportSettings);
            encoder.WriteExtensionObject("MessageSettings", MessageSettings);
            encoder.WriteEncodeableArray("DataSetWriters", DataSetWriters.ToArray(), typeof(DataSetWriterDataType));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            WriterGroupId = decoder.ReadUInt16("WriterGroupId");
            PublishingInterval = decoder.ReadDouble("PublishingInterval");
            KeepAliveTime = decoder.ReadDouble("KeepAliveTime");
            Priority = decoder.ReadByte("Priority");
            LocaleIds = decoder.ReadStringArray("LocaleIds");
            HeaderLayoutUri = decoder.ReadString("HeaderLayoutUri");
            TransportSettings = decoder.ReadExtensionObject("TransportSettings");
            MessageSettings = decoder.ReadExtensionObject("MessageSettings");
            DataSetWriters = (DataSetWriterDataTypeCollection)decoder.ReadEncodeableArray("DataSetWriters", typeof(DataSetWriterDataType));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            WriterGroupDataType value = encodeable as WriterGroupDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_writerGroupId, value.m_writerGroupId)) return false;
            if (!Utils.IsEqual(m_publishingInterval, value.m_publishingInterval)) return false;
            if (!Utils.IsEqual(m_keepAliveTime, value.m_keepAliveTime)) return false;
            if (!Utils.IsEqual(m_priority, value.m_priority)) return false;
            if (!Utils.IsEqual(m_localeIds, value.m_localeIds)) return false;
            if (!Utils.IsEqual(m_headerLayoutUri, value.m_headerLayoutUri)) return false;
            if (!Utils.IsEqual(m_transportSettings, value.m_transportSettings)) return false;
            if (!Utils.IsEqual(m_messageSettings, value.m_messageSettings)) return false;
            if (!Utils.IsEqual(m_dataSetWriters, value.m_dataSetWriters)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (WriterGroupDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            WriterGroupDataType clone = (WriterGroupDataType)base.MemberwiseClone();

            clone.m_writerGroupId = (ushort)Utils.Clone(this.m_writerGroupId);
            clone.m_publishingInterval = (double)Utils.Clone(this.m_publishingInterval);
            clone.m_keepAliveTime = (double)Utils.Clone(this.m_keepAliveTime);
            clone.m_priority = (byte)Utils.Clone(this.m_priority);
            clone.m_localeIds = (StringCollection)Utils.Clone(this.m_localeIds);
            clone.m_headerLayoutUri = (string)Utils.Clone(this.m_headerLayoutUri);
            clone.m_transportSettings = (ExtensionObject)Utils.Clone(this.m_transportSettings);
            clone.m_messageSettings = (ExtensionObject)Utils.Clone(this.m_messageSettings);
            clone.m_dataSetWriters = (DataSetWriterDataTypeCollection)Utils.Clone(this.m_dataSetWriters);

            return clone;
        }
        #endregion

        #region Private Fields
        private ushort m_writerGroupId;
        private double m_publishingInterval;
        private double m_keepAliveTime;
        private byte m_priority;
        private StringCollection m_localeIds;
        private string m_headerLayoutUri;
        private ExtensionObject m_transportSettings;
        private ExtensionObject m_messageSettings;
        private DataSetWriterDataTypeCollection m_dataSetWriters;
        #endregion
    }

    #region WriterGroupDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfWriterGroupDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "WriterGroupDataType")]
    #if !NET_STANDARD
    public partial class WriterGroupDataTypeCollection : List<WriterGroupDataType>, ICloneable
    #else
    public partial class WriterGroupDataTypeCollection : List<WriterGroupDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public WriterGroupDataTypeCollection() {}

        /// <remarks />
        public WriterGroupDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public WriterGroupDataTypeCollection(IEnumerable<WriterGroupDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator WriterGroupDataTypeCollection(WriterGroupDataType[] values)
        {
            if (values != null)
            {
                return new WriterGroupDataTypeCollection(values);
            }

            return new WriterGroupDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator WriterGroupDataType[](WriterGroupDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (WriterGroupDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            WriterGroupDataTypeCollection clone = new WriterGroupDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((WriterGroupDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region WriterGroupTransportDataType Class
    #if (!OPCUA_EXCLUDE_WriterGroupTransportDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class WriterGroupTransportDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public WriterGroupTransportDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.WriterGroupTransportDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.WriterGroupTransportDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.WriterGroupTransportDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.WriterGroupTransportDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            WriterGroupTransportDataType value = encodeable as WriterGroupTransportDataType;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (WriterGroupTransportDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            WriterGroupTransportDataType clone = (WriterGroupTransportDataType)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region WriterGroupTransportDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfWriterGroupTransportDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "WriterGroupTransportDataType")]
    #if !NET_STANDARD
    public partial class WriterGroupTransportDataTypeCollection : List<WriterGroupTransportDataType>, ICloneable
    #else
    public partial class WriterGroupTransportDataTypeCollection : List<WriterGroupTransportDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public WriterGroupTransportDataTypeCollection() {}

        /// <remarks />
        public WriterGroupTransportDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public WriterGroupTransportDataTypeCollection(IEnumerable<WriterGroupTransportDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator WriterGroupTransportDataTypeCollection(WriterGroupTransportDataType[] values)
        {
            if (values != null)
            {
                return new WriterGroupTransportDataTypeCollection(values);
            }

            return new WriterGroupTransportDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator WriterGroupTransportDataType[](WriterGroupTransportDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (WriterGroupTransportDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            WriterGroupTransportDataTypeCollection clone = new WriterGroupTransportDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((WriterGroupTransportDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region WriterGroupMessageDataType Class
    #if (!OPCUA_EXCLUDE_WriterGroupMessageDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class WriterGroupMessageDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public WriterGroupMessageDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.WriterGroupMessageDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.WriterGroupMessageDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.WriterGroupMessageDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.WriterGroupMessageDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            WriterGroupMessageDataType value = encodeable as WriterGroupMessageDataType;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (WriterGroupMessageDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            WriterGroupMessageDataType clone = (WriterGroupMessageDataType)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region WriterGroupMessageDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfWriterGroupMessageDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "WriterGroupMessageDataType")]
    #if !NET_STANDARD
    public partial class WriterGroupMessageDataTypeCollection : List<WriterGroupMessageDataType>, ICloneable
    #else
    public partial class WriterGroupMessageDataTypeCollection : List<WriterGroupMessageDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public WriterGroupMessageDataTypeCollection() {}

        /// <remarks />
        public WriterGroupMessageDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public WriterGroupMessageDataTypeCollection(IEnumerable<WriterGroupMessageDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator WriterGroupMessageDataTypeCollection(WriterGroupMessageDataType[] values)
        {
            if (values != null)
            {
                return new WriterGroupMessageDataTypeCollection(values);
            }

            return new WriterGroupMessageDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator WriterGroupMessageDataType[](WriterGroupMessageDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (WriterGroupMessageDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            WriterGroupMessageDataTypeCollection clone = new WriterGroupMessageDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((WriterGroupMessageDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PubSubConnectionDataType Class
    #if (!OPCUA_EXCLUDE_PubSubConnectionDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class PubSubConnectionDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public PubSubConnectionDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_name = null;
            m_enabled = true;
            m_publisherId = Variant.Null;
            m_transportProfileUri = null;
            m_address = null;
            m_connectionProperties = new KeyValuePairCollection();
            m_transportSettings = null;
            m_writerGroups = new WriterGroupDataTypeCollection();
            m_readerGroups = new ReaderGroupDataTypeCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 1)]
        public string Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }

        /// <remarks />
        [DataMember(Name = "Enabled", IsRequired = false, Order = 2)]
        public bool Enabled
        {
            get { return m_enabled;  }
            set { m_enabled = value; }
        }

        /// <remarks />
        [DataMember(Name = "PublisherId", IsRequired = false, Order = 3)]
        public Variant PublisherId
        {
            get { return m_publisherId;  }
            set { m_publisherId = value; }
        }

        /// <remarks />
        [DataMember(Name = "TransportProfileUri", IsRequired = false, Order = 4)]
        public string TransportProfileUri
        {
            get { return m_transportProfileUri;  }
            set { m_transportProfileUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "Address", IsRequired = false, Order = 5)]
        public ExtensionObject Address
        {
            get { return m_address;  }
            set { m_address = value; }
        }

        /// <remarks />
        [DataMember(Name = "ConnectionProperties", IsRequired = false, Order = 6)]
        public KeyValuePairCollection ConnectionProperties
        {
            get
            {
                return m_connectionProperties;
            }

            set
            {
                m_connectionProperties = value;

                if (value == null)
                {
                    m_connectionProperties = new KeyValuePairCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "TransportSettings", IsRequired = false, Order = 7)]
        public ExtensionObject TransportSettings
        {
            get { return m_transportSettings;  }
            set { m_transportSettings = value; }
        }

        /// <remarks />
        [DataMember(Name = "WriterGroups", IsRequired = false, Order = 8)]
        public WriterGroupDataTypeCollection WriterGroups
        {
            get
            {
                return m_writerGroups;
            }

            set
            {
                m_writerGroups = value;

                if (value == null)
                {
                    m_writerGroups = new WriterGroupDataTypeCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "ReaderGroups", IsRequired = false, Order = 9)]
        public ReaderGroupDataTypeCollection ReaderGroups
        {
            get
            {
                return m_readerGroups;
            }

            set
            {
                m_readerGroups = value;

                if (value == null)
                {
                    m_readerGroups = new ReaderGroupDataTypeCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.PubSubConnectionDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.PubSubConnectionDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.PubSubConnectionDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.PubSubConnectionDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("Name", Name);
            encoder.WriteBoolean("Enabled", Enabled);
            encoder.WriteVariant("PublisherId", PublisherId);
            encoder.WriteString("TransportProfileUri", TransportProfileUri);
            encoder.WriteExtensionObject("Address", Address);
            encoder.WriteEncodeableArray("ConnectionProperties", ConnectionProperties.ToArray(), typeof(KeyValuePair));
            encoder.WriteExtensionObject("TransportSettings", TransportSettings);
            encoder.WriteEncodeableArray("WriterGroups", WriterGroups.ToArray(), typeof(WriterGroupDataType));
            encoder.WriteEncodeableArray("ReaderGroups", ReaderGroups.ToArray(), typeof(ReaderGroupDataType));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Name = decoder.ReadString("Name");
            Enabled = decoder.ReadBoolean("Enabled");
            PublisherId = decoder.ReadVariant("PublisherId");
            TransportProfileUri = decoder.ReadString("TransportProfileUri");
            Address = decoder.ReadExtensionObject("Address");
            ConnectionProperties = (KeyValuePairCollection)decoder.ReadEncodeableArray("ConnectionProperties", typeof(KeyValuePair));
            TransportSettings = decoder.ReadExtensionObject("TransportSettings");
            WriterGroups = (WriterGroupDataTypeCollection)decoder.ReadEncodeableArray("WriterGroups", typeof(WriterGroupDataType));
            ReaderGroups = (ReaderGroupDataTypeCollection)decoder.ReadEncodeableArray("ReaderGroups", typeof(ReaderGroupDataType));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            PubSubConnectionDataType value = encodeable as PubSubConnectionDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_name, value.m_name)) return false;
            if (!Utils.IsEqual(m_enabled, value.m_enabled)) return false;
            if (!Utils.IsEqual(m_publisherId, value.m_publisherId)) return false;
            if (!Utils.IsEqual(m_transportProfileUri, value.m_transportProfileUri)) return false;
            if (!Utils.IsEqual(m_address, value.m_address)) return false;
            if (!Utils.IsEqual(m_connectionProperties, value.m_connectionProperties)) return false;
            if (!Utils.IsEqual(m_transportSettings, value.m_transportSettings)) return false;
            if (!Utils.IsEqual(m_writerGroups, value.m_writerGroups)) return false;
            if (!Utils.IsEqual(m_readerGroups, value.m_readerGroups)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (PubSubConnectionDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PubSubConnectionDataType clone = (PubSubConnectionDataType)base.MemberwiseClone();

            clone.m_name = (string)Utils.Clone(this.m_name);
            clone.m_enabled = (bool)Utils.Clone(this.m_enabled);
            clone.m_publisherId = (Variant)Utils.Clone(this.m_publisherId);
            clone.m_transportProfileUri = (string)Utils.Clone(this.m_transportProfileUri);
            clone.m_address = (ExtensionObject)Utils.Clone(this.m_address);
            clone.m_connectionProperties = (KeyValuePairCollection)Utils.Clone(this.m_connectionProperties);
            clone.m_transportSettings = (ExtensionObject)Utils.Clone(this.m_transportSettings);
            clone.m_writerGroups = (WriterGroupDataTypeCollection)Utils.Clone(this.m_writerGroups);
            clone.m_readerGroups = (ReaderGroupDataTypeCollection)Utils.Clone(this.m_readerGroups);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_name;
        private bool m_enabled;
        private Variant m_publisherId;
        private string m_transportProfileUri;
        private ExtensionObject m_address;
        private KeyValuePairCollection m_connectionProperties;
        private ExtensionObject m_transportSettings;
        private WriterGroupDataTypeCollection m_writerGroups;
        private ReaderGroupDataTypeCollection m_readerGroups;
        #endregion
    }

    #region PubSubConnectionDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPubSubConnectionDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PubSubConnectionDataType")]
    #if !NET_STANDARD
    public partial class PubSubConnectionDataTypeCollection : List<PubSubConnectionDataType>, ICloneable
    #else
    public partial class PubSubConnectionDataTypeCollection : List<PubSubConnectionDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public PubSubConnectionDataTypeCollection() {}

        /// <remarks />
        public PubSubConnectionDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public PubSubConnectionDataTypeCollection(IEnumerable<PubSubConnectionDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator PubSubConnectionDataTypeCollection(PubSubConnectionDataType[] values)
        {
            if (values != null)
            {
                return new PubSubConnectionDataTypeCollection(values);
            }

            return new PubSubConnectionDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator PubSubConnectionDataType[](PubSubConnectionDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (PubSubConnectionDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PubSubConnectionDataTypeCollection clone = new PubSubConnectionDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PubSubConnectionDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region ConnectionTransportDataType Class
    #if (!OPCUA_EXCLUDE_ConnectionTransportDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ConnectionTransportDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public ConnectionTransportDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.ConnectionTransportDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.ConnectionTransportDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.ConnectionTransportDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.ConnectionTransportDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ConnectionTransportDataType value = encodeable as ConnectionTransportDataType;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (ConnectionTransportDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ConnectionTransportDataType clone = (ConnectionTransportDataType)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region ConnectionTransportDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfConnectionTransportDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ConnectionTransportDataType")]
    #if !NET_STANDARD
    public partial class ConnectionTransportDataTypeCollection : List<ConnectionTransportDataType>, ICloneable
    #else
    public partial class ConnectionTransportDataTypeCollection : List<ConnectionTransportDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public ConnectionTransportDataTypeCollection() {}

        /// <remarks />
        public ConnectionTransportDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public ConnectionTransportDataTypeCollection(IEnumerable<ConnectionTransportDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator ConnectionTransportDataTypeCollection(ConnectionTransportDataType[] values)
        {
            if (values != null)
            {
                return new ConnectionTransportDataTypeCollection(values);
            }

            return new ConnectionTransportDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator ConnectionTransportDataType[](ConnectionTransportDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (ConnectionTransportDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ConnectionTransportDataTypeCollection clone = new ConnectionTransportDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ConnectionTransportDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region NetworkAddressDataType Class
    #if (!OPCUA_EXCLUDE_NetworkAddressDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class NetworkAddressDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public NetworkAddressDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_networkInterface = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "NetworkInterface", IsRequired = false, Order = 1)]
        public string NetworkInterface
        {
            get { return m_networkInterface;  }
            set { m_networkInterface = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.NetworkAddressDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.NetworkAddressDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.NetworkAddressDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.NetworkAddressDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("NetworkInterface", NetworkInterface);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            NetworkInterface = decoder.ReadString("NetworkInterface");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            NetworkAddressDataType value = encodeable as NetworkAddressDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_networkInterface, value.m_networkInterface)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (NetworkAddressDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            NetworkAddressDataType clone = (NetworkAddressDataType)base.MemberwiseClone();

            clone.m_networkInterface = (string)Utils.Clone(this.m_networkInterface);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_networkInterface;
        #endregion
    }

    #region NetworkAddressDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfNetworkAddressDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "NetworkAddressDataType")]
    #if !NET_STANDARD
    public partial class NetworkAddressDataTypeCollection : List<NetworkAddressDataType>, ICloneable
    #else
    public partial class NetworkAddressDataTypeCollection : List<NetworkAddressDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public NetworkAddressDataTypeCollection() {}

        /// <remarks />
        public NetworkAddressDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public NetworkAddressDataTypeCollection(IEnumerable<NetworkAddressDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator NetworkAddressDataTypeCollection(NetworkAddressDataType[] values)
        {
            if (values != null)
            {
                return new NetworkAddressDataTypeCollection(values);
            }

            return new NetworkAddressDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator NetworkAddressDataType[](NetworkAddressDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (NetworkAddressDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            NetworkAddressDataTypeCollection clone = new NetworkAddressDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((NetworkAddressDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region NetworkAddressUrlDataType Class
    #if (!OPCUA_EXCLUDE_NetworkAddressUrlDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class NetworkAddressUrlDataType : NetworkAddressDataType
    {
        #region Constructors
        /// <remarks />
        public NetworkAddressUrlDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_url = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Url", IsRequired = false, Order = 1)]
        public string Url
        {
            get { return m_url;  }
            set { m_url = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.NetworkAddressUrlDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.NetworkAddressUrlDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.NetworkAddressUrlDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.NetworkAddressUrlDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("Url", Url);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Url = decoder.ReadString("Url");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            NetworkAddressUrlDataType value = encodeable as NetworkAddressUrlDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_url, value.m_url)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (NetworkAddressUrlDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            NetworkAddressUrlDataType clone = (NetworkAddressUrlDataType)base.MemberwiseClone();

            clone.m_url = (string)Utils.Clone(this.m_url);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_url;
        #endregion
    }

    #region NetworkAddressUrlDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfNetworkAddressUrlDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "NetworkAddressUrlDataType")]
    #if !NET_STANDARD
    public partial class NetworkAddressUrlDataTypeCollection : List<NetworkAddressUrlDataType>, ICloneable
    #else
    public partial class NetworkAddressUrlDataTypeCollection : List<NetworkAddressUrlDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public NetworkAddressUrlDataTypeCollection() {}

        /// <remarks />
        public NetworkAddressUrlDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public NetworkAddressUrlDataTypeCollection(IEnumerable<NetworkAddressUrlDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator NetworkAddressUrlDataTypeCollection(NetworkAddressUrlDataType[] values)
        {
            if (values != null)
            {
                return new NetworkAddressUrlDataTypeCollection(values);
            }

            return new NetworkAddressUrlDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator NetworkAddressUrlDataType[](NetworkAddressUrlDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (NetworkAddressUrlDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            NetworkAddressUrlDataTypeCollection clone = new NetworkAddressUrlDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((NetworkAddressUrlDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region ReaderGroupDataType Class
    #if (!OPCUA_EXCLUDE_ReaderGroupDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ReaderGroupDataType : PubSubGroupDataType
    {
        #region Constructors
        /// <remarks />
        public ReaderGroupDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_transportSettings = null;
            m_messageSettings = null;
            m_dataSetReaders = new DataSetReaderDataTypeCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "TransportSettings", IsRequired = false, Order = 1)]
        public ExtensionObject TransportSettings
        {
            get { return m_transportSettings;  }
            set { m_transportSettings = value; }
        }

        /// <remarks />
        [DataMember(Name = "MessageSettings", IsRequired = false, Order = 2)]
        public ExtensionObject MessageSettings
        {
            get { return m_messageSettings;  }
            set { m_messageSettings = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetReaders", IsRequired = false, Order = 3)]
        public DataSetReaderDataTypeCollection DataSetReaders
        {
            get
            {
                return m_dataSetReaders;
            }

            set
            {
                m_dataSetReaders = value;

                if (value == null)
                {
                    m_dataSetReaders = new DataSetReaderDataTypeCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.ReaderGroupDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.ReaderGroupDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.ReaderGroupDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.ReaderGroupDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteExtensionObject("TransportSettings", TransportSettings);
            encoder.WriteExtensionObject("MessageSettings", MessageSettings);
            encoder.WriteEncodeableArray("DataSetReaders", DataSetReaders.ToArray(), typeof(DataSetReaderDataType));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            TransportSettings = decoder.ReadExtensionObject("TransportSettings");
            MessageSettings = decoder.ReadExtensionObject("MessageSettings");
            DataSetReaders = (DataSetReaderDataTypeCollection)decoder.ReadEncodeableArray("DataSetReaders", typeof(DataSetReaderDataType));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ReaderGroupDataType value = encodeable as ReaderGroupDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_transportSettings, value.m_transportSettings)) return false;
            if (!Utils.IsEqual(m_messageSettings, value.m_messageSettings)) return false;
            if (!Utils.IsEqual(m_dataSetReaders, value.m_dataSetReaders)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (ReaderGroupDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReaderGroupDataType clone = (ReaderGroupDataType)base.MemberwiseClone();

            clone.m_transportSettings = (ExtensionObject)Utils.Clone(this.m_transportSettings);
            clone.m_messageSettings = (ExtensionObject)Utils.Clone(this.m_messageSettings);
            clone.m_dataSetReaders = (DataSetReaderDataTypeCollection)Utils.Clone(this.m_dataSetReaders);

            return clone;
        }
        #endregion

        #region Private Fields
        private ExtensionObject m_transportSettings;
        private ExtensionObject m_messageSettings;
        private DataSetReaderDataTypeCollection m_dataSetReaders;
        #endregion
    }

    #region ReaderGroupDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfReaderGroupDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ReaderGroupDataType")]
    #if !NET_STANDARD
    public partial class ReaderGroupDataTypeCollection : List<ReaderGroupDataType>, ICloneable
    #else
    public partial class ReaderGroupDataTypeCollection : List<ReaderGroupDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public ReaderGroupDataTypeCollection() {}

        /// <remarks />
        public ReaderGroupDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public ReaderGroupDataTypeCollection(IEnumerable<ReaderGroupDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator ReaderGroupDataTypeCollection(ReaderGroupDataType[] values)
        {
            if (values != null)
            {
                return new ReaderGroupDataTypeCollection(values);
            }

            return new ReaderGroupDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator ReaderGroupDataType[](ReaderGroupDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (ReaderGroupDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReaderGroupDataTypeCollection clone = new ReaderGroupDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ReaderGroupDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region ReaderGroupTransportDataType Class
    #if (!OPCUA_EXCLUDE_ReaderGroupTransportDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ReaderGroupTransportDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public ReaderGroupTransportDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.ReaderGroupTransportDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.ReaderGroupTransportDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.ReaderGroupTransportDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.ReaderGroupTransportDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ReaderGroupTransportDataType value = encodeable as ReaderGroupTransportDataType;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (ReaderGroupTransportDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReaderGroupTransportDataType clone = (ReaderGroupTransportDataType)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region ReaderGroupTransportDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfReaderGroupTransportDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ReaderGroupTransportDataType")]
    #if !NET_STANDARD
    public partial class ReaderGroupTransportDataTypeCollection : List<ReaderGroupTransportDataType>, ICloneable
    #else
    public partial class ReaderGroupTransportDataTypeCollection : List<ReaderGroupTransportDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public ReaderGroupTransportDataTypeCollection() {}

        /// <remarks />
        public ReaderGroupTransportDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public ReaderGroupTransportDataTypeCollection(IEnumerable<ReaderGroupTransportDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator ReaderGroupTransportDataTypeCollection(ReaderGroupTransportDataType[] values)
        {
            if (values != null)
            {
                return new ReaderGroupTransportDataTypeCollection(values);
            }

            return new ReaderGroupTransportDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator ReaderGroupTransportDataType[](ReaderGroupTransportDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (ReaderGroupTransportDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReaderGroupTransportDataTypeCollection clone = new ReaderGroupTransportDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ReaderGroupTransportDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region ReaderGroupMessageDataType Class
    #if (!OPCUA_EXCLUDE_ReaderGroupMessageDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ReaderGroupMessageDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public ReaderGroupMessageDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.ReaderGroupMessageDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.ReaderGroupMessageDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.ReaderGroupMessageDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.ReaderGroupMessageDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ReaderGroupMessageDataType value = encodeable as ReaderGroupMessageDataType;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (ReaderGroupMessageDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReaderGroupMessageDataType clone = (ReaderGroupMessageDataType)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region ReaderGroupMessageDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfReaderGroupMessageDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ReaderGroupMessageDataType")]
    #if !NET_STANDARD
    public partial class ReaderGroupMessageDataTypeCollection : List<ReaderGroupMessageDataType>, ICloneable
    #else
    public partial class ReaderGroupMessageDataTypeCollection : List<ReaderGroupMessageDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public ReaderGroupMessageDataTypeCollection() {}

        /// <remarks />
        public ReaderGroupMessageDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public ReaderGroupMessageDataTypeCollection(IEnumerable<ReaderGroupMessageDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator ReaderGroupMessageDataTypeCollection(ReaderGroupMessageDataType[] values)
        {
            if (values != null)
            {
                return new ReaderGroupMessageDataTypeCollection(values);
            }

            return new ReaderGroupMessageDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator ReaderGroupMessageDataType[](ReaderGroupMessageDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (ReaderGroupMessageDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReaderGroupMessageDataTypeCollection clone = new ReaderGroupMessageDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ReaderGroupMessageDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DataSetReaderDataType Class
    #if (!OPCUA_EXCLUDE_DataSetReaderDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DataSetReaderDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public DataSetReaderDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_name = null;
            m_enabled = true;
            m_publisherId = Variant.Null;
            m_writerGroupId = (ushort)0;
            m_dataSetWriterId = (ushort)0;
            m_dataSetMetaData = new DataSetMetaDataType();
            m_dataSetFieldContentMask = 0;
            m_messageReceiveTimeout = (double)0;
            m_keyFrameCount = (uint)0;
            m_headerLayoutUri = null;
            m_securityMode = MessageSecurityMode.Invalid;
            m_securityGroupId = null;
            m_securityKeyServices = new EndpointDescriptionCollection();
            m_dataSetReaderProperties = new KeyValuePairCollection();
            m_transportSettings = null;
            m_messageSettings = null;
            m_subscribedDataSet = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 1)]
        public string Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }

        /// <remarks />
        [DataMember(Name = "Enabled", IsRequired = false, Order = 2)]
        public bool Enabled
        {
            get { return m_enabled;  }
            set { m_enabled = value; }
        }

        /// <remarks />
        [DataMember(Name = "PublisherId", IsRequired = false, Order = 3)]
        public Variant PublisherId
        {
            get { return m_publisherId;  }
            set { m_publisherId = value; }
        }

        /// <remarks />
        [DataMember(Name = "WriterGroupId", IsRequired = false, Order = 4)]
        public ushort WriterGroupId
        {
            get { return m_writerGroupId;  }
            set { m_writerGroupId = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetWriterId", IsRequired = false, Order = 5)]
        public ushort DataSetWriterId
        {
            get { return m_dataSetWriterId;  }
            set { m_dataSetWriterId = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetMetaData", IsRequired = false, Order = 6)]
        public DataSetMetaDataType DataSetMetaData
        {
            get
            {
                return m_dataSetMetaData;
            }

            set
            {
                m_dataSetMetaData = value;

                if (value == null)
                {
                    m_dataSetMetaData = new DataSetMetaDataType();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "DataSetFieldContentMask", IsRequired = false, Order = 7)]
        public uint DataSetFieldContentMask
        {
            get { return m_dataSetFieldContentMask;  }
            set { m_dataSetFieldContentMask = value; }
        }

        /// <remarks />
        [DataMember(Name = "MessageReceiveTimeout", IsRequired = false, Order = 8)]
        public double MessageReceiveTimeout
        {
            get { return m_messageReceiveTimeout;  }
            set { m_messageReceiveTimeout = value; }
        }

        /// <remarks />
        [DataMember(Name = "KeyFrameCount", IsRequired = false, Order = 9)]
        public uint KeyFrameCount
        {
            get { return m_keyFrameCount;  }
            set { m_keyFrameCount = value; }
        }

        /// <remarks />
        [DataMember(Name = "HeaderLayoutUri", IsRequired = false, Order = 10)]
        public string HeaderLayoutUri
        {
            get { return m_headerLayoutUri;  }
            set { m_headerLayoutUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityMode", IsRequired = false, Order = 11)]
        public MessageSecurityMode SecurityMode
        {
            get { return m_securityMode;  }
            set { m_securityMode = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityGroupId", IsRequired = false, Order = 12)]
        public string SecurityGroupId
        {
            get { return m_securityGroupId;  }
            set { m_securityGroupId = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityKeyServices", IsRequired = false, Order = 13)]
        public EndpointDescriptionCollection SecurityKeyServices
        {
            get
            {
                return m_securityKeyServices;
            }

            set
            {
                m_securityKeyServices = value;

                if (value == null)
                {
                    m_securityKeyServices = new EndpointDescriptionCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "DataSetReaderProperties", IsRequired = false, Order = 14)]
        public KeyValuePairCollection DataSetReaderProperties
        {
            get
            {
                return m_dataSetReaderProperties;
            }

            set
            {
                m_dataSetReaderProperties = value;

                if (value == null)
                {
                    m_dataSetReaderProperties = new KeyValuePairCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "TransportSettings", IsRequired = false, Order = 15)]
        public ExtensionObject TransportSettings
        {
            get { return m_transportSettings;  }
            set { m_transportSettings = value; }
        }

        /// <remarks />
        [DataMember(Name = "MessageSettings", IsRequired = false, Order = 16)]
        public ExtensionObject MessageSettings
        {
            get { return m_messageSettings;  }
            set { m_messageSettings = value; }
        }

        /// <remarks />
        [DataMember(Name = "SubscribedDataSet", IsRequired = false, Order = 17)]
        public ExtensionObject SubscribedDataSet
        {
            get { return m_subscribedDataSet;  }
            set { m_subscribedDataSet = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.DataSetReaderDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.DataSetReaderDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.DataSetReaderDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.DataSetReaderDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("Name", Name);
            encoder.WriteBoolean("Enabled", Enabled);
            encoder.WriteVariant("PublisherId", PublisherId);
            encoder.WriteUInt16("WriterGroupId", WriterGroupId);
            encoder.WriteUInt16("DataSetWriterId", DataSetWriterId);
            encoder.WriteEncodeable("DataSetMetaData", DataSetMetaData, typeof(DataSetMetaDataType));
            encoder.WriteUInt32("DataSetFieldContentMask", DataSetFieldContentMask);
            encoder.WriteDouble("MessageReceiveTimeout", MessageReceiveTimeout);
            encoder.WriteUInt32("KeyFrameCount", KeyFrameCount);
            encoder.WriteString("HeaderLayoutUri", HeaderLayoutUri);
            encoder.WriteEnumerated("SecurityMode", SecurityMode);
            encoder.WriteString("SecurityGroupId", SecurityGroupId);
            encoder.WriteEncodeableArray("SecurityKeyServices", SecurityKeyServices.ToArray(), typeof(EndpointDescription));
            encoder.WriteEncodeableArray("DataSetReaderProperties", DataSetReaderProperties.ToArray(), typeof(KeyValuePair));
            encoder.WriteExtensionObject("TransportSettings", TransportSettings);
            encoder.WriteExtensionObject("MessageSettings", MessageSettings);
            encoder.WriteExtensionObject("SubscribedDataSet", SubscribedDataSet);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Name = decoder.ReadString("Name");
            Enabled = decoder.ReadBoolean("Enabled");
            PublisherId = decoder.ReadVariant("PublisherId");
            WriterGroupId = decoder.ReadUInt16("WriterGroupId");
            DataSetWriterId = decoder.ReadUInt16("DataSetWriterId");
            DataSetMetaData = (DataSetMetaDataType)decoder.ReadEncodeable("DataSetMetaData", typeof(DataSetMetaDataType));
            DataSetFieldContentMask = decoder.ReadUInt32("DataSetFieldContentMask");
            MessageReceiveTimeout = decoder.ReadDouble("MessageReceiveTimeout");
            KeyFrameCount = decoder.ReadUInt32("KeyFrameCount");
            HeaderLayoutUri = decoder.ReadString("HeaderLayoutUri");
            SecurityMode = (MessageSecurityMode)decoder.ReadEnumerated("SecurityMode", typeof(MessageSecurityMode));
            SecurityGroupId = decoder.ReadString("SecurityGroupId");
            SecurityKeyServices = (EndpointDescriptionCollection)decoder.ReadEncodeableArray("SecurityKeyServices", typeof(EndpointDescription));
            DataSetReaderProperties = (KeyValuePairCollection)decoder.ReadEncodeableArray("DataSetReaderProperties", typeof(KeyValuePair));
            TransportSettings = decoder.ReadExtensionObject("TransportSettings");
            MessageSettings = decoder.ReadExtensionObject("MessageSettings");
            SubscribedDataSet = decoder.ReadExtensionObject("SubscribedDataSet");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DataSetReaderDataType value = encodeable as DataSetReaderDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_name, value.m_name)) return false;
            if (!Utils.IsEqual(m_enabled, value.m_enabled)) return false;
            if (!Utils.IsEqual(m_publisherId, value.m_publisherId)) return false;
            if (!Utils.IsEqual(m_writerGroupId, value.m_writerGroupId)) return false;
            if (!Utils.IsEqual(m_dataSetWriterId, value.m_dataSetWriterId)) return false;
            if (!Utils.IsEqual(m_dataSetMetaData, value.m_dataSetMetaData)) return false;
            if (!Utils.IsEqual(m_dataSetFieldContentMask, value.m_dataSetFieldContentMask)) return false;
            if (!Utils.IsEqual(m_messageReceiveTimeout, value.m_messageReceiveTimeout)) return false;
            if (!Utils.IsEqual(m_keyFrameCount, value.m_keyFrameCount)) return false;
            if (!Utils.IsEqual(m_headerLayoutUri, value.m_headerLayoutUri)) return false;
            if (!Utils.IsEqual(m_securityMode, value.m_securityMode)) return false;
            if (!Utils.IsEqual(m_securityGroupId, value.m_securityGroupId)) return false;
            if (!Utils.IsEqual(m_securityKeyServices, value.m_securityKeyServices)) return false;
            if (!Utils.IsEqual(m_dataSetReaderProperties, value.m_dataSetReaderProperties)) return false;
            if (!Utils.IsEqual(m_transportSettings, value.m_transportSettings)) return false;
            if (!Utils.IsEqual(m_messageSettings, value.m_messageSettings)) return false;
            if (!Utils.IsEqual(m_subscribedDataSet, value.m_subscribedDataSet)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (DataSetReaderDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetReaderDataType clone = (DataSetReaderDataType)base.MemberwiseClone();

            clone.m_name = (string)Utils.Clone(this.m_name);
            clone.m_enabled = (bool)Utils.Clone(this.m_enabled);
            clone.m_publisherId = (Variant)Utils.Clone(this.m_publisherId);
            clone.m_writerGroupId = (ushort)Utils.Clone(this.m_writerGroupId);
            clone.m_dataSetWriterId = (ushort)Utils.Clone(this.m_dataSetWriterId);
            clone.m_dataSetMetaData = (DataSetMetaDataType)Utils.Clone(this.m_dataSetMetaData);
            clone.m_dataSetFieldContentMask = (uint)Utils.Clone(this.m_dataSetFieldContentMask);
            clone.m_messageReceiveTimeout = (double)Utils.Clone(this.m_messageReceiveTimeout);
            clone.m_keyFrameCount = (uint)Utils.Clone(this.m_keyFrameCount);
            clone.m_headerLayoutUri = (string)Utils.Clone(this.m_headerLayoutUri);
            clone.m_securityMode = (MessageSecurityMode)Utils.Clone(this.m_securityMode);
            clone.m_securityGroupId = (string)Utils.Clone(this.m_securityGroupId);
            clone.m_securityKeyServices = (EndpointDescriptionCollection)Utils.Clone(this.m_securityKeyServices);
            clone.m_dataSetReaderProperties = (KeyValuePairCollection)Utils.Clone(this.m_dataSetReaderProperties);
            clone.m_transportSettings = (ExtensionObject)Utils.Clone(this.m_transportSettings);
            clone.m_messageSettings = (ExtensionObject)Utils.Clone(this.m_messageSettings);
            clone.m_subscribedDataSet = (ExtensionObject)Utils.Clone(this.m_subscribedDataSet);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_name;
        private bool m_enabled;
        private Variant m_publisherId;
        private ushort m_writerGroupId;
        private ushort m_dataSetWriterId;
        private DataSetMetaDataType m_dataSetMetaData;
        private uint m_dataSetFieldContentMask;
        private double m_messageReceiveTimeout;
        private uint m_keyFrameCount;
        private string m_headerLayoutUri;
        private MessageSecurityMode m_securityMode;
        private string m_securityGroupId;
        private EndpointDescriptionCollection m_securityKeyServices;
        private KeyValuePairCollection m_dataSetReaderProperties;
        private ExtensionObject m_transportSettings;
        private ExtensionObject m_messageSettings;
        private ExtensionObject m_subscribedDataSet;
        #endregion
    }

    #region DataSetReaderDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDataSetReaderDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DataSetReaderDataType")]
    #if !NET_STANDARD
    public partial class DataSetReaderDataTypeCollection : List<DataSetReaderDataType>, ICloneable
    #else
    public partial class DataSetReaderDataTypeCollection : List<DataSetReaderDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public DataSetReaderDataTypeCollection() {}

        /// <remarks />
        public DataSetReaderDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public DataSetReaderDataTypeCollection(IEnumerable<DataSetReaderDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator DataSetReaderDataTypeCollection(DataSetReaderDataType[] values)
        {
            if (values != null)
            {
                return new DataSetReaderDataTypeCollection(values);
            }

            return new DataSetReaderDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator DataSetReaderDataType[](DataSetReaderDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (DataSetReaderDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetReaderDataTypeCollection clone = new DataSetReaderDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DataSetReaderDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DataSetReaderTransportDataType Class
    #if (!OPCUA_EXCLUDE_DataSetReaderTransportDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DataSetReaderTransportDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public DataSetReaderTransportDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.DataSetReaderTransportDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.DataSetReaderTransportDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.DataSetReaderTransportDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.DataSetReaderTransportDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DataSetReaderTransportDataType value = encodeable as DataSetReaderTransportDataType;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (DataSetReaderTransportDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetReaderTransportDataType clone = (DataSetReaderTransportDataType)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region DataSetReaderTransportDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDataSetReaderTransportDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DataSetReaderTransportDataType")]
    #if !NET_STANDARD
    public partial class DataSetReaderTransportDataTypeCollection : List<DataSetReaderTransportDataType>, ICloneable
    #else
    public partial class DataSetReaderTransportDataTypeCollection : List<DataSetReaderTransportDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public DataSetReaderTransportDataTypeCollection() {}

        /// <remarks />
        public DataSetReaderTransportDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public DataSetReaderTransportDataTypeCollection(IEnumerable<DataSetReaderTransportDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator DataSetReaderTransportDataTypeCollection(DataSetReaderTransportDataType[] values)
        {
            if (values != null)
            {
                return new DataSetReaderTransportDataTypeCollection(values);
            }

            return new DataSetReaderTransportDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator DataSetReaderTransportDataType[](DataSetReaderTransportDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (DataSetReaderTransportDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetReaderTransportDataTypeCollection clone = new DataSetReaderTransportDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DataSetReaderTransportDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DataSetReaderMessageDataType Class
    #if (!OPCUA_EXCLUDE_DataSetReaderMessageDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DataSetReaderMessageDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public DataSetReaderMessageDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.DataSetReaderMessageDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.DataSetReaderMessageDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.DataSetReaderMessageDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.DataSetReaderMessageDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DataSetReaderMessageDataType value = encodeable as DataSetReaderMessageDataType;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (DataSetReaderMessageDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetReaderMessageDataType clone = (DataSetReaderMessageDataType)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region DataSetReaderMessageDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDataSetReaderMessageDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DataSetReaderMessageDataType")]
    #if !NET_STANDARD
    public partial class DataSetReaderMessageDataTypeCollection : List<DataSetReaderMessageDataType>, ICloneable
    #else
    public partial class DataSetReaderMessageDataTypeCollection : List<DataSetReaderMessageDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public DataSetReaderMessageDataTypeCollection() {}

        /// <remarks />
        public DataSetReaderMessageDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public DataSetReaderMessageDataTypeCollection(IEnumerable<DataSetReaderMessageDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator DataSetReaderMessageDataTypeCollection(DataSetReaderMessageDataType[] values)
        {
            if (values != null)
            {
                return new DataSetReaderMessageDataTypeCollection(values);
            }

            return new DataSetReaderMessageDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator DataSetReaderMessageDataType[](DataSetReaderMessageDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (DataSetReaderMessageDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetReaderMessageDataTypeCollection clone = new DataSetReaderMessageDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DataSetReaderMessageDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region SubscribedDataSetDataType Class
    #if (!OPCUA_EXCLUDE_SubscribedDataSetDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class SubscribedDataSetDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public SubscribedDataSetDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.SubscribedDataSetDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.SubscribedDataSetDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.SubscribedDataSetDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.SubscribedDataSetDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            SubscribedDataSetDataType value = encodeable as SubscribedDataSetDataType;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (SubscribedDataSetDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            SubscribedDataSetDataType clone = (SubscribedDataSetDataType)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region SubscribedDataSetDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfSubscribedDataSetDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "SubscribedDataSetDataType")]
    #if !NET_STANDARD
    public partial class SubscribedDataSetDataTypeCollection : List<SubscribedDataSetDataType>, ICloneable
    #else
    public partial class SubscribedDataSetDataTypeCollection : List<SubscribedDataSetDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public SubscribedDataSetDataTypeCollection() {}

        /// <remarks />
        public SubscribedDataSetDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public SubscribedDataSetDataTypeCollection(IEnumerable<SubscribedDataSetDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator SubscribedDataSetDataTypeCollection(SubscribedDataSetDataType[] values)
        {
            if (values != null)
            {
                return new SubscribedDataSetDataTypeCollection(values);
            }

            return new SubscribedDataSetDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator SubscribedDataSetDataType[](SubscribedDataSetDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (SubscribedDataSetDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            SubscribedDataSetDataTypeCollection clone = new SubscribedDataSetDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((SubscribedDataSetDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region TargetVariablesDataType Class
    #if (!OPCUA_EXCLUDE_TargetVariablesDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class TargetVariablesDataType : SubscribedDataSetDataType
    {
        #region Constructors
        /// <remarks />
        public TargetVariablesDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_targetVariables = new FieldTargetDataTypeCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "TargetVariables", IsRequired = false, Order = 1)]
        public FieldTargetDataTypeCollection TargetVariables
        {
            get
            {
                return m_targetVariables;
            }

            set
            {
                m_targetVariables = value;

                if (value == null)
                {
                    m_targetVariables = new FieldTargetDataTypeCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.TargetVariablesDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.TargetVariablesDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.TargetVariablesDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.TargetVariablesDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeableArray("TargetVariables", TargetVariables.ToArray(), typeof(FieldTargetDataType));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            TargetVariables = (FieldTargetDataTypeCollection)decoder.ReadEncodeableArray("TargetVariables", typeof(FieldTargetDataType));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            TargetVariablesDataType value = encodeable as TargetVariablesDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_targetVariables, value.m_targetVariables)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (TargetVariablesDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TargetVariablesDataType clone = (TargetVariablesDataType)base.MemberwiseClone();

            clone.m_targetVariables = (FieldTargetDataTypeCollection)Utils.Clone(this.m_targetVariables);

            return clone;
        }
        #endregion

        #region Private Fields
        private FieldTargetDataTypeCollection m_targetVariables;
        #endregion
    }

    #region TargetVariablesDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfTargetVariablesDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "TargetVariablesDataType")]
    #if !NET_STANDARD
    public partial class TargetVariablesDataTypeCollection : List<TargetVariablesDataType>, ICloneable
    #else
    public partial class TargetVariablesDataTypeCollection : List<TargetVariablesDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public TargetVariablesDataTypeCollection() {}

        /// <remarks />
        public TargetVariablesDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public TargetVariablesDataTypeCollection(IEnumerable<TargetVariablesDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator TargetVariablesDataTypeCollection(TargetVariablesDataType[] values)
        {
            if (values != null)
            {
                return new TargetVariablesDataTypeCollection(values);
            }

            return new TargetVariablesDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator TargetVariablesDataType[](TargetVariablesDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (TargetVariablesDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TargetVariablesDataTypeCollection clone = new TargetVariablesDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((TargetVariablesDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region FieldTargetDataType Class
    #if (!OPCUA_EXCLUDE_FieldTargetDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class FieldTargetDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public FieldTargetDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_dataSetFieldId = Uuid.Empty;
            m_receiverIndexRange = null;
            m_targetNodeId = null;
            m_attributeId = (uint)0;
            m_writeIndexRange = null;
            m_overrideValueHandling = OverrideValueHandling.Disabled;
            m_overrideValue = Variant.Null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "DataSetFieldId", IsRequired = false, Order = 1)]
        public Uuid DataSetFieldId
        {
            get { return m_dataSetFieldId;  }
            set { m_dataSetFieldId = value; }
        }

        /// <remarks />
        [DataMember(Name = "ReceiverIndexRange", IsRequired = false, Order = 2)]
        public string ReceiverIndexRange
        {
            get { return m_receiverIndexRange;  }
            set { m_receiverIndexRange = value; }
        }

        /// <remarks />
        [DataMember(Name = "TargetNodeId", IsRequired = false, Order = 3)]
        public NodeId TargetNodeId
        {
            get { return m_targetNodeId;  }
            set { m_targetNodeId = value; }
        }

        /// <remarks />
        [DataMember(Name = "AttributeId", IsRequired = false, Order = 4)]
        public uint AttributeId
        {
            get { return m_attributeId;  }
            set { m_attributeId = value; }
        }

        /// <remarks />
        [DataMember(Name = "WriteIndexRange", IsRequired = false, Order = 5)]
        public string WriteIndexRange
        {
            get { return m_writeIndexRange;  }
            set { m_writeIndexRange = value; }
        }

        /// <remarks />
        [DataMember(Name = "OverrideValueHandling", IsRequired = false, Order = 6)]
        public OverrideValueHandling OverrideValueHandling
        {
            get { return m_overrideValueHandling;  }
            set { m_overrideValueHandling = value; }
        }

        /// <remarks />
        [DataMember(Name = "OverrideValue", IsRequired = false, Order = 7)]
        public Variant OverrideValue
        {
            get { return m_overrideValue;  }
            set { m_overrideValue = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.FieldTargetDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.FieldTargetDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.FieldTargetDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.FieldTargetDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteGuid("DataSetFieldId", DataSetFieldId);
            encoder.WriteString("ReceiverIndexRange", ReceiverIndexRange);
            encoder.WriteNodeId("TargetNodeId", TargetNodeId);
            encoder.WriteUInt32("AttributeId", AttributeId);
            encoder.WriteString("WriteIndexRange", WriteIndexRange);
            encoder.WriteEnumerated("OverrideValueHandling", OverrideValueHandling);
            encoder.WriteVariant("OverrideValue", OverrideValue);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            DataSetFieldId = decoder.ReadGuid("DataSetFieldId");
            ReceiverIndexRange = decoder.ReadString("ReceiverIndexRange");
            TargetNodeId = decoder.ReadNodeId("TargetNodeId");
            AttributeId = decoder.ReadUInt32("AttributeId");
            WriteIndexRange = decoder.ReadString("WriteIndexRange");
            OverrideValueHandling = (OverrideValueHandling)decoder.ReadEnumerated("OverrideValueHandling", typeof(OverrideValueHandling));
            OverrideValue = decoder.ReadVariant("OverrideValue");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            FieldTargetDataType value = encodeable as FieldTargetDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_dataSetFieldId, value.m_dataSetFieldId)) return false;
            if (!Utils.IsEqual(m_receiverIndexRange, value.m_receiverIndexRange)) return false;
            if (!Utils.IsEqual(m_targetNodeId, value.m_targetNodeId)) return false;
            if (!Utils.IsEqual(m_attributeId, value.m_attributeId)) return false;
            if (!Utils.IsEqual(m_writeIndexRange, value.m_writeIndexRange)) return false;
            if (!Utils.IsEqual(m_overrideValueHandling, value.m_overrideValueHandling)) return false;
            if (!Utils.IsEqual(m_overrideValue, value.m_overrideValue)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (FieldTargetDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            FieldTargetDataType clone = (FieldTargetDataType)base.MemberwiseClone();

            clone.m_dataSetFieldId = (Uuid)Utils.Clone(this.m_dataSetFieldId);
            clone.m_receiverIndexRange = (string)Utils.Clone(this.m_receiverIndexRange);
            clone.m_targetNodeId = (NodeId)Utils.Clone(this.m_targetNodeId);
            clone.m_attributeId = (uint)Utils.Clone(this.m_attributeId);
            clone.m_writeIndexRange = (string)Utils.Clone(this.m_writeIndexRange);
            clone.m_overrideValueHandling = (OverrideValueHandling)Utils.Clone(this.m_overrideValueHandling);
            clone.m_overrideValue = (Variant)Utils.Clone(this.m_overrideValue);

            return clone;
        }
        #endregion

        #region Private Fields
        private Uuid m_dataSetFieldId;
        private string m_receiverIndexRange;
        private NodeId m_targetNodeId;
        private uint m_attributeId;
        private string m_writeIndexRange;
        private OverrideValueHandling m_overrideValueHandling;
        private Variant m_overrideValue;
        #endregion
    }

    #region FieldTargetDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfFieldTargetDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "FieldTargetDataType")]
    #if !NET_STANDARD
    public partial class FieldTargetDataTypeCollection : List<FieldTargetDataType>, ICloneable
    #else
    public partial class FieldTargetDataTypeCollection : List<FieldTargetDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public FieldTargetDataTypeCollection() {}

        /// <remarks />
        public FieldTargetDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public FieldTargetDataTypeCollection(IEnumerable<FieldTargetDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator FieldTargetDataTypeCollection(FieldTargetDataType[] values)
        {
            if (values != null)
            {
                return new FieldTargetDataTypeCollection(values);
            }

            return new FieldTargetDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator FieldTargetDataType[](FieldTargetDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (FieldTargetDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            FieldTargetDataTypeCollection clone = new FieldTargetDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((FieldTargetDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region OverrideValueHandling Enumeration
    #if (!OPCUA_EXCLUDE_OverrideValueHandling)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum OverrideValueHandling
    {
        /// <remarks />
        [EnumMember(Value = "Disabled_0")]
        Disabled = 0,

        /// <remarks />
        [EnumMember(Value = "LastUsableValue_1")]
        LastUsableValue = 1,

        /// <remarks />
        [EnumMember(Value = "OverrideValue_2")]
        OverrideValue = 2,
    }

    #region OverrideValueHandlingCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfOverrideValueHandling", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "OverrideValueHandling")]
    #if !NET_STANDARD
    public partial class OverrideValueHandlingCollection : List<OverrideValueHandling>, ICloneable
    #else
    public partial class OverrideValueHandlingCollection : List<OverrideValueHandling>
    #endif
    {
        #region Constructors
        /// <remarks />
        public OverrideValueHandlingCollection() {}

        /// <remarks />
        public OverrideValueHandlingCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public OverrideValueHandlingCollection(IEnumerable<OverrideValueHandling> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator OverrideValueHandlingCollection(OverrideValueHandling[] values)
        {
            if (values != null)
            {
                return new OverrideValueHandlingCollection(values);
            }

            return new OverrideValueHandlingCollection();
        }

        /// <remarks />
        public static explicit operator OverrideValueHandling[](OverrideValueHandlingCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (OverrideValueHandlingCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            OverrideValueHandlingCollection clone = new OverrideValueHandlingCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((OverrideValueHandling)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region SubscribedDataSetMirrorDataType Class
    #if (!OPCUA_EXCLUDE_SubscribedDataSetMirrorDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class SubscribedDataSetMirrorDataType : SubscribedDataSetDataType
    {
        #region Constructors
        /// <remarks />
        public SubscribedDataSetMirrorDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_parentNodeName = null;
            m_rolePermissions = new RolePermissionTypeCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "ParentNodeName", IsRequired = false, Order = 1)]
        public string ParentNodeName
        {
            get { return m_parentNodeName;  }
            set { m_parentNodeName = value; }
        }

        /// <remarks />
        [DataMember(Name = "RolePermissions", IsRequired = false, Order = 2)]
        public RolePermissionTypeCollection RolePermissions
        {
            get
            {
                return m_rolePermissions;
            }

            set
            {
                m_rolePermissions = value;

                if (value == null)
                {
                    m_rolePermissions = new RolePermissionTypeCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.SubscribedDataSetMirrorDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.SubscribedDataSetMirrorDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.SubscribedDataSetMirrorDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.SubscribedDataSetMirrorDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("ParentNodeName", ParentNodeName);
            encoder.WriteEncodeableArray("RolePermissions", RolePermissions.ToArray(), typeof(RolePermissionType));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ParentNodeName = decoder.ReadString("ParentNodeName");
            RolePermissions = (RolePermissionTypeCollection)decoder.ReadEncodeableArray("RolePermissions", typeof(RolePermissionType));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            SubscribedDataSetMirrorDataType value = encodeable as SubscribedDataSetMirrorDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_parentNodeName, value.m_parentNodeName)) return false;
            if (!Utils.IsEqual(m_rolePermissions, value.m_rolePermissions)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (SubscribedDataSetMirrorDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            SubscribedDataSetMirrorDataType clone = (SubscribedDataSetMirrorDataType)base.MemberwiseClone();

            clone.m_parentNodeName = (string)Utils.Clone(this.m_parentNodeName);
            clone.m_rolePermissions = (RolePermissionTypeCollection)Utils.Clone(this.m_rolePermissions);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_parentNodeName;
        private RolePermissionTypeCollection m_rolePermissions;
        #endregion
    }

    #region SubscribedDataSetMirrorDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfSubscribedDataSetMirrorDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "SubscribedDataSetMirrorDataType")]
    #if !NET_STANDARD
    public partial class SubscribedDataSetMirrorDataTypeCollection : List<SubscribedDataSetMirrorDataType>, ICloneable
    #else
    public partial class SubscribedDataSetMirrorDataTypeCollection : List<SubscribedDataSetMirrorDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public SubscribedDataSetMirrorDataTypeCollection() {}

        /// <remarks />
        public SubscribedDataSetMirrorDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public SubscribedDataSetMirrorDataTypeCollection(IEnumerable<SubscribedDataSetMirrorDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator SubscribedDataSetMirrorDataTypeCollection(SubscribedDataSetMirrorDataType[] values)
        {
            if (values != null)
            {
                return new SubscribedDataSetMirrorDataTypeCollection(values);
            }

            return new SubscribedDataSetMirrorDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator SubscribedDataSetMirrorDataType[](SubscribedDataSetMirrorDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (SubscribedDataSetMirrorDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            SubscribedDataSetMirrorDataTypeCollection clone = new SubscribedDataSetMirrorDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((SubscribedDataSetMirrorDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PubSubConfigurationDataType Class
    #if (!OPCUA_EXCLUDE_PubSubConfigurationDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class PubSubConfigurationDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public PubSubConfigurationDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_publishedDataSets = new PublishedDataSetDataTypeCollection();
            m_connections = new PubSubConnectionDataTypeCollection();
            m_enabled = true;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "PublishedDataSets", IsRequired = false, Order = 1)]
        public PublishedDataSetDataTypeCollection PublishedDataSets
        {
            get
            {
                return m_publishedDataSets;
            }

            set
            {
                m_publishedDataSets = value;

                if (value == null)
                {
                    m_publishedDataSets = new PublishedDataSetDataTypeCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "Connections", IsRequired = false, Order = 2)]
        public PubSubConnectionDataTypeCollection Connections
        {
            get
            {
                return m_connections;
            }

            set
            {
                m_connections = value;

                if (value == null)
                {
                    m_connections = new PubSubConnectionDataTypeCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "Enabled", IsRequired = false, Order = 3)]
        public bool Enabled
        {
            get { return m_enabled;  }
            set { m_enabled = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.PubSubConfigurationDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.PubSubConfigurationDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.PubSubConfigurationDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.PubSubConfigurationDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeableArray("PublishedDataSets", PublishedDataSets.ToArray(), typeof(PublishedDataSetDataType));
            encoder.WriteEncodeableArray("Connections", Connections.ToArray(), typeof(PubSubConnectionDataType));
            encoder.WriteBoolean("Enabled", Enabled);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            PublishedDataSets = (PublishedDataSetDataTypeCollection)decoder.ReadEncodeableArray("PublishedDataSets", typeof(PublishedDataSetDataType));
            Connections = (PubSubConnectionDataTypeCollection)decoder.ReadEncodeableArray("Connections", typeof(PubSubConnectionDataType));
            Enabled = decoder.ReadBoolean("Enabled");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            PubSubConfigurationDataType value = encodeable as PubSubConfigurationDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_publishedDataSets, value.m_publishedDataSets)) return false;
            if (!Utils.IsEqual(m_connections, value.m_connections)) return false;
            if (!Utils.IsEqual(m_enabled, value.m_enabled)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (PubSubConfigurationDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PubSubConfigurationDataType clone = (PubSubConfigurationDataType)base.MemberwiseClone();

            clone.m_publishedDataSets = (PublishedDataSetDataTypeCollection)Utils.Clone(this.m_publishedDataSets);
            clone.m_connections = (PubSubConnectionDataTypeCollection)Utils.Clone(this.m_connections);
            clone.m_enabled = (bool)Utils.Clone(this.m_enabled);

            return clone;
        }
        #endregion

        #region Private Fields
        private PublishedDataSetDataTypeCollection m_publishedDataSets;
        private PubSubConnectionDataTypeCollection m_connections;
        private bool m_enabled;
        #endregion
    }

    #region PubSubConfigurationDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPubSubConfigurationDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PubSubConfigurationDataType")]
    #if !NET_STANDARD
    public partial class PubSubConfigurationDataTypeCollection : List<PubSubConfigurationDataType>, ICloneable
    #else
    public partial class PubSubConfigurationDataTypeCollection : List<PubSubConfigurationDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public PubSubConfigurationDataTypeCollection() {}

        /// <remarks />
        public PubSubConfigurationDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public PubSubConfigurationDataTypeCollection(IEnumerable<PubSubConfigurationDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator PubSubConfigurationDataTypeCollection(PubSubConfigurationDataType[] values)
        {
            if (values != null)
            {
                return new PubSubConfigurationDataTypeCollection(values);
            }

            return new PubSubConfigurationDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator PubSubConfigurationDataType[](PubSubConfigurationDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (PubSubConfigurationDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PubSubConfigurationDataTypeCollection clone = new PubSubConfigurationDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PubSubConfigurationDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region StandaloneSubscribedDataSetRefDataType Class
    #if (!OPCUA_EXCLUDE_StandaloneSubscribedDataSetRefDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class StandaloneSubscribedDataSetRefDataType : SubscribedDataSetDataType
    {
        #region Constructors
        /// <remarks />
        public StandaloneSubscribedDataSetRefDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_dataSetName = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "DataSetName", IsRequired = false, Order = 1)]
        public string DataSetName
        {
            get { return m_dataSetName;  }
            set { m_dataSetName = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.StandaloneSubscribedDataSetRefDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.StandaloneSubscribedDataSetRefDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.StandaloneSubscribedDataSetRefDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.StandaloneSubscribedDataSetRefDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("DataSetName", DataSetName);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            DataSetName = decoder.ReadString("DataSetName");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            StandaloneSubscribedDataSetRefDataType value = encodeable as StandaloneSubscribedDataSetRefDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_dataSetName, value.m_dataSetName)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (StandaloneSubscribedDataSetRefDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            StandaloneSubscribedDataSetRefDataType clone = (StandaloneSubscribedDataSetRefDataType)base.MemberwiseClone();

            clone.m_dataSetName = (string)Utils.Clone(this.m_dataSetName);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_dataSetName;
        #endregion
    }

    #region StandaloneSubscribedDataSetRefDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfStandaloneSubscribedDataSetRefDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "StandaloneSubscribedDataSetRefDataType")]
    #if !NET_STANDARD
    public partial class StandaloneSubscribedDataSetRefDataTypeCollection : List<StandaloneSubscribedDataSetRefDataType>, ICloneable
    #else
    public partial class StandaloneSubscribedDataSetRefDataTypeCollection : List<StandaloneSubscribedDataSetRefDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public StandaloneSubscribedDataSetRefDataTypeCollection() {}

        /// <remarks />
        public StandaloneSubscribedDataSetRefDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public StandaloneSubscribedDataSetRefDataTypeCollection(IEnumerable<StandaloneSubscribedDataSetRefDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator StandaloneSubscribedDataSetRefDataTypeCollection(StandaloneSubscribedDataSetRefDataType[] values)
        {
            if (values != null)
            {
                return new StandaloneSubscribedDataSetRefDataTypeCollection(values);
            }

            return new StandaloneSubscribedDataSetRefDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator StandaloneSubscribedDataSetRefDataType[](StandaloneSubscribedDataSetRefDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (StandaloneSubscribedDataSetRefDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            StandaloneSubscribedDataSetRefDataTypeCollection clone = new StandaloneSubscribedDataSetRefDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((StandaloneSubscribedDataSetRefDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region StandaloneSubscribedDataSetDataType Class
    #if (!OPCUA_EXCLUDE_StandaloneSubscribedDataSetDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class StandaloneSubscribedDataSetDataType : SubscribedDataSetDataType
    {
        #region Constructors
        /// <remarks />
        public StandaloneSubscribedDataSetDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_name = null;
            m_dataSetFolder = new StringCollection();
            m_dataSetMetaData = new DataSetMetaDataType();
            m_subscribedDataSet = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 1)]
        public string Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetFolder", IsRequired = false, Order = 2)]
        public StringCollection DataSetFolder
        {
            get
            {
                return m_dataSetFolder;
            }

            set
            {
                m_dataSetFolder = value;

                if (value == null)
                {
                    m_dataSetFolder = new StringCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "DataSetMetaData", IsRequired = false, Order = 3)]
        public DataSetMetaDataType DataSetMetaData
        {
            get
            {
                return m_dataSetMetaData;
            }

            set
            {
                m_dataSetMetaData = value;

                if (value == null)
                {
                    m_dataSetMetaData = new DataSetMetaDataType();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "SubscribedDataSet", IsRequired = false, Order = 4)]
        public ExtensionObject SubscribedDataSet
        {
            get { return m_subscribedDataSet;  }
            set { m_subscribedDataSet = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.StandaloneSubscribedDataSetDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.StandaloneSubscribedDataSetDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.StandaloneSubscribedDataSetDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.StandaloneSubscribedDataSetDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("Name", Name);
            encoder.WriteStringArray("DataSetFolder", DataSetFolder);
            encoder.WriteEncodeable("DataSetMetaData", DataSetMetaData, typeof(DataSetMetaDataType));
            encoder.WriteExtensionObject("SubscribedDataSet", SubscribedDataSet);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Name = decoder.ReadString("Name");
            DataSetFolder = decoder.ReadStringArray("DataSetFolder");
            DataSetMetaData = (DataSetMetaDataType)decoder.ReadEncodeable("DataSetMetaData", typeof(DataSetMetaDataType));
            SubscribedDataSet = decoder.ReadExtensionObject("SubscribedDataSet");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            StandaloneSubscribedDataSetDataType value = encodeable as StandaloneSubscribedDataSetDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_name, value.m_name)) return false;
            if (!Utils.IsEqual(m_dataSetFolder, value.m_dataSetFolder)) return false;
            if (!Utils.IsEqual(m_dataSetMetaData, value.m_dataSetMetaData)) return false;
            if (!Utils.IsEqual(m_subscribedDataSet, value.m_subscribedDataSet)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (StandaloneSubscribedDataSetDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            StandaloneSubscribedDataSetDataType clone = (StandaloneSubscribedDataSetDataType)base.MemberwiseClone();

            clone.m_name = (string)Utils.Clone(this.m_name);
            clone.m_dataSetFolder = (StringCollection)Utils.Clone(this.m_dataSetFolder);
            clone.m_dataSetMetaData = (DataSetMetaDataType)Utils.Clone(this.m_dataSetMetaData);
            clone.m_subscribedDataSet = (ExtensionObject)Utils.Clone(this.m_subscribedDataSet);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_name;
        private StringCollection m_dataSetFolder;
        private DataSetMetaDataType m_dataSetMetaData;
        private ExtensionObject m_subscribedDataSet;
        #endregion
    }

    #region StandaloneSubscribedDataSetDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfStandaloneSubscribedDataSetDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "StandaloneSubscribedDataSetDataType")]
    #if !NET_STANDARD
    public partial class StandaloneSubscribedDataSetDataTypeCollection : List<StandaloneSubscribedDataSetDataType>, ICloneable
    #else
    public partial class StandaloneSubscribedDataSetDataTypeCollection : List<StandaloneSubscribedDataSetDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public StandaloneSubscribedDataSetDataTypeCollection() {}

        /// <remarks />
        public StandaloneSubscribedDataSetDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public StandaloneSubscribedDataSetDataTypeCollection(IEnumerable<StandaloneSubscribedDataSetDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator StandaloneSubscribedDataSetDataTypeCollection(StandaloneSubscribedDataSetDataType[] values)
        {
            if (values != null)
            {
                return new StandaloneSubscribedDataSetDataTypeCollection(values);
            }

            return new StandaloneSubscribedDataSetDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator StandaloneSubscribedDataSetDataType[](StandaloneSubscribedDataSetDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (StandaloneSubscribedDataSetDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            StandaloneSubscribedDataSetDataTypeCollection clone = new StandaloneSubscribedDataSetDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((StandaloneSubscribedDataSetDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region SecurityGroupDataType Class
    #if (!OPCUA_EXCLUDE_SecurityGroupDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class SecurityGroupDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public SecurityGroupDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_name = null;
            m_securityGroupFolder = new StringCollection();
            m_keyLifetime = (double)0;
            m_securityPolicyUri = null;
            m_maxFutureKeyCount = (uint)0;
            m_maxPastKeyCount = (uint)0;
            m_securityGroupId = null;
            m_rolePermissions = new RolePermissionTypeCollection();
            m_groupProperties = new KeyValuePairCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 1)]
        public string Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityGroupFolder", IsRequired = false, Order = 2)]
        public StringCollection SecurityGroupFolder
        {
            get
            {
                return m_securityGroupFolder;
            }

            set
            {
                m_securityGroupFolder = value;

                if (value == null)
                {
                    m_securityGroupFolder = new StringCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "KeyLifetime", IsRequired = false, Order = 3)]
        public double KeyLifetime
        {
            get { return m_keyLifetime;  }
            set { m_keyLifetime = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityPolicyUri", IsRequired = false, Order = 4)]
        public string SecurityPolicyUri
        {
            get { return m_securityPolicyUri;  }
            set { m_securityPolicyUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "MaxFutureKeyCount", IsRequired = false, Order = 5)]
        public uint MaxFutureKeyCount
        {
            get { return m_maxFutureKeyCount;  }
            set { m_maxFutureKeyCount = value; }
        }

        /// <remarks />
        [DataMember(Name = "MaxPastKeyCount", IsRequired = false, Order = 6)]
        public uint MaxPastKeyCount
        {
            get { return m_maxPastKeyCount;  }
            set { m_maxPastKeyCount = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityGroupId", IsRequired = false, Order = 7)]
        public string SecurityGroupId
        {
            get { return m_securityGroupId;  }
            set { m_securityGroupId = value; }
        }

        /// <remarks />
        [DataMember(Name = "RolePermissions", IsRequired = false, Order = 8)]
        public RolePermissionTypeCollection RolePermissions
        {
            get
            {
                return m_rolePermissions;
            }

            set
            {
                m_rolePermissions = value;

                if (value == null)
                {
                    m_rolePermissions = new RolePermissionTypeCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "GroupProperties", IsRequired = false, Order = 9)]
        public KeyValuePairCollection GroupProperties
        {
            get
            {
                return m_groupProperties;
            }

            set
            {
                m_groupProperties = value;

                if (value == null)
                {
                    m_groupProperties = new KeyValuePairCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.SecurityGroupDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.SecurityGroupDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.SecurityGroupDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.SecurityGroupDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("Name", Name);
            encoder.WriteStringArray("SecurityGroupFolder", SecurityGroupFolder);
            encoder.WriteDouble("KeyLifetime", KeyLifetime);
            encoder.WriteString("SecurityPolicyUri", SecurityPolicyUri);
            encoder.WriteUInt32("MaxFutureKeyCount", MaxFutureKeyCount);
            encoder.WriteUInt32("MaxPastKeyCount", MaxPastKeyCount);
            encoder.WriteString("SecurityGroupId", SecurityGroupId);
            encoder.WriteEncodeableArray("RolePermissions", RolePermissions.ToArray(), typeof(RolePermissionType));
            encoder.WriteEncodeableArray("GroupProperties", GroupProperties.ToArray(), typeof(KeyValuePair));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Name = decoder.ReadString("Name");
            SecurityGroupFolder = decoder.ReadStringArray("SecurityGroupFolder");
            KeyLifetime = decoder.ReadDouble("KeyLifetime");
            SecurityPolicyUri = decoder.ReadString("SecurityPolicyUri");
            MaxFutureKeyCount = decoder.ReadUInt32("MaxFutureKeyCount");
            MaxPastKeyCount = decoder.ReadUInt32("MaxPastKeyCount");
            SecurityGroupId = decoder.ReadString("SecurityGroupId");
            RolePermissions = (RolePermissionTypeCollection)decoder.ReadEncodeableArray("RolePermissions", typeof(RolePermissionType));
            GroupProperties = (KeyValuePairCollection)decoder.ReadEncodeableArray("GroupProperties", typeof(KeyValuePair));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            SecurityGroupDataType value = encodeable as SecurityGroupDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_name, value.m_name)) return false;
            if (!Utils.IsEqual(m_securityGroupFolder, value.m_securityGroupFolder)) return false;
            if (!Utils.IsEqual(m_keyLifetime, value.m_keyLifetime)) return false;
            if (!Utils.IsEqual(m_securityPolicyUri, value.m_securityPolicyUri)) return false;
            if (!Utils.IsEqual(m_maxFutureKeyCount, value.m_maxFutureKeyCount)) return false;
            if (!Utils.IsEqual(m_maxPastKeyCount, value.m_maxPastKeyCount)) return false;
            if (!Utils.IsEqual(m_securityGroupId, value.m_securityGroupId)) return false;
            if (!Utils.IsEqual(m_rolePermissions, value.m_rolePermissions)) return false;
            if (!Utils.IsEqual(m_groupProperties, value.m_groupProperties)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (SecurityGroupDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            SecurityGroupDataType clone = (SecurityGroupDataType)base.MemberwiseClone();

            clone.m_name = (string)Utils.Clone(this.m_name);
            clone.m_securityGroupFolder = (StringCollection)Utils.Clone(this.m_securityGroupFolder);
            clone.m_keyLifetime = (double)Utils.Clone(this.m_keyLifetime);
            clone.m_securityPolicyUri = (string)Utils.Clone(this.m_securityPolicyUri);
            clone.m_maxFutureKeyCount = (uint)Utils.Clone(this.m_maxFutureKeyCount);
            clone.m_maxPastKeyCount = (uint)Utils.Clone(this.m_maxPastKeyCount);
            clone.m_securityGroupId = (string)Utils.Clone(this.m_securityGroupId);
            clone.m_rolePermissions = (RolePermissionTypeCollection)Utils.Clone(this.m_rolePermissions);
            clone.m_groupProperties = (KeyValuePairCollection)Utils.Clone(this.m_groupProperties);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_name;
        private StringCollection m_securityGroupFolder;
        private double m_keyLifetime;
        private string m_securityPolicyUri;
        private uint m_maxFutureKeyCount;
        private uint m_maxPastKeyCount;
        private string m_securityGroupId;
        private RolePermissionTypeCollection m_rolePermissions;
        private KeyValuePairCollection m_groupProperties;
        #endregion
    }

    #region SecurityGroupDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfSecurityGroupDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "SecurityGroupDataType")]
    #if !NET_STANDARD
    public partial class SecurityGroupDataTypeCollection : List<SecurityGroupDataType>, ICloneable
    #else
    public partial class SecurityGroupDataTypeCollection : List<SecurityGroupDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public SecurityGroupDataTypeCollection() {}

        /// <remarks />
        public SecurityGroupDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public SecurityGroupDataTypeCollection(IEnumerable<SecurityGroupDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator SecurityGroupDataTypeCollection(SecurityGroupDataType[] values)
        {
            if (values != null)
            {
                return new SecurityGroupDataTypeCollection(values);
            }

            return new SecurityGroupDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator SecurityGroupDataType[](SecurityGroupDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (SecurityGroupDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            SecurityGroupDataTypeCollection clone = new SecurityGroupDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((SecurityGroupDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PubSubKeyPushTargetDataType Class
    #if (!OPCUA_EXCLUDE_PubSubKeyPushTargetDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class PubSubKeyPushTargetDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public PubSubKeyPushTargetDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_applicationUri = null;
            m_pushTargetFolder = new StringCollection();
            m_endpointUrl = null;
            m_securityPolicyUri = null;
            m_userTokenType = new UserTokenPolicy();
            m_requestedKeyCount = (ushort)0;
            m_retryInterval = (double)0;
            m_pushTargetProperties = new KeyValuePairCollection();
            m_securityGroups = new StringCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "ApplicationUri", IsRequired = false, Order = 1)]
        public string ApplicationUri
        {
            get { return m_applicationUri;  }
            set { m_applicationUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "PushTargetFolder", IsRequired = false, Order = 2)]
        public StringCollection PushTargetFolder
        {
            get
            {
                return m_pushTargetFolder;
            }

            set
            {
                m_pushTargetFolder = value;

                if (value == null)
                {
                    m_pushTargetFolder = new StringCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "EndpointUrl", IsRequired = false, Order = 3)]
        public string EndpointUrl
        {
            get { return m_endpointUrl;  }
            set { m_endpointUrl = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityPolicyUri", IsRequired = false, Order = 4)]
        public string SecurityPolicyUri
        {
            get { return m_securityPolicyUri;  }
            set { m_securityPolicyUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "UserTokenType", IsRequired = false, Order = 5)]
        public UserTokenPolicy UserTokenType
        {
            get
            {
                return m_userTokenType;
            }

            set
            {
                m_userTokenType = value;

                if (value == null)
                {
                    m_userTokenType = new UserTokenPolicy();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "RequestedKeyCount", IsRequired = false, Order = 6)]
        public ushort RequestedKeyCount
        {
            get { return m_requestedKeyCount;  }
            set { m_requestedKeyCount = value; }
        }

        /// <remarks />
        [DataMember(Name = "RetryInterval", IsRequired = false, Order = 7)]
        public double RetryInterval
        {
            get { return m_retryInterval;  }
            set { m_retryInterval = value; }
        }

        /// <remarks />
        [DataMember(Name = "PushTargetProperties", IsRequired = false, Order = 8)]
        public KeyValuePairCollection PushTargetProperties
        {
            get
            {
                return m_pushTargetProperties;
            }

            set
            {
                m_pushTargetProperties = value;

                if (value == null)
                {
                    m_pushTargetProperties = new KeyValuePairCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "SecurityGroups", IsRequired = false, Order = 9)]
        public StringCollection SecurityGroups
        {
            get
            {
                return m_securityGroups;
            }

            set
            {
                m_securityGroups = value;

                if (value == null)
                {
                    m_securityGroups = new StringCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.PubSubKeyPushTargetDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.PubSubKeyPushTargetDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.PubSubKeyPushTargetDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.PubSubKeyPushTargetDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("ApplicationUri", ApplicationUri);
            encoder.WriteStringArray("PushTargetFolder", PushTargetFolder);
            encoder.WriteString("EndpointUrl", EndpointUrl);
            encoder.WriteString("SecurityPolicyUri", SecurityPolicyUri);
            encoder.WriteEncodeable("UserTokenType", UserTokenType, typeof(UserTokenPolicy));
            encoder.WriteUInt16("RequestedKeyCount", RequestedKeyCount);
            encoder.WriteDouble("RetryInterval", RetryInterval);
            encoder.WriteEncodeableArray("PushTargetProperties", PushTargetProperties.ToArray(), typeof(KeyValuePair));
            encoder.WriteStringArray("SecurityGroups", SecurityGroups);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ApplicationUri = decoder.ReadString("ApplicationUri");
            PushTargetFolder = decoder.ReadStringArray("PushTargetFolder");
            EndpointUrl = decoder.ReadString("EndpointUrl");
            SecurityPolicyUri = decoder.ReadString("SecurityPolicyUri");
            UserTokenType = (UserTokenPolicy)decoder.ReadEncodeable("UserTokenType", typeof(UserTokenPolicy));
            RequestedKeyCount = decoder.ReadUInt16("RequestedKeyCount");
            RetryInterval = decoder.ReadDouble("RetryInterval");
            PushTargetProperties = (KeyValuePairCollection)decoder.ReadEncodeableArray("PushTargetProperties", typeof(KeyValuePair));
            SecurityGroups = decoder.ReadStringArray("SecurityGroups");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            PubSubKeyPushTargetDataType value = encodeable as PubSubKeyPushTargetDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_applicationUri, value.m_applicationUri)) return false;
            if (!Utils.IsEqual(m_pushTargetFolder, value.m_pushTargetFolder)) return false;
            if (!Utils.IsEqual(m_endpointUrl, value.m_endpointUrl)) return false;
            if (!Utils.IsEqual(m_securityPolicyUri, value.m_securityPolicyUri)) return false;
            if (!Utils.IsEqual(m_userTokenType, value.m_userTokenType)) return false;
            if (!Utils.IsEqual(m_requestedKeyCount, value.m_requestedKeyCount)) return false;
            if (!Utils.IsEqual(m_retryInterval, value.m_retryInterval)) return false;
            if (!Utils.IsEqual(m_pushTargetProperties, value.m_pushTargetProperties)) return false;
            if (!Utils.IsEqual(m_securityGroups, value.m_securityGroups)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (PubSubKeyPushTargetDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PubSubKeyPushTargetDataType clone = (PubSubKeyPushTargetDataType)base.MemberwiseClone();

            clone.m_applicationUri = (string)Utils.Clone(this.m_applicationUri);
            clone.m_pushTargetFolder = (StringCollection)Utils.Clone(this.m_pushTargetFolder);
            clone.m_endpointUrl = (string)Utils.Clone(this.m_endpointUrl);
            clone.m_securityPolicyUri = (string)Utils.Clone(this.m_securityPolicyUri);
            clone.m_userTokenType = (UserTokenPolicy)Utils.Clone(this.m_userTokenType);
            clone.m_requestedKeyCount = (ushort)Utils.Clone(this.m_requestedKeyCount);
            clone.m_retryInterval = (double)Utils.Clone(this.m_retryInterval);
            clone.m_pushTargetProperties = (KeyValuePairCollection)Utils.Clone(this.m_pushTargetProperties);
            clone.m_securityGroups = (StringCollection)Utils.Clone(this.m_securityGroups);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_applicationUri;
        private StringCollection m_pushTargetFolder;
        private string m_endpointUrl;
        private string m_securityPolicyUri;
        private UserTokenPolicy m_userTokenType;
        private ushort m_requestedKeyCount;
        private double m_retryInterval;
        private KeyValuePairCollection m_pushTargetProperties;
        private StringCollection m_securityGroups;
        #endregion
    }

    #region PubSubKeyPushTargetDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPubSubKeyPushTargetDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PubSubKeyPushTargetDataType")]
    #if !NET_STANDARD
    public partial class PubSubKeyPushTargetDataTypeCollection : List<PubSubKeyPushTargetDataType>, ICloneable
    #else
    public partial class PubSubKeyPushTargetDataTypeCollection : List<PubSubKeyPushTargetDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public PubSubKeyPushTargetDataTypeCollection() {}

        /// <remarks />
        public PubSubKeyPushTargetDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public PubSubKeyPushTargetDataTypeCollection(IEnumerable<PubSubKeyPushTargetDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator PubSubKeyPushTargetDataTypeCollection(PubSubKeyPushTargetDataType[] values)
        {
            if (values != null)
            {
                return new PubSubKeyPushTargetDataTypeCollection(values);
            }

            return new PubSubKeyPushTargetDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator PubSubKeyPushTargetDataType[](PubSubKeyPushTargetDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (PubSubKeyPushTargetDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PubSubKeyPushTargetDataTypeCollection clone = new PubSubKeyPushTargetDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PubSubKeyPushTargetDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PubSubConfiguration2DataType Class
    #if (!OPCUA_EXCLUDE_PubSubConfiguration2DataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class PubSubConfiguration2DataType : PubSubConfigurationDataType
    {
        #region Constructors
        /// <remarks />
        public PubSubConfiguration2DataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_subscribedDataSets = new StandaloneSubscribedDataSetDataTypeCollection();
            m_dataSetClasses = new DataSetMetaDataTypeCollection();
            m_defaultSecurityKeyServices = new EndpointDescriptionCollection();
            m_securityGroups = new SecurityGroupDataTypeCollection();
            m_pubSubKeyPushTargets = new PubSubKeyPushTargetDataTypeCollection();
            m_configurationVersion = (uint)0;
            m_configurationProperties = new KeyValuePairCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "SubscribedDataSets", IsRequired = false, Order = 1)]
        public StandaloneSubscribedDataSetDataTypeCollection SubscribedDataSets
        {
            get
            {
                return m_subscribedDataSets;
            }

            set
            {
                m_subscribedDataSets = value;

                if (value == null)
                {
                    m_subscribedDataSets = new StandaloneSubscribedDataSetDataTypeCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "DataSetClasses", IsRequired = false, Order = 2)]
        public DataSetMetaDataTypeCollection DataSetClasses
        {
            get
            {
                return m_dataSetClasses;
            }

            set
            {
                m_dataSetClasses = value;

                if (value == null)
                {
                    m_dataSetClasses = new DataSetMetaDataTypeCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "DefaultSecurityKeyServices", IsRequired = false, Order = 3)]
        public EndpointDescriptionCollection DefaultSecurityKeyServices
        {
            get
            {
                return m_defaultSecurityKeyServices;
            }

            set
            {
                m_defaultSecurityKeyServices = value;

                if (value == null)
                {
                    m_defaultSecurityKeyServices = new EndpointDescriptionCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "SecurityGroups", IsRequired = false, Order = 4)]
        public SecurityGroupDataTypeCollection SecurityGroups
        {
            get
            {
                return m_securityGroups;
            }

            set
            {
                m_securityGroups = value;

                if (value == null)
                {
                    m_securityGroups = new SecurityGroupDataTypeCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "PubSubKeyPushTargets", IsRequired = false, Order = 5)]
        public PubSubKeyPushTargetDataTypeCollection PubSubKeyPushTargets
        {
            get
            {
                return m_pubSubKeyPushTargets;
            }

            set
            {
                m_pubSubKeyPushTargets = value;

                if (value == null)
                {
                    m_pubSubKeyPushTargets = new PubSubKeyPushTargetDataTypeCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "ConfigurationVersion", IsRequired = false, Order = 6)]
        public uint ConfigurationVersion
        {
            get { return m_configurationVersion;  }
            set { m_configurationVersion = value; }
        }

        /// <remarks />
        [DataMember(Name = "ConfigurationProperties", IsRequired = false, Order = 7)]
        public KeyValuePairCollection ConfigurationProperties
        {
            get
            {
                return m_configurationProperties;
            }

            set
            {
                m_configurationProperties = value;

                if (value == null)
                {
                    m_configurationProperties = new KeyValuePairCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.PubSubConfiguration2DataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.PubSubConfiguration2DataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.PubSubConfiguration2DataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.PubSubConfiguration2DataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeableArray("SubscribedDataSets", SubscribedDataSets.ToArray(), typeof(StandaloneSubscribedDataSetDataType));
            encoder.WriteEncodeableArray("DataSetClasses", DataSetClasses.ToArray(), typeof(DataSetMetaDataType));
            encoder.WriteEncodeableArray("DefaultSecurityKeyServices", DefaultSecurityKeyServices.ToArray(), typeof(EndpointDescription));
            encoder.WriteEncodeableArray("SecurityGroups", SecurityGroups.ToArray(), typeof(SecurityGroupDataType));
            encoder.WriteEncodeableArray("PubSubKeyPushTargets", PubSubKeyPushTargets.ToArray(), typeof(PubSubKeyPushTargetDataType));
            encoder.WriteUInt32("ConfigurationVersion", ConfigurationVersion);
            encoder.WriteEncodeableArray("ConfigurationProperties", ConfigurationProperties.ToArray(), typeof(KeyValuePair));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            SubscribedDataSets = (StandaloneSubscribedDataSetDataTypeCollection)decoder.ReadEncodeableArray("SubscribedDataSets", typeof(StandaloneSubscribedDataSetDataType));
            DataSetClasses = (DataSetMetaDataTypeCollection)decoder.ReadEncodeableArray("DataSetClasses", typeof(DataSetMetaDataType));
            DefaultSecurityKeyServices = (EndpointDescriptionCollection)decoder.ReadEncodeableArray("DefaultSecurityKeyServices", typeof(EndpointDescription));
            SecurityGroups = (SecurityGroupDataTypeCollection)decoder.ReadEncodeableArray("SecurityGroups", typeof(SecurityGroupDataType));
            PubSubKeyPushTargets = (PubSubKeyPushTargetDataTypeCollection)decoder.ReadEncodeableArray("PubSubKeyPushTargets", typeof(PubSubKeyPushTargetDataType));
            ConfigurationVersion = decoder.ReadUInt32("ConfigurationVersion");
            ConfigurationProperties = (KeyValuePairCollection)decoder.ReadEncodeableArray("ConfigurationProperties", typeof(KeyValuePair));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            PubSubConfiguration2DataType value = encodeable as PubSubConfiguration2DataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_subscribedDataSets, value.m_subscribedDataSets)) return false;
            if (!Utils.IsEqual(m_dataSetClasses, value.m_dataSetClasses)) return false;
            if (!Utils.IsEqual(m_defaultSecurityKeyServices, value.m_defaultSecurityKeyServices)) return false;
            if (!Utils.IsEqual(m_securityGroups, value.m_securityGroups)) return false;
            if (!Utils.IsEqual(m_pubSubKeyPushTargets, value.m_pubSubKeyPushTargets)) return false;
            if (!Utils.IsEqual(m_configurationVersion, value.m_configurationVersion)) return false;
            if (!Utils.IsEqual(m_configurationProperties, value.m_configurationProperties)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (PubSubConfiguration2DataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PubSubConfiguration2DataType clone = (PubSubConfiguration2DataType)base.MemberwiseClone();

            clone.m_subscribedDataSets = (StandaloneSubscribedDataSetDataTypeCollection)Utils.Clone(this.m_subscribedDataSets);
            clone.m_dataSetClasses = (DataSetMetaDataTypeCollection)Utils.Clone(this.m_dataSetClasses);
            clone.m_defaultSecurityKeyServices = (EndpointDescriptionCollection)Utils.Clone(this.m_defaultSecurityKeyServices);
            clone.m_securityGroups = (SecurityGroupDataTypeCollection)Utils.Clone(this.m_securityGroups);
            clone.m_pubSubKeyPushTargets = (PubSubKeyPushTargetDataTypeCollection)Utils.Clone(this.m_pubSubKeyPushTargets);
            clone.m_configurationVersion = (uint)Utils.Clone(this.m_configurationVersion);
            clone.m_configurationProperties = (KeyValuePairCollection)Utils.Clone(this.m_configurationProperties);

            return clone;
        }
        #endregion

        #region Private Fields
        private StandaloneSubscribedDataSetDataTypeCollection m_subscribedDataSets;
        private DataSetMetaDataTypeCollection m_dataSetClasses;
        private EndpointDescriptionCollection m_defaultSecurityKeyServices;
        private SecurityGroupDataTypeCollection m_securityGroups;
        private PubSubKeyPushTargetDataTypeCollection m_pubSubKeyPushTargets;
        private uint m_configurationVersion;
        private KeyValuePairCollection m_configurationProperties;
        #endregion
    }

    #region PubSubConfiguration2DataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPubSubConfiguration2DataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PubSubConfiguration2DataType")]
    #if !NET_STANDARD
    public partial class PubSubConfiguration2DataTypeCollection : List<PubSubConfiguration2DataType>, ICloneable
    #else
    public partial class PubSubConfiguration2DataTypeCollection : List<PubSubConfiguration2DataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public PubSubConfiguration2DataTypeCollection() {}

        /// <remarks />
        public PubSubConfiguration2DataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public PubSubConfiguration2DataTypeCollection(IEnumerable<PubSubConfiguration2DataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator PubSubConfiguration2DataTypeCollection(PubSubConfiguration2DataType[] values)
        {
            if (values != null)
            {
                return new PubSubConfiguration2DataTypeCollection(values);
            }

            return new PubSubConfiguration2DataTypeCollection();
        }

        /// <remarks />
        public static explicit operator PubSubConfiguration2DataType[](PubSubConfiguration2DataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (PubSubConfiguration2DataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PubSubConfiguration2DataTypeCollection clone = new PubSubConfiguration2DataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PubSubConfiguration2DataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DataSetOrderingType Enumeration
    #if (!OPCUA_EXCLUDE_DataSetOrderingType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum DataSetOrderingType
    {
        /// <remarks />
        [EnumMember(Value = "Undefined_0")]
        Undefined = 0,

        /// <remarks />
        [EnumMember(Value = "AscendingWriterId_1")]
        AscendingWriterId = 1,

        /// <remarks />
        [EnumMember(Value = "AscendingWriterIdSingle_2")]
        AscendingWriterIdSingle = 2,
    }

    #region DataSetOrderingTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDataSetOrderingType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DataSetOrderingType")]
    #if !NET_STANDARD
    public partial class DataSetOrderingTypeCollection : List<DataSetOrderingType>, ICloneable
    #else
    public partial class DataSetOrderingTypeCollection : List<DataSetOrderingType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public DataSetOrderingTypeCollection() {}

        /// <remarks />
        public DataSetOrderingTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public DataSetOrderingTypeCollection(IEnumerable<DataSetOrderingType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator DataSetOrderingTypeCollection(DataSetOrderingType[] values)
        {
            if (values != null)
            {
                return new DataSetOrderingTypeCollection(values);
            }

            return new DataSetOrderingTypeCollection();
        }

        /// <remarks />
        public static explicit operator DataSetOrderingType[](DataSetOrderingTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (DataSetOrderingTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataSetOrderingTypeCollection clone = new DataSetOrderingTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DataSetOrderingType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region UadpNetworkMessageContentMask Enumeration
    #if (!OPCUA_EXCLUDE_UadpNetworkMessageContentMask)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum UadpNetworkMessageContentMask : UInt32
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "PublisherId_1")]
        PublisherId = 1,

        /// <remarks />
        [EnumMember(Value = "GroupHeader_2")]
        GroupHeader = 2,

        /// <remarks />
        [EnumMember(Value = "WriterGroupId_4")]
        WriterGroupId = 4,

        /// <remarks />
        [EnumMember(Value = "GroupVersion_8")]
        GroupVersion = 8,

        /// <remarks />
        [EnumMember(Value = "NetworkMessageNumber_16")]
        NetworkMessageNumber = 16,

        /// <remarks />
        [EnumMember(Value = "SequenceNumber_32")]
        SequenceNumber = 32,

        /// <remarks />
        [EnumMember(Value = "PayloadHeader_64")]
        PayloadHeader = 64,

        /// <remarks />
        [EnumMember(Value = "Timestamp_128")]
        Timestamp = 128,

        /// <remarks />
        [EnumMember(Value = "PicoSeconds_256")]
        PicoSeconds = 256,

        /// <remarks />
        [EnumMember(Value = "DataSetClassId_512")]
        DataSetClassId = 512,

        /// <remarks />
        [EnumMember(Value = "PromotedFields_1024")]
        PromotedFields = 1024,
    }

    #region UadpNetworkMessageContentMaskCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfUadpNetworkMessageContentMask", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "UadpNetworkMessageContentMask")]
    #if !NET_STANDARD
    public partial class UadpNetworkMessageContentMaskCollection : List<UadpNetworkMessageContentMask>, ICloneable
    #else
    public partial class UadpNetworkMessageContentMaskCollection : List<UadpNetworkMessageContentMask>
    #endif
    {
        #region Constructors
        /// <remarks />
        public UadpNetworkMessageContentMaskCollection() {}

        /// <remarks />
        public UadpNetworkMessageContentMaskCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public UadpNetworkMessageContentMaskCollection(IEnumerable<UadpNetworkMessageContentMask> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator UadpNetworkMessageContentMaskCollection(UadpNetworkMessageContentMask[] values)
        {
            if (values != null)
            {
                return new UadpNetworkMessageContentMaskCollection(values);
            }

            return new UadpNetworkMessageContentMaskCollection();
        }

        /// <remarks />
        public static explicit operator UadpNetworkMessageContentMask[](UadpNetworkMessageContentMaskCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (UadpNetworkMessageContentMaskCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UadpNetworkMessageContentMaskCollection clone = new UadpNetworkMessageContentMaskCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((UadpNetworkMessageContentMask)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region UadpWriterGroupMessageDataType Class
    #if (!OPCUA_EXCLUDE_UadpWriterGroupMessageDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class UadpWriterGroupMessageDataType : WriterGroupMessageDataType
    {
        #region Constructors
        /// <remarks />
        public UadpWriterGroupMessageDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_groupVersion = (uint)0;
            m_dataSetOrdering = DataSetOrderingType.Undefined;
            m_networkMessageContentMask = 0;
            m_samplingOffset = (double)0;
            m_publishingOffset = new DoubleCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "GroupVersion", IsRequired = false, Order = 1)]
        public uint GroupVersion
        {
            get { return m_groupVersion;  }
            set { m_groupVersion = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetOrdering", IsRequired = false, Order = 2)]
        public DataSetOrderingType DataSetOrdering
        {
            get { return m_dataSetOrdering;  }
            set { m_dataSetOrdering = value; }
        }

        /// <remarks />
        [DataMember(Name = "NetworkMessageContentMask", IsRequired = false, Order = 3)]
        public uint NetworkMessageContentMask
        {
            get { return m_networkMessageContentMask;  }
            set { m_networkMessageContentMask = value; }
        }

        /// <remarks />
        [DataMember(Name = "SamplingOffset", IsRequired = false, Order = 4)]
        public double SamplingOffset
        {
            get { return m_samplingOffset;  }
            set { m_samplingOffset = value; }
        }

        /// <remarks />
        [DataMember(Name = "PublishingOffset", IsRequired = false, Order = 5)]
        public DoubleCollection PublishingOffset
        {
            get
            {
                return m_publishingOffset;
            }

            set
            {
                m_publishingOffset = value;

                if (value == null)
                {
                    m_publishingOffset = new DoubleCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.UadpWriterGroupMessageDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.UadpWriterGroupMessageDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.UadpWriterGroupMessageDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.UadpWriterGroupMessageDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt32("GroupVersion", GroupVersion);
            encoder.WriteEnumerated("DataSetOrdering", DataSetOrdering);
            encoder.WriteUInt32("NetworkMessageContentMask", NetworkMessageContentMask);
            encoder.WriteDouble("SamplingOffset", SamplingOffset);
            encoder.WriteDoubleArray("PublishingOffset", PublishingOffset);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            GroupVersion = decoder.ReadUInt32("GroupVersion");
            DataSetOrdering = (DataSetOrderingType)decoder.ReadEnumerated("DataSetOrdering", typeof(DataSetOrderingType));
            NetworkMessageContentMask = decoder.ReadUInt32("NetworkMessageContentMask");
            SamplingOffset = decoder.ReadDouble("SamplingOffset");
            PublishingOffset = decoder.ReadDoubleArray("PublishingOffset");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            UadpWriterGroupMessageDataType value = encodeable as UadpWriterGroupMessageDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_groupVersion, value.m_groupVersion)) return false;
            if (!Utils.IsEqual(m_dataSetOrdering, value.m_dataSetOrdering)) return false;
            if (!Utils.IsEqual(m_networkMessageContentMask, value.m_networkMessageContentMask)) return false;
            if (!Utils.IsEqual(m_samplingOffset, value.m_samplingOffset)) return false;
            if (!Utils.IsEqual(m_publishingOffset, value.m_publishingOffset)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (UadpWriterGroupMessageDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UadpWriterGroupMessageDataType clone = (UadpWriterGroupMessageDataType)base.MemberwiseClone();

            clone.m_groupVersion = (uint)Utils.Clone(this.m_groupVersion);
            clone.m_dataSetOrdering = (DataSetOrderingType)Utils.Clone(this.m_dataSetOrdering);
            clone.m_networkMessageContentMask = (uint)Utils.Clone(this.m_networkMessageContentMask);
            clone.m_samplingOffset = (double)Utils.Clone(this.m_samplingOffset);
            clone.m_publishingOffset = (DoubleCollection)Utils.Clone(this.m_publishingOffset);

            return clone;
        }
        #endregion

        #region Private Fields
        private uint m_groupVersion;
        private DataSetOrderingType m_dataSetOrdering;
        private uint m_networkMessageContentMask;
        private double m_samplingOffset;
        private DoubleCollection m_publishingOffset;
        #endregion
    }

    #region UadpWriterGroupMessageDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfUadpWriterGroupMessageDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "UadpWriterGroupMessageDataType")]
    #if !NET_STANDARD
    public partial class UadpWriterGroupMessageDataTypeCollection : List<UadpWriterGroupMessageDataType>, ICloneable
    #else
    public partial class UadpWriterGroupMessageDataTypeCollection : List<UadpWriterGroupMessageDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public UadpWriterGroupMessageDataTypeCollection() {}

        /// <remarks />
        public UadpWriterGroupMessageDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public UadpWriterGroupMessageDataTypeCollection(IEnumerable<UadpWriterGroupMessageDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator UadpWriterGroupMessageDataTypeCollection(UadpWriterGroupMessageDataType[] values)
        {
            if (values != null)
            {
                return new UadpWriterGroupMessageDataTypeCollection(values);
            }

            return new UadpWriterGroupMessageDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator UadpWriterGroupMessageDataType[](UadpWriterGroupMessageDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (UadpWriterGroupMessageDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UadpWriterGroupMessageDataTypeCollection clone = new UadpWriterGroupMessageDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((UadpWriterGroupMessageDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region UadpDataSetMessageContentMask Enumeration
    #if (!OPCUA_EXCLUDE_UadpDataSetMessageContentMask)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum UadpDataSetMessageContentMask : UInt32
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "Timestamp_1")]
        Timestamp = 1,

        /// <remarks />
        [EnumMember(Value = "PicoSeconds_2")]
        PicoSeconds = 2,

        /// <remarks />
        [EnumMember(Value = "Status_4")]
        Status = 4,

        /// <remarks />
        [EnumMember(Value = "MajorVersion_8")]
        MajorVersion = 8,

        /// <remarks />
        [EnumMember(Value = "MinorVersion_16")]
        MinorVersion = 16,

        /// <remarks />
        [EnumMember(Value = "SequenceNumber_32")]
        SequenceNumber = 32,
    }

    #region UadpDataSetMessageContentMaskCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfUadpDataSetMessageContentMask", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "UadpDataSetMessageContentMask")]
    #if !NET_STANDARD
    public partial class UadpDataSetMessageContentMaskCollection : List<UadpDataSetMessageContentMask>, ICloneable
    #else
    public partial class UadpDataSetMessageContentMaskCollection : List<UadpDataSetMessageContentMask>
    #endif
    {
        #region Constructors
        /// <remarks />
        public UadpDataSetMessageContentMaskCollection() {}

        /// <remarks />
        public UadpDataSetMessageContentMaskCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public UadpDataSetMessageContentMaskCollection(IEnumerable<UadpDataSetMessageContentMask> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator UadpDataSetMessageContentMaskCollection(UadpDataSetMessageContentMask[] values)
        {
            if (values != null)
            {
                return new UadpDataSetMessageContentMaskCollection(values);
            }

            return new UadpDataSetMessageContentMaskCollection();
        }

        /// <remarks />
        public static explicit operator UadpDataSetMessageContentMask[](UadpDataSetMessageContentMaskCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (UadpDataSetMessageContentMaskCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UadpDataSetMessageContentMaskCollection clone = new UadpDataSetMessageContentMaskCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((UadpDataSetMessageContentMask)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region UadpDataSetWriterMessageDataType Class
    #if (!OPCUA_EXCLUDE_UadpDataSetWriterMessageDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class UadpDataSetWriterMessageDataType : DataSetWriterMessageDataType
    {
        #region Constructors
        /// <remarks />
        public UadpDataSetWriterMessageDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_dataSetMessageContentMask = 0;
            m_configuredSize = (ushort)0;
            m_networkMessageNumber = (ushort)0;
            m_dataSetOffset = (ushort)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "DataSetMessageContentMask", IsRequired = false, Order = 1)]
        public uint DataSetMessageContentMask
        {
            get { return m_dataSetMessageContentMask;  }
            set { m_dataSetMessageContentMask = value; }
        }

        /// <remarks />
        [DataMember(Name = "ConfiguredSize", IsRequired = false, Order = 2)]
        public ushort ConfiguredSize
        {
            get { return m_configuredSize;  }
            set { m_configuredSize = value; }
        }

        /// <remarks />
        [DataMember(Name = "NetworkMessageNumber", IsRequired = false, Order = 3)]
        public ushort NetworkMessageNumber
        {
            get { return m_networkMessageNumber;  }
            set { m_networkMessageNumber = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetOffset", IsRequired = false, Order = 4)]
        public ushort DataSetOffset
        {
            get { return m_dataSetOffset;  }
            set { m_dataSetOffset = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.UadpDataSetWriterMessageDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.UadpDataSetWriterMessageDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.UadpDataSetWriterMessageDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.UadpDataSetWriterMessageDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt32("DataSetMessageContentMask", DataSetMessageContentMask);
            encoder.WriteUInt16("ConfiguredSize", ConfiguredSize);
            encoder.WriteUInt16("NetworkMessageNumber", NetworkMessageNumber);
            encoder.WriteUInt16("DataSetOffset", DataSetOffset);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            DataSetMessageContentMask = decoder.ReadUInt32("DataSetMessageContentMask");
            ConfiguredSize = decoder.ReadUInt16("ConfiguredSize");
            NetworkMessageNumber = decoder.ReadUInt16("NetworkMessageNumber");
            DataSetOffset = decoder.ReadUInt16("DataSetOffset");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            UadpDataSetWriterMessageDataType value = encodeable as UadpDataSetWriterMessageDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_dataSetMessageContentMask, value.m_dataSetMessageContentMask)) return false;
            if (!Utils.IsEqual(m_configuredSize, value.m_configuredSize)) return false;
            if (!Utils.IsEqual(m_networkMessageNumber, value.m_networkMessageNumber)) return false;
            if (!Utils.IsEqual(m_dataSetOffset, value.m_dataSetOffset)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (UadpDataSetWriterMessageDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UadpDataSetWriterMessageDataType clone = (UadpDataSetWriterMessageDataType)base.MemberwiseClone();

            clone.m_dataSetMessageContentMask = (uint)Utils.Clone(this.m_dataSetMessageContentMask);
            clone.m_configuredSize = (ushort)Utils.Clone(this.m_configuredSize);
            clone.m_networkMessageNumber = (ushort)Utils.Clone(this.m_networkMessageNumber);
            clone.m_dataSetOffset = (ushort)Utils.Clone(this.m_dataSetOffset);

            return clone;
        }
        #endregion

        #region Private Fields
        private uint m_dataSetMessageContentMask;
        private ushort m_configuredSize;
        private ushort m_networkMessageNumber;
        private ushort m_dataSetOffset;
        #endregion
    }

    #region UadpDataSetWriterMessageDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfUadpDataSetWriterMessageDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "UadpDataSetWriterMessageDataType")]
    #if !NET_STANDARD
    public partial class UadpDataSetWriterMessageDataTypeCollection : List<UadpDataSetWriterMessageDataType>, ICloneable
    #else
    public partial class UadpDataSetWriterMessageDataTypeCollection : List<UadpDataSetWriterMessageDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public UadpDataSetWriterMessageDataTypeCollection() {}

        /// <remarks />
        public UadpDataSetWriterMessageDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public UadpDataSetWriterMessageDataTypeCollection(IEnumerable<UadpDataSetWriterMessageDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator UadpDataSetWriterMessageDataTypeCollection(UadpDataSetWriterMessageDataType[] values)
        {
            if (values != null)
            {
                return new UadpDataSetWriterMessageDataTypeCollection(values);
            }

            return new UadpDataSetWriterMessageDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator UadpDataSetWriterMessageDataType[](UadpDataSetWriterMessageDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (UadpDataSetWriterMessageDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UadpDataSetWriterMessageDataTypeCollection clone = new UadpDataSetWriterMessageDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((UadpDataSetWriterMessageDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region UadpDataSetReaderMessageDataType Class
    #if (!OPCUA_EXCLUDE_UadpDataSetReaderMessageDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class UadpDataSetReaderMessageDataType : DataSetReaderMessageDataType
    {
        #region Constructors
        /// <remarks />
        public UadpDataSetReaderMessageDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_groupVersion = (uint)0;
            m_networkMessageNumber = (ushort)0;
            m_dataSetOffset = (ushort)0;
            m_dataSetClassId = Uuid.Empty;
            m_networkMessageContentMask = 0;
            m_dataSetMessageContentMask = 0;
            m_publishingInterval = (double)0;
            m_receiveOffset = (double)0;
            m_processingOffset = (double)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "GroupVersion", IsRequired = false, Order = 1)]
        public uint GroupVersion
        {
            get { return m_groupVersion;  }
            set { m_groupVersion = value; }
        }

        /// <remarks />
        [DataMember(Name = "NetworkMessageNumber", IsRequired = false, Order = 2)]
        public ushort NetworkMessageNumber
        {
            get { return m_networkMessageNumber;  }
            set { m_networkMessageNumber = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetOffset", IsRequired = false, Order = 3)]
        public ushort DataSetOffset
        {
            get { return m_dataSetOffset;  }
            set { m_dataSetOffset = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetClassId", IsRequired = false, Order = 4)]
        public Uuid DataSetClassId
        {
            get { return m_dataSetClassId;  }
            set { m_dataSetClassId = value; }
        }

        /// <remarks />
        [DataMember(Name = "NetworkMessageContentMask", IsRequired = false, Order = 5)]
        public uint NetworkMessageContentMask
        {
            get { return m_networkMessageContentMask;  }
            set { m_networkMessageContentMask = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetMessageContentMask", IsRequired = false, Order = 6)]
        public uint DataSetMessageContentMask
        {
            get { return m_dataSetMessageContentMask;  }
            set { m_dataSetMessageContentMask = value; }
        }

        /// <remarks />
        [DataMember(Name = "PublishingInterval", IsRequired = false, Order = 7)]
        public double PublishingInterval
        {
            get { return m_publishingInterval;  }
            set { m_publishingInterval = value; }
        }

        /// <remarks />
        [DataMember(Name = "ReceiveOffset", IsRequired = false, Order = 8)]
        public double ReceiveOffset
        {
            get { return m_receiveOffset;  }
            set { m_receiveOffset = value; }
        }

        /// <remarks />
        [DataMember(Name = "ProcessingOffset", IsRequired = false, Order = 9)]
        public double ProcessingOffset
        {
            get { return m_processingOffset;  }
            set { m_processingOffset = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.UadpDataSetReaderMessageDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.UadpDataSetReaderMessageDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.UadpDataSetReaderMessageDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.UadpDataSetReaderMessageDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt32("GroupVersion", GroupVersion);
            encoder.WriteUInt16("NetworkMessageNumber", NetworkMessageNumber);
            encoder.WriteUInt16("DataSetOffset", DataSetOffset);
            encoder.WriteGuid("DataSetClassId", DataSetClassId);
            encoder.WriteUInt32("NetworkMessageContentMask", NetworkMessageContentMask);
            encoder.WriteUInt32("DataSetMessageContentMask", DataSetMessageContentMask);
            encoder.WriteDouble("PublishingInterval", PublishingInterval);
            encoder.WriteDouble("ReceiveOffset", ReceiveOffset);
            encoder.WriteDouble("ProcessingOffset", ProcessingOffset);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            GroupVersion = decoder.ReadUInt32("GroupVersion");
            NetworkMessageNumber = decoder.ReadUInt16("NetworkMessageNumber");
            DataSetOffset = decoder.ReadUInt16("DataSetOffset");
            DataSetClassId = decoder.ReadGuid("DataSetClassId");
            NetworkMessageContentMask = decoder.ReadUInt32("NetworkMessageContentMask");
            DataSetMessageContentMask = decoder.ReadUInt32("DataSetMessageContentMask");
            PublishingInterval = decoder.ReadDouble("PublishingInterval");
            ReceiveOffset = decoder.ReadDouble("ReceiveOffset");
            ProcessingOffset = decoder.ReadDouble("ProcessingOffset");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            UadpDataSetReaderMessageDataType value = encodeable as UadpDataSetReaderMessageDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_groupVersion, value.m_groupVersion)) return false;
            if (!Utils.IsEqual(m_networkMessageNumber, value.m_networkMessageNumber)) return false;
            if (!Utils.IsEqual(m_dataSetOffset, value.m_dataSetOffset)) return false;
            if (!Utils.IsEqual(m_dataSetClassId, value.m_dataSetClassId)) return false;
            if (!Utils.IsEqual(m_networkMessageContentMask, value.m_networkMessageContentMask)) return false;
            if (!Utils.IsEqual(m_dataSetMessageContentMask, value.m_dataSetMessageContentMask)) return false;
            if (!Utils.IsEqual(m_publishingInterval, value.m_publishingInterval)) return false;
            if (!Utils.IsEqual(m_receiveOffset, value.m_receiveOffset)) return false;
            if (!Utils.IsEqual(m_processingOffset, value.m_processingOffset)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (UadpDataSetReaderMessageDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UadpDataSetReaderMessageDataType clone = (UadpDataSetReaderMessageDataType)base.MemberwiseClone();

            clone.m_groupVersion = (uint)Utils.Clone(this.m_groupVersion);
            clone.m_networkMessageNumber = (ushort)Utils.Clone(this.m_networkMessageNumber);
            clone.m_dataSetOffset = (ushort)Utils.Clone(this.m_dataSetOffset);
            clone.m_dataSetClassId = (Uuid)Utils.Clone(this.m_dataSetClassId);
            clone.m_networkMessageContentMask = (uint)Utils.Clone(this.m_networkMessageContentMask);
            clone.m_dataSetMessageContentMask = (uint)Utils.Clone(this.m_dataSetMessageContentMask);
            clone.m_publishingInterval = (double)Utils.Clone(this.m_publishingInterval);
            clone.m_receiveOffset = (double)Utils.Clone(this.m_receiveOffset);
            clone.m_processingOffset = (double)Utils.Clone(this.m_processingOffset);

            return clone;
        }
        #endregion

        #region Private Fields
        private uint m_groupVersion;
        private ushort m_networkMessageNumber;
        private ushort m_dataSetOffset;
        private Uuid m_dataSetClassId;
        private uint m_networkMessageContentMask;
        private uint m_dataSetMessageContentMask;
        private double m_publishingInterval;
        private double m_receiveOffset;
        private double m_processingOffset;
        #endregion
    }

    #region UadpDataSetReaderMessageDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfUadpDataSetReaderMessageDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "UadpDataSetReaderMessageDataType")]
    #if !NET_STANDARD
    public partial class UadpDataSetReaderMessageDataTypeCollection : List<UadpDataSetReaderMessageDataType>, ICloneable
    #else
    public partial class UadpDataSetReaderMessageDataTypeCollection : List<UadpDataSetReaderMessageDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public UadpDataSetReaderMessageDataTypeCollection() {}

        /// <remarks />
        public UadpDataSetReaderMessageDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public UadpDataSetReaderMessageDataTypeCollection(IEnumerable<UadpDataSetReaderMessageDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator UadpDataSetReaderMessageDataTypeCollection(UadpDataSetReaderMessageDataType[] values)
        {
            if (values != null)
            {
                return new UadpDataSetReaderMessageDataTypeCollection(values);
            }

            return new UadpDataSetReaderMessageDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator UadpDataSetReaderMessageDataType[](UadpDataSetReaderMessageDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (UadpDataSetReaderMessageDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UadpDataSetReaderMessageDataTypeCollection clone = new UadpDataSetReaderMessageDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((UadpDataSetReaderMessageDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region JsonNetworkMessageContentMask Enumeration
    #if (!OPCUA_EXCLUDE_JsonNetworkMessageContentMask)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum JsonNetworkMessageContentMask : UInt32
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "NetworkMessageHeader_1")]
        NetworkMessageHeader = 1,

        /// <remarks />
        [EnumMember(Value = "DataSetMessageHeader_2")]
        DataSetMessageHeader = 2,

        /// <remarks />
        [EnumMember(Value = "SingleDataSetMessage_4")]
        SingleDataSetMessage = 4,

        /// <remarks />
        [EnumMember(Value = "PublisherId_8")]
        PublisherId = 8,

        /// <remarks />
        [EnumMember(Value = "DataSetClassId_16")]
        DataSetClassId = 16,

        /// <remarks />
        [EnumMember(Value = "ReplyTo_32")]
        ReplyTo = 32,
    }

    #region JsonNetworkMessageContentMaskCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfJsonNetworkMessageContentMask", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "JsonNetworkMessageContentMask")]
    #if !NET_STANDARD
    public partial class JsonNetworkMessageContentMaskCollection : List<JsonNetworkMessageContentMask>, ICloneable
    #else
    public partial class JsonNetworkMessageContentMaskCollection : List<JsonNetworkMessageContentMask>
    #endif
    {
        #region Constructors
        /// <remarks />
        public JsonNetworkMessageContentMaskCollection() {}

        /// <remarks />
        public JsonNetworkMessageContentMaskCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public JsonNetworkMessageContentMaskCollection(IEnumerable<JsonNetworkMessageContentMask> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator JsonNetworkMessageContentMaskCollection(JsonNetworkMessageContentMask[] values)
        {
            if (values != null)
            {
                return new JsonNetworkMessageContentMaskCollection(values);
            }

            return new JsonNetworkMessageContentMaskCollection();
        }

        /// <remarks />
        public static explicit operator JsonNetworkMessageContentMask[](JsonNetworkMessageContentMaskCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (JsonNetworkMessageContentMaskCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            JsonNetworkMessageContentMaskCollection clone = new JsonNetworkMessageContentMaskCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((JsonNetworkMessageContentMask)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region JsonWriterGroupMessageDataType Class
    #if (!OPCUA_EXCLUDE_JsonWriterGroupMessageDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class JsonWriterGroupMessageDataType : WriterGroupMessageDataType
    {
        #region Constructors
        /// <remarks />
        public JsonWriterGroupMessageDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_networkMessageContentMask = 0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "NetworkMessageContentMask", IsRequired = false, Order = 1)]
        public uint NetworkMessageContentMask
        {
            get { return m_networkMessageContentMask;  }
            set { m_networkMessageContentMask = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.JsonWriterGroupMessageDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.JsonWriterGroupMessageDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.JsonWriterGroupMessageDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.JsonWriterGroupMessageDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt32("NetworkMessageContentMask", NetworkMessageContentMask);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            NetworkMessageContentMask = decoder.ReadUInt32("NetworkMessageContentMask");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            JsonWriterGroupMessageDataType value = encodeable as JsonWriterGroupMessageDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_networkMessageContentMask, value.m_networkMessageContentMask)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (JsonWriterGroupMessageDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            JsonWriterGroupMessageDataType clone = (JsonWriterGroupMessageDataType)base.MemberwiseClone();

            clone.m_networkMessageContentMask = (uint)Utils.Clone(this.m_networkMessageContentMask);

            return clone;
        }
        #endregion

        #region Private Fields
        private uint m_networkMessageContentMask;
        #endregion
    }

    #region JsonWriterGroupMessageDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfJsonWriterGroupMessageDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "JsonWriterGroupMessageDataType")]
    #if !NET_STANDARD
    public partial class JsonWriterGroupMessageDataTypeCollection : List<JsonWriterGroupMessageDataType>, ICloneable
    #else
    public partial class JsonWriterGroupMessageDataTypeCollection : List<JsonWriterGroupMessageDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public JsonWriterGroupMessageDataTypeCollection() {}

        /// <remarks />
        public JsonWriterGroupMessageDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public JsonWriterGroupMessageDataTypeCollection(IEnumerable<JsonWriterGroupMessageDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator JsonWriterGroupMessageDataTypeCollection(JsonWriterGroupMessageDataType[] values)
        {
            if (values != null)
            {
                return new JsonWriterGroupMessageDataTypeCollection(values);
            }

            return new JsonWriterGroupMessageDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator JsonWriterGroupMessageDataType[](JsonWriterGroupMessageDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (JsonWriterGroupMessageDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            JsonWriterGroupMessageDataTypeCollection clone = new JsonWriterGroupMessageDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((JsonWriterGroupMessageDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region JsonDataSetMessageContentMask Enumeration
    #if (!OPCUA_EXCLUDE_JsonDataSetMessageContentMask)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum JsonDataSetMessageContentMask : UInt32
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "DataSetWriterId_1")]
        DataSetWriterId = 1,

        /// <remarks />
        [EnumMember(Value = "MetaDataVersion_2")]
        MetaDataVersion = 2,

        /// <remarks />
        [EnumMember(Value = "SequenceNumber_4")]
        SequenceNumber = 4,

        /// <remarks />
        [EnumMember(Value = "Timestamp_8")]
        Timestamp = 8,

        /// <remarks />
        [EnumMember(Value = "Status_16")]
        Status = 16,

        /// <remarks />
        [EnumMember(Value = "MessageType_32")]
        MessageType = 32,

        /// <remarks />
        [EnumMember(Value = "DataSetWriterName_64")]
        DataSetWriterName = 64,

        /// <remarks />
        [EnumMember(Value = "ReversibleFieldEncoding_128")]
        ReversibleFieldEncoding = 128,
    }

    #region JsonDataSetMessageContentMaskCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfJsonDataSetMessageContentMask", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "JsonDataSetMessageContentMask")]
    #if !NET_STANDARD
    public partial class JsonDataSetMessageContentMaskCollection : List<JsonDataSetMessageContentMask>, ICloneable
    #else
    public partial class JsonDataSetMessageContentMaskCollection : List<JsonDataSetMessageContentMask>
    #endif
    {
        #region Constructors
        /// <remarks />
        public JsonDataSetMessageContentMaskCollection() {}

        /// <remarks />
        public JsonDataSetMessageContentMaskCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public JsonDataSetMessageContentMaskCollection(IEnumerable<JsonDataSetMessageContentMask> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator JsonDataSetMessageContentMaskCollection(JsonDataSetMessageContentMask[] values)
        {
            if (values != null)
            {
                return new JsonDataSetMessageContentMaskCollection(values);
            }

            return new JsonDataSetMessageContentMaskCollection();
        }

        /// <remarks />
        public static explicit operator JsonDataSetMessageContentMask[](JsonDataSetMessageContentMaskCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (JsonDataSetMessageContentMaskCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            JsonDataSetMessageContentMaskCollection clone = new JsonDataSetMessageContentMaskCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((JsonDataSetMessageContentMask)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region JsonDataSetWriterMessageDataType Class
    #if (!OPCUA_EXCLUDE_JsonDataSetWriterMessageDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class JsonDataSetWriterMessageDataType : DataSetWriterMessageDataType
    {
        #region Constructors
        /// <remarks />
        public JsonDataSetWriterMessageDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_dataSetMessageContentMask = 0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "DataSetMessageContentMask", IsRequired = false, Order = 1)]
        public uint DataSetMessageContentMask
        {
            get { return m_dataSetMessageContentMask;  }
            set { m_dataSetMessageContentMask = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.JsonDataSetWriterMessageDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.JsonDataSetWriterMessageDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.JsonDataSetWriterMessageDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.JsonDataSetWriterMessageDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt32("DataSetMessageContentMask", DataSetMessageContentMask);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            DataSetMessageContentMask = decoder.ReadUInt32("DataSetMessageContentMask");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            JsonDataSetWriterMessageDataType value = encodeable as JsonDataSetWriterMessageDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_dataSetMessageContentMask, value.m_dataSetMessageContentMask)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (JsonDataSetWriterMessageDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            JsonDataSetWriterMessageDataType clone = (JsonDataSetWriterMessageDataType)base.MemberwiseClone();

            clone.m_dataSetMessageContentMask = (uint)Utils.Clone(this.m_dataSetMessageContentMask);

            return clone;
        }
        #endregion

        #region Private Fields
        private uint m_dataSetMessageContentMask;
        #endregion
    }

    #region JsonDataSetWriterMessageDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfJsonDataSetWriterMessageDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "JsonDataSetWriterMessageDataType")]
    #if !NET_STANDARD
    public partial class JsonDataSetWriterMessageDataTypeCollection : List<JsonDataSetWriterMessageDataType>, ICloneable
    #else
    public partial class JsonDataSetWriterMessageDataTypeCollection : List<JsonDataSetWriterMessageDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public JsonDataSetWriterMessageDataTypeCollection() {}

        /// <remarks />
        public JsonDataSetWriterMessageDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public JsonDataSetWriterMessageDataTypeCollection(IEnumerable<JsonDataSetWriterMessageDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator JsonDataSetWriterMessageDataTypeCollection(JsonDataSetWriterMessageDataType[] values)
        {
            if (values != null)
            {
                return new JsonDataSetWriterMessageDataTypeCollection(values);
            }

            return new JsonDataSetWriterMessageDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator JsonDataSetWriterMessageDataType[](JsonDataSetWriterMessageDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (JsonDataSetWriterMessageDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            JsonDataSetWriterMessageDataTypeCollection clone = new JsonDataSetWriterMessageDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((JsonDataSetWriterMessageDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region JsonDataSetReaderMessageDataType Class
    #if (!OPCUA_EXCLUDE_JsonDataSetReaderMessageDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class JsonDataSetReaderMessageDataType : DataSetReaderMessageDataType
    {
        #region Constructors
        /// <remarks />
        public JsonDataSetReaderMessageDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_networkMessageContentMask = 0;
            m_dataSetMessageContentMask = 0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "NetworkMessageContentMask", IsRequired = false, Order = 1)]
        public uint NetworkMessageContentMask
        {
            get { return m_networkMessageContentMask;  }
            set { m_networkMessageContentMask = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataSetMessageContentMask", IsRequired = false, Order = 2)]
        public uint DataSetMessageContentMask
        {
            get { return m_dataSetMessageContentMask;  }
            set { m_dataSetMessageContentMask = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.JsonDataSetReaderMessageDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.JsonDataSetReaderMessageDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.JsonDataSetReaderMessageDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.JsonDataSetReaderMessageDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt32("NetworkMessageContentMask", NetworkMessageContentMask);
            encoder.WriteUInt32("DataSetMessageContentMask", DataSetMessageContentMask);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            NetworkMessageContentMask = decoder.ReadUInt32("NetworkMessageContentMask");
            DataSetMessageContentMask = decoder.ReadUInt32("DataSetMessageContentMask");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            JsonDataSetReaderMessageDataType value = encodeable as JsonDataSetReaderMessageDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_networkMessageContentMask, value.m_networkMessageContentMask)) return false;
            if (!Utils.IsEqual(m_dataSetMessageContentMask, value.m_dataSetMessageContentMask)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (JsonDataSetReaderMessageDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            JsonDataSetReaderMessageDataType clone = (JsonDataSetReaderMessageDataType)base.MemberwiseClone();

            clone.m_networkMessageContentMask = (uint)Utils.Clone(this.m_networkMessageContentMask);
            clone.m_dataSetMessageContentMask = (uint)Utils.Clone(this.m_dataSetMessageContentMask);

            return clone;
        }
        #endregion

        #region Private Fields
        private uint m_networkMessageContentMask;
        private uint m_dataSetMessageContentMask;
        #endregion
    }

    #region JsonDataSetReaderMessageDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfJsonDataSetReaderMessageDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "JsonDataSetReaderMessageDataType")]
    #if !NET_STANDARD
    public partial class JsonDataSetReaderMessageDataTypeCollection : List<JsonDataSetReaderMessageDataType>, ICloneable
    #else
    public partial class JsonDataSetReaderMessageDataTypeCollection : List<JsonDataSetReaderMessageDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public JsonDataSetReaderMessageDataTypeCollection() {}

        /// <remarks />
        public JsonDataSetReaderMessageDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public JsonDataSetReaderMessageDataTypeCollection(IEnumerable<JsonDataSetReaderMessageDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator JsonDataSetReaderMessageDataTypeCollection(JsonDataSetReaderMessageDataType[] values)
        {
            if (values != null)
            {
                return new JsonDataSetReaderMessageDataTypeCollection(values);
            }

            return new JsonDataSetReaderMessageDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator JsonDataSetReaderMessageDataType[](JsonDataSetReaderMessageDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (JsonDataSetReaderMessageDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            JsonDataSetReaderMessageDataTypeCollection clone = new JsonDataSetReaderMessageDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((JsonDataSetReaderMessageDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region QosDataType Class
    #if (!OPCUA_EXCLUDE_QosDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class QosDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public QosDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.QosDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.QosDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.QosDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.QosDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            QosDataType value = encodeable as QosDataType;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (QosDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            QosDataType clone = (QosDataType)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region QosDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfQosDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "QosDataType")]
    #if !NET_STANDARD
    public partial class QosDataTypeCollection : List<QosDataType>, ICloneable
    #else
    public partial class QosDataTypeCollection : List<QosDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public QosDataTypeCollection() {}

        /// <remarks />
        public QosDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public QosDataTypeCollection(IEnumerable<QosDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator QosDataTypeCollection(QosDataType[] values)
        {
            if (values != null)
            {
                return new QosDataTypeCollection(values);
            }

            return new QosDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator QosDataType[](QosDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (QosDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            QosDataTypeCollection clone = new QosDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((QosDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region TransmitQosDataType Class
    #if (!OPCUA_EXCLUDE_TransmitQosDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class TransmitQosDataType : QosDataType
    {
        #region Constructors
        /// <remarks />
        public TransmitQosDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.TransmitQosDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.TransmitQosDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.TransmitQosDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.TransmitQosDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            TransmitQosDataType value = encodeable as TransmitQosDataType;

            if (value == null)
            {
                return false;
            }


            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (TransmitQosDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TransmitQosDataType clone = (TransmitQosDataType)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region TransmitQosDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfTransmitQosDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "TransmitQosDataType")]
    #if !NET_STANDARD
    public partial class TransmitQosDataTypeCollection : List<TransmitQosDataType>, ICloneable
    #else
    public partial class TransmitQosDataTypeCollection : List<TransmitQosDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public TransmitQosDataTypeCollection() {}

        /// <remarks />
        public TransmitQosDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public TransmitQosDataTypeCollection(IEnumerable<TransmitQosDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator TransmitQosDataTypeCollection(TransmitQosDataType[] values)
        {
            if (values != null)
            {
                return new TransmitQosDataTypeCollection(values);
            }

            return new TransmitQosDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator TransmitQosDataType[](TransmitQosDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (TransmitQosDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TransmitQosDataTypeCollection clone = new TransmitQosDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((TransmitQosDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region TransmitQosPriorityDataType Class
    #if (!OPCUA_EXCLUDE_TransmitQosPriorityDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class TransmitQosPriorityDataType : TransmitQosDataType
    {
        #region Constructors
        /// <remarks />
        public TransmitQosPriorityDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_priorityLabel = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "PriorityLabel", IsRequired = false, Order = 1)]
        public string PriorityLabel
        {
            get { return m_priorityLabel;  }
            set { m_priorityLabel = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.TransmitQosPriorityDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.TransmitQosPriorityDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.TransmitQosPriorityDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.TransmitQosPriorityDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("PriorityLabel", PriorityLabel);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            PriorityLabel = decoder.ReadString("PriorityLabel");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            TransmitQosPriorityDataType value = encodeable as TransmitQosPriorityDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_priorityLabel, value.m_priorityLabel)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (TransmitQosPriorityDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TransmitQosPriorityDataType clone = (TransmitQosPriorityDataType)base.MemberwiseClone();

            clone.m_priorityLabel = (string)Utils.Clone(this.m_priorityLabel);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_priorityLabel;
        #endregion
    }

    #region TransmitQosPriorityDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfTransmitQosPriorityDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "TransmitQosPriorityDataType")]
    #if !NET_STANDARD
    public partial class TransmitQosPriorityDataTypeCollection : List<TransmitQosPriorityDataType>, ICloneable
    #else
    public partial class TransmitQosPriorityDataTypeCollection : List<TransmitQosPriorityDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public TransmitQosPriorityDataTypeCollection() {}

        /// <remarks />
        public TransmitQosPriorityDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public TransmitQosPriorityDataTypeCollection(IEnumerable<TransmitQosPriorityDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator TransmitQosPriorityDataTypeCollection(TransmitQosPriorityDataType[] values)
        {
            if (values != null)
            {
                return new TransmitQosPriorityDataTypeCollection(values);
            }

            return new TransmitQosPriorityDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator TransmitQosPriorityDataType[](TransmitQosPriorityDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (TransmitQosPriorityDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TransmitQosPriorityDataTypeCollection clone = new TransmitQosPriorityDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((TransmitQosPriorityDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region ReceiveQosDataType Class
    #if (!OPCUA_EXCLUDE_ReceiveQosDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ReceiveQosDataType : QosDataType
    {
        #region Constructors
        /// <remarks />
        public ReceiveQosDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.ReceiveQosDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.ReceiveQosDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.ReceiveQosDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.ReceiveQosDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ReceiveQosDataType value = encodeable as ReceiveQosDataType;

            if (value == null)
            {
                return false;
            }


            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (ReceiveQosDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReceiveQosDataType clone = (ReceiveQosDataType)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region ReceiveQosDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfReceiveQosDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ReceiveQosDataType")]
    #if !NET_STANDARD
    public partial class ReceiveQosDataTypeCollection : List<ReceiveQosDataType>, ICloneable
    #else
    public partial class ReceiveQosDataTypeCollection : List<ReceiveQosDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public ReceiveQosDataTypeCollection() {}

        /// <remarks />
        public ReceiveQosDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public ReceiveQosDataTypeCollection(IEnumerable<ReceiveQosDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator ReceiveQosDataTypeCollection(ReceiveQosDataType[] values)
        {
            if (values != null)
            {
                return new ReceiveQosDataTypeCollection(values);
            }

            return new ReceiveQosDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator ReceiveQosDataType[](ReceiveQosDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (ReceiveQosDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReceiveQosDataTypeCollection clone = new ReceiveQosDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ReceiveQosDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region ReceiveQosPriorityDataType Class
    #if (!OPCUA_EXCLUDE_ReceiveQosPriorityDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ReceiveQosPriorityDataType : ReceiveQosDataType
    {
        #region Constructors
        /// <remarks />
        public ReceiveQosPriorityDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_priorityLabel = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "PriorityLabel", IsRequired = false, Order = 1)]
        public string PriorityLabel
        {
            get { return m_priorityLabel;  }
            set { m_priorityLabel = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.ReceiveQosPriorityDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.ReceiveQosPriorityDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.ReceiveQosPriorityDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.ReceiveQosPriorityDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("PriorityLabel", PriorityLabel);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            PriorityLabel = decoder.ReadString("PriorityLabel");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ReceiveQosPriorityDataType value = encodeable as ReceiveQosPriorityDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_priorityLabel, value.m_priorityLabel)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (ReceiveQosPriorityDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReceiveQosPriorityDataType clone = (ReceiveQosPriorityDataType)base.MemberwiseClone();

            clone.m_priorityLabel = (string)Utils.Clone(this.m_priorityLabel);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_priorityLabel;
        #endregion
    }

    #region ReceiveQosPriorityDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfReceiveQosPriorityDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ReceiveQosPriorityDataType")]
    #if !NET_STANDARD
    public partial class ReceiveQosPriorityDataTypeCollection : List<ReceiveQosPriorityDataType>, ICloneable
    #else
    public partial class ReceiveQosPriorityDataTypeCollection : List<ReceiveQosPriorityDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public ReceiveQosPriorityDataTypeCollection() {}

        /// <remarks />
        public ReceiveQosPriorityDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public ReceiveQosPriorityDataTypeCollection(IEnumerable<ReceiveQosPriorityDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator ReceiveQosPriorityDataTypeCollection(ReceiveQosPriorityDataType[] values)
        {
            if (values != null)
            {
                return new ReceiveQosPriorityDataTypeCollection(values);
            }

            return new ReceiveQosPriorityDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator ReceiveQosPriorityDataType[](ReceiveQosPriorityDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (ReceiveQosPriorityDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReceiveQosPriorityDataTypeCollection clone = new ReceiveQosPriorityDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ReceiveQosPriorityDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DatagramConnectionTransportDataType Class
    #if (!OPCUA_EXCLUDE_DatagramConnectionTransportDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DatagramConnectionTransportDataType : ConnectionTransportDataType
    {
        #region Constructors
        /// <remarks />
        public DatagramConnectionTransportDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_discoveryAddress = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "DiscoveryAddress", IsRequired = false, Order = 1)]
        public ExtensionObject DiscoveryAddress
        {
            get { return m_discoveryAddress;  }
            set { m_discoveryAddress = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.DatagramConnectionTransportDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.DatagramConnectionTransportDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.DatagramConnectionTransportDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.DatagramConnectionTransportDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteExtensionObject("DiscoveryAddress", DiscoveryAddress);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            DiscoveryAddress = decoder.ReadExtensionObject("DiscoveryAddress");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DatagramConnectionTransportDataType value = encodeable as DatagramConnectionTransportDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_discoveryAddress, value.m_discoveryAddress)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (DatagramConnectionTransportDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DatagramConnectionTransportDataType clone = (DatagramConnectionTransportDataType)base.MemberwiseClone();

            clone.m_discoveryAddress = (ExtensionObject)Utils.Clone(this.m_discoveryAddress);

            return clone;
        }
        #endregion

        #region Private Fields
        private ExtensionObject m_discoveryAddress;
        #endregion
    }

    #region DatagramConnectionTransportDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDatagramConnectionTransportDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DatagramConnectionTransportDataType")]
    #if !NET_STANDARD
    public partial class DatagramConnectionTransportDataTypeCollection : List<DatagramConnectionTransportDataType>, ICloneable
    #else
    public partial class DatagramConnectionTransportDataTypeCollection : List<DatagramConnectionTransportDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public DatagramConnectionTransportDataTypeCollection() {}

        /// <remarks />
        public DatagramConnectionTransportDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public DatagramConnectionTransportDataTypeCollection(IEnumerable<DatagramConnectionTransportDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator DatagramConnectionTransportDataTypeCollection(DatagramConnectionTransportDataType[] values)
        {
            if (values != null)
            {
                return new DatagramConnectionTransportDataTypeCollection(values);
            }

            return new DatagramConnectionTransportDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator DatagramConnectionTransportDataType[](DatagramConnectionTransportDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (DatagramConnectionTransportDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DatagramConnectionTransportDataTypeCollection clone = new DatagramConnectionTransportDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DatagramConnectionTransportDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DatagramConnectionTransport2DataType Class
    #if (!OPCUA_EXCLUDE_DatagramConnectionTransport2DataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DatagramConnectionTransport2DataType : DatagramConnectionTransportDataType
    {
        #region Constructors
        /// <remarks />
        public DatagramConnectionTransport2DataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_discoveryAnnounceRate = (uint)0;
            m_discoveryMaxMessageSize = (uint)0;
            m_qosCategory = null;
            m_datagramQos = new ExtensionObjectCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "DiscoveryAnnounceRate", IsRequired = false, Order = 1)]
        public uint DiscoveryAnnounceRate
        {
            get { return m_discoveryAnnounceRate;  }
            set { m_discoveryAnnounceRate = value; }
        }

        /// <remarks />
        [DataMember(Name = "DiscoveryMaxMessageSize", IsRequired = false, Order = 2)]
        public uint DiscoveryMaxMessageSize
        {
            get { return m_discoveryMaxMessageSize;  }
            set { m_discoveryMaxMessageSize = value; }
        }

        /// <remarks />
        [DataMember(Name = "QosCategory", IsRequired = false, Order = 3)]
        public string QosCategory
        {
            get { return m_qosCategory;  }
            set { m_qosCategory = value; }
        }

        /// <remarks />
        [DataMember(Name = "DatagramQos", IsRequired = false, Order = 4)]
        public ExtensionObjectCollection DatagramQos
        {
            get { return m_datagramQos;  }
            set { m_datagramQos = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.DatagramConnectionTransport2DataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.DatagramConnectionTransport2DataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.DatagramConnectionTransport2DataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.DatagramConnectionTransport2DataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt32("DiscoveryAnnounceRate", DiscoveryAnnounceRate);
            encoder.WriteUInt32("DiscoveryMaxMessageSize", DiscoveryMaxMessageSize);
            encoder.WriteString("QosCategory", QosCategory);
            encoder.WriteExtensionObjectArray("DatagramQos", DatagramQos);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            DiscoveryAnnounceRate = decoder.ReadUInt32("DiscoveryAnnounceRate");
            DiscoveryMaxMessageSize = decoder.ReadUInt32("DiscoveryMaxMessageSize");
            QosCategory = decoder.ReadString("QosCategory");
            DatagramQos = decoder.ReadExtensionObjectArray("DatagramQos");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DatagramConnectionTransport2DataType value = encodeable as DatagramConnectionTransport2DataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_discoveryAnnounceRate, value.m_discoveryAnnounceRate)) return false;
            if (!Utils.IsEqual(m_discoveryMaxMessageSize, value.m_discoveryMaxMessageSize)) return false;
            if (!Utils.IsEqual(m_qosCategory, value.m_qosCategory)) return false;
            if (!Utils.IsEqual(m_datagramQos, value.m_datagramQos)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (DatagramConnectionTransport2DataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DatagramConnectionTransport2DataType clone = (DatagramConnectionTransport2DataType)base.MemberwiseClone();

            clone.m_discoveryAnnounceRate = (uint)Utils.Clone(this.m_discoveryAnnounceRate);
            clone.m_discoveryMaxMessageSize = (uint)Utils.Clone(this.m_discoveryMaxMessageSize);
            clone.m_qosCategory = (string)Utils.Clone(this.m_qosCategory);
            clone.m_datagramQos = (ExtensionObjectCollection)Utils.Clone(this.m_datagramQos);

            return clone;
        }
        #endregion

        #region Private Fields
        private uint m_discoveryAnnounceRate;
        private uint m_discoveryMaxMessageSize;
        private string m_qosCategory;
        private ExtensionObjectCollection m_datagramQos;
        #endregion
    }

    #region DatagramConnectionTransport2DataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDatagramConnectionTransport2DataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DatagramConnectionTransport2DataType")]
    #if !NET_STANDARD
    public partial class DatagramConnectionTransport2DataTypeCollection : List<DatagramConnectionTransport2DataType>, ICloneable
    #else
    public partial class DatagramConnectionTransport2DataTypeCollection : List<DatagramConnectionTransport2DataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public DatagramConnectionTransport2DataTypeCollection() {}

        /// <remarks />
        public DatagramConnectionTransport2DataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public DatagramConnectionTransport2DataTypeCollection(IEnumerable<DatagramConnectionTransport2DataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator DatagramConnectionTransport2DataTypeCollection(DatagramConnectionTransport2DataType[] values)
        {
            if (values != null)
            {
                return new DatagramConnectionTransport2DataTypeCollection(values);
            }

            return new DatagramConnectionTransport2DataTypeCollection();
        }

        /// <remarks />
        public static explicit operator DatagramConnectionTransport2DataType[](DatagramConnectionTransport2DataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (DatagramConnectionTransport2DataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DatagramConnectionTransport2DataTypeCollection clone = new DatagramConnectionTransport2DataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DatagramConnectionTransport2DataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DatagramWriterGroupTransportDataType Class
    #if (!OPCUA_EXCLUDE_DatagramWriterGroupTransportDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DatagramWriterGroupTransportDataType : WriterGroupTransportDataType
    {
        #region Constructors
        /// <remarks />
        public DatagramWriterGroupTransportDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_messageRepeatCount = (byte)0;
            m_messageRepeatDelay = (double)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "MessageRepeatCount", IsRequired = false, Order = 1)]
        public byte MessageRepeatCount
        {
            get { return m_messageRepeatCount;  }
            set { m_messageRepeatCount = value; }
        }

        /// <remarks />
        [DataMember(Name = "MessageRepeatDelay", IsRequired = false, Order = 2)]
        public double MessageRepeatDelay
        {
            get { return m_messageRepeatDelay;  }
            set { m_messageRepeatDelay = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.DatagramWriterGroupTransportDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.DatagramWriterGroupTransportDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.DatagramWriterGroupTransportDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.DatagramWriterGroupTransportDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteByte("MessageRepeatCount", MessageRepeatCount);
            encoder.WriteDouble("MessageRepeatDelay", MessageRepeatDelay);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            MessageRepeatCount = decoder.ReadByte("MessageRepeatCount");
            MessageRepeatDelay = decoder.ReadDouble("MessageRepeatDelay");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DatagramWriterGroupTransportDataType value = encodeable as DatagramWriterGroupTransportDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_messageRepeatCount, value.m_messageRepeatCount)) return false;
            if (!Utils.IsEqual(m_messageRepeatDelay, value.m_messageRepeatDelay)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (DatagramWriterGroupTransportDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DatagramWriterGroupTransportDataType clone = (DatagramWriterGroupTransportDataType)base.MemberwiseClone();

            clone.m_messageRepeatCount = (byte)Utils.Clone(this.m_messageRepeatCount);
            clone.m_messageRepeatDelay = (double)Utils.Clone(this.m_messageRepeatDelay);

            return clone;
        }
        #endregion

        #region Private Fields
        private byte m_messageRepeatCount;
        private double m_messageRepeatDelay;
        #endregion
    }

    #region DatagramWriterGroupTransportDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDatagramWriterGroupTransportDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DatagramWriterGroupTransportDataType")]
    #if !NET_STANDARD
    public partial class DatagramWriterGroupTransportDataTypeCollection : List<DatagramWriterGroupTransportDataType>, ICloneable
    #else
    public partial class DatagramWriterGroupTransportDataTypeCollection : List<DatagramWriterGroupTransportDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public DatagramWriterGroupTransportDataTypeCollection() {}

        /// <remarks />
        public DatagramWriterGroupTransportDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public DatagramWriterGroupTransportDataTypeCollection(IEnumerable<DatagramWriterGroupTransportDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator DatagramWriterGroupTransportDataTypeCollection(DatagramWriterGroupTransportDataType[] values)
        {
            if (values != null)
            {
                return new DatagramWriterGroupTransportDataTypeCollection(values);
            }

            return new DatagramWriterGroupTransportDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator DatagramWriterGroupTransportDataType[](DatagramWriterGroupTransportDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (DatagramWriterGroupTransportDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DatagramWriterGroupTransportDataTypeCollection clone = new DatagramWriterGroupTransportDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DatagramWriterGroupTransportDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DatagramWriterGroupTransport2DataType Class
    #if (!OPCUA_EXCLUDE_DatagramWriterGroupTransport2DataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DatagramWriterGroupTransport2DataType : DatagramWriterGroupTransportDataType
    {
        #region Constructors
        /// <remarks />
        public DatagramWriterGroupTransport2DataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_address = null;
            m_qosCategory = null;
            m_datagramQos = new ExtensionObjectCollection();
            m_discoveryAnnounceRate = (uint)0;
            m_topic = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Address", IsRequired = false, Order = 1)]
        public ExtensionObject Address
        {
            get { return m_address;  }
            set { m_address = value; }
        }

        /// <remarks />
        [DataMember(Name = "QosCategory", IsRequired = false, Order = 2)]
        public string QosCategory
        {
            get { return m_qosCategory;  }
            set { m_qosCategory = value; }
        }

        /// <remarks />
        [DataMember(Name = "DatagramQos", IsRequired = false, Order = 3)]
        public ExtensionObjectCollection DatagramQos
        {
            get { return m_datagramQos;  }
            set { m_datagramQos = value; }
        }

        /// <remarks />
        [DataMember(Name = "DiscoveryAnnounceRate", IsRequired = false, Order = 4)]
        public uint DiscoveryAnnounceRate
        {
            get { return m_discoveryAnnounceRate;  }
            set { m_discoveryAnnounceRate = value; }
        }

        /// <remarks />
        [DataMember(Name = "Topic", IsRequired = false, Order = 5)]
        public string Topic
        {
            get { return m_topic;  }
            set { m_topic = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.DatagramWriterGroupTransport2DataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.DatagramWriterGroupTransport2DataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.DatagramWriterGroupTransport2DataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.DatagramWriterGroupTransport2DataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteExtensionObject("Address", Address);
            encoder.WriteString("QosCategory", QosCategory);
            encoder.WriteExtensionObjectArray("DatagramQos", DatagramQos);
            encoder.WriteUInt32("DiscoveryAnnounceRate", DiscoveryAnnounceRate);
            encoder.WriteString("Topic", Topic);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Address = decoder.ReadExtensionObject("Address");
            QosCategory = decoder.ReadString("QosCategory");
            DatagramQos = decoder.ReadExtensionObjectArray("DatagramQos");
            DiscoveryAnnounceRate = decoder.ReadUInt32("DiscoveryAnnounceRate");
            Topic = decoder.ReadString("Topic");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DatagramWriterGroupTransport2DataType value = encodeable as DatagramWriterGroupTransport2DataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_address, value.m_address)) return false;
            if (!Utils.IsEqual(m_qosCategory, value.m_qosCategory)) return false;
            if (!Utils.IsEqual(m_datagramQos, value.m_datagramQos)) return false;
            if (!Utils.IsEqual(m_discoveryAnnounceRate, value.m_discoveryAnnounceRate)) return false;
            if (!Utils.IsEqual(m_topic, value.m_topic)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (DatagramWriterGroupTransport2DataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DatagramWriterGroupTransport2DataType clone = (DatagramWriterGroupTransport2DataType)base.MemberwiseClone();

            clone.m_address = (ExtensionObject)Utils.Clone(this.m_address);
            clone.m_qosCategory = (string)Utils.Clone(this.m_qosCategory);
            clone.m_datagramQos = (ExtensionObjectCollection)Utils.Clone(this.m_datagramQos);
            clone.m_discoveryAnnounceRate = (uint)Utils.Clone(this.m_discoveryAnnounceRate);
            clone.m_topic = (string)Utils.Clone(this.m_topic);

            return clone;
        }
        #endregion

        #region Private Fields
        private ExtensionObject m_address;
        private string m_qosCategory;
        private ExtensionObjectCollection m_datagramQos;
        private uint m_discoveryAnnounceRate;
        private string m_topic;
        #endregion
    }

    #region DatagramWriterGroupTransport2DataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDatagramWriterGroupTransport2DataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DatagramWriterGroupTransport2DataType")]
    #if !NET_STANDARD
    public partial class DatagramWriterGroupTransport2DataTypeCollection : List<DatagramWriterGroupTransport2DataType>, ICloneable
    #else
    public partial class DatagramWriterGroupTransport2DataTypeCollection : List<DatagramWriterGroupTransport2DataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public DatagramWriterGroupTransport2DataTypeCollection() {}

        /// <remarks />
        public DatagramWriterGroupTransport2DataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public DatagramWriterGroupTransport2DataTypeCollection(IEnumerable<DatagramWriterGroupTransport2DataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator DatagramWriterGroupTransport2DataTypeCollection(DatagramWriterGroupTransport2DataType[] values)
        {
            if (values != null)
            {
                return new DatagramWriterGroupTransport2DataTypeCollection(values);
            }

            return new DatagramWriterGroupTransport2DataTypeCollection();
        }

        /// <remarks />
        public static explicit operator DatagramWriterGroupTransport2DataType[](DatagramWriterGroupTransport2DataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (DatagramWriterGroupTransport2DataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DatagramWriterGroupTransport2DataTypeCollection clone = new DatagramWriterGroupTransport2DataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DatagramWriterGroupTransport2DataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DatagramDataSetReaderTransportDataType Class
    #if (!OPCUA_EXCLUDE_DatagramDataSetReaderTransportDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DatagramDataSetReaderTransportDataType : DataSetReaderTransportDataType
    {
        #region Constructors
        /// <remarks />
        public DatagramDataSetReaderTransportDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_address = null;
            m_qosCategory = null;
            m_datagramQos = new ExtensionObjectCollection();
            m_topic = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Address", IsRequired = false, Order = 1)]
        public ExtensionObject Address
        {
            get { return m_address;  }
            set { m_address = value; }
        }

        /// <remarks />
        [DataMember(Name = "QosCategory", IsRequired = false, Order = 2)]
        public string QosCategory
        {
            get { return m_qosCategory;  }
            set { m_qosCategory = value; }
        }

        /// <remarks />
        [DataMember(Name = "DatagramQos", IsRequired = false, Order = 3)]
        public ExtensionObjectCollection DatagramQos
        {
            get { return m_datagramQos;  }
            set { m_datagramQos = value; }
        }

        /// <remarks />
        [DataMember(Name = "Topic", IsRequired = false, Order = 4)]
        public string Topic
        {
            get { return m_topic;  }
            set { m_topic = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.DatagramDataSetReaderTransportDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.DatagramDataSetReaderTransportDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.DatagramDataSetReaderTransportDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.DatagramDataSetReaderTransportDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteExtensionObject("Address", Address);
            encoder.WriteString("QosCategory", QosCategory);
            encoder.WriteExtensionObjectArray("DatagramQos", DatagramQos);
            encoder.WriteString("Topic", Topic);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Address = decoder.ReadExtensionObject("Address");
            QosCategory = decoder.ReadString("QosCategory");
            DatagramQos = decoder.ReadExtensionObjectArray("DatagramQos");
            Topic = decoder.ReadString("Topic");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DatagramDataSetReaderTransportDataType value = encodeable as DatagramDataSetReaderTransportDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_address, value.m_address)) return false;
            if (!Utils.IsEqual(m_qosCategory, value.m_qosCategory)) return false;
            if (!Utils.IsEqual(m_datagramQos, value.m_datagramQos)) return false;
            if (!Utils.IsEqual(m_topic, value.m_topic)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (DatagramDataSetReaderTransportDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DatagramDataSetReaderTransportDataType clone = (DatagramDataSetReaderTransportDataType)base.MemberwiseClone();

            clone.m_address = (ExtensionObject)Utils.Clone(this.m_address);
            clone.m_qosCategory = (string)Utils.Clone(this.m_qosCategory);
            clone.m_datagramQos = (ExtensionObjectCollection)Utils.Clone(this.m_datagramQos);
            clone.m_topic = (string)Utils.Clone(this.m_topic);

            return clone;
        }
        #endregion

        #region Private Fields
        private ExtensionObject m_address;
        private string m_qosCategory;
        private ExtensionObjectCollection m_datagramQos;
        private string m_topic;
        #endregion
    }

    #region DatagramDataSetReaderTransportDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDatagramDataSetReaderTransportDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DatagramDataSetReaderTransportDataType")]
    #if !NET_STANDARD
    public partial class DatagramDataSetReaderTransportDataTypeCollection : List<DatagramDataSetReaderTransportDataType>, ICloneable
    #else
    public partial class DatagramDataSetReaderTransportDataTypeCollection : List<DatagramDataSetReaderTransportDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public DatagramDataSetReaderTransportDataTypeCollection() {}

        /// <remarks />
        public DatagramDataSetReaderTransportDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public DatagramDataSetReaderTransportDataTypeCollection(IEnumerable<DatagramDataSetReaderTransportDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator DatagramDataSetReaderTransportDataTypeCollection(DatagramDataSetReaderTransportDataType[] values)
        {
            if (values != null)
            {
                return new DatagramDataSetReaderTransportDataTypeCollection(values);
            }

            return new DatagramDataSetReaderTransportDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator DatagramDataSetReaderTransportDataType[](DatagramDataSetReaderTransportDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (DatagramDataSetReaderTransportDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DatagramDataSetReaderTransportDataTypeCollection clone = new DatagramDataSetReaderTransportDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DatagramDataSetReaderTransportDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region BrokerConnectionTransportDataType Class
    #if (!OPCUA_EXCLUDE_BrokerConnectionTransportDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class BrokerConnectionTransportDataType : ConnectionTransportDataType
    {
        #region Constructors
        /// <remarks />
        public BrokerConnectionTransportDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_resourceUri = null;
            m_authenticationProfileUri = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "ResourceUri", IsRequired = false, Order = 1)]
        public string ResourceUri
        {
            get { return m_resourceUri;  }
            set { m_resourceUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "AuthenticationProfileUri", IsRequired = false, Order = 2)]
        public string AuthenticationProfileUri
        {
            get { return m_authenticationProfileUri;  }
            set { m_authenticationProfileUri = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.BrokerConnectionTransportDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.BrokerConnectionTransportDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.BrokerConnectionTransportDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.BrokerConnectionTransportDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("ResourceUri", ResourceUri);
            encoder.WriteString("AuthenticationProfileUri", AuthenticationProfileUri);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ResourceUri = decoder.ReadString("ResourceUri");
            AuthenticationProfileUri = decoder.ReadString("AuthenticationProfileUri");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            BrokerConnectionTransportDataType value = encodeable as BrokerConnectionTransportDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_resourceUri, value.m_resourceUri)) return false;
            if (!Utils.IsEqual(m_authenticationProfileUri, value.m_authenticationProfileUri)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (BrokerConnectionTransportDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            BrokerConnectionTransportDataType clone = (BrokerConnectionTransportDataType)base.MemberwiseClone();

            clone.m_resourceUri = (string)Utils.Clone(this.m_resourceUri);
            clone.m_authenticationProfileUri = (string)Utils.Clone(this.m_authenticationProfileUri);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_resourceUri;
        private string m_authenticationProfileUri;
        #endregion
    }

    #region BrokerConnectionTransportDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfBrokerConnectionTransportDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "BrokerConnectionTransportDataType")]
    #if !NET_STANDARD
    public partial class BrokerConnectionTransportDataTypeCollection : List<BrokerConnectionTransportDataType>, ICloneable
    #else
    public partial class BrokerConnectionTransportDataTypeCollection : List<BrokerConnectionTransportDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public BrokerConnectionTransportDataTypeCollection() {}

        /// <remarks />
        public BrokerConnectionTransportDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public BrokerConnectionTransportDataTypeCollection(IEnumerable<BrokerConnectionTransportDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator BrokerConnectionTransportDataTypeCollection(BrokerConnectionTransportDataType[] values)
        {
            if (values != null)
            {
                return new BrokerConnectionTransportDataTypeCollection(values);
            }

            return new BrokerConnectionTransportDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator BrokerConnectionTransportDataType[](BrokerConnectionTransportDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (BrokerConnectionTransportDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            BrokerConnectionTransportDataTypeCollection clone = new BrokerConnectionTransportDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((BrokerConnectionTransportDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region BrokerTransportQualityOfService Enumeration
    #if (!OPCUA_EXCLUDE_BrokerTransportQualityOfService)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum BrokerTransportQualityOfService
    {
        /// <remarks />
        [EnumMember(Value = "NotSpecified_0")]
        NotSpecified = 0,

        /// <remarks />
        [EnumMember(Value = "BestEffort_1")]
        BestEffort = 1,

        /// <remarks />
        [EnumMember(Value = "AtLeastOnce_2")]
        AtLeastOnce = 2,

        /// <remarks />
        [EnumMember(Value = "AtMostOnce_3")]
        AtMostOnce = 3,

        /// <remarks />
        [EnumMember(Value = "ExactlyOnce_4")]
        ExactlyOnce = 4,
    }

    #region BrokerTransportQualityOfServiceCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfBrokerTransportQualityOfService", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "BrokerTransportQualityOfService")]
    #if !NET_STANDARD
    public partial class BrokerTransportQualityOfServiceCollection : List<BrokerTransportQualityOfService>, ICloneable
    #else
    public partial class BrokerTransportQualityOfServiceCollection : List<BrokerTransportQualityOfService>
    #endif
    {
        #region Constructors
        /// <remarks />
        public BrokerTransportQualityOfServiceCollection() {}

        /// <remarks />
        public BrokerTransportQualityOfServiceCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public BrokerTransportQualityOfServiceCollection(IEnumerable<BrokerTransportQualityOfService> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator BrokerTransportQualityOfServiceCollection(BrokerTransportQualityOfService[] values)
        {
            if (values != null)
            {
                return new BrokerTransportQualityOfServiceCollection(values);
            }

            return new BrokerTransportQualityOfServiceCollection();
        }

        /// <remarks />
        public static explicit operator BrokerTransportQualityOfService[](BrokerTransportQualityOfServiceCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (BrokerTransportQualityOfServiceCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            BrokerTransportQualityOfServiceCollection clone = new BrokerTransportQualityOfServiceCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((BrokerTransportQualityOfService)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region BrokerWriterGroupTransportDataType Class
    #if (!OPCUA_EXCLUDE_BrokerWriterGroupTransportDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class BrokerWriterGroupTransportDataType : WriterGroupTransportDataType
    {
        #region Constructors
        /// <remarks />
        public BrokerWriterGroupTransportDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_queueName = null;
            m_resourceUri = null;
            m_authenticationProfileUri = null;
            m_requestedDeliveryGuarantee = BrokerTransportQualityOfService.NotSpecified;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "QueueName", IsRequired = false, Order = 1)]
        public string QueueName
        {
            get { return m_queueName;  }
            set { m_queueName = value; }
        }

        /// <remarks />
        [DataMember(Name = "ResourceUri", IsRequired = false, Order = 2)]
        public string ResourceUri
        {
            get { return m_resourceUri;  }
            set { m_resourceUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "AuthenticationProfileUri", IsRequired = false, Order = 3)]
        public string AuthenticationProfileUri
        {
            get { return m_authenticationProfileUri;  }
            set { m_authenticationProfileUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "RequestedDeliveryGuarantee", IsRequired = false, Order = 4)]
        public BrokerTransportQualityOfService RequestedDeliveryGuarantee
        {
            get { return m_requestedDeliveryGuarantee;  }
            set { m_requestedDeliveryGuarantee = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.BrokerWriterGroupTransportDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.BrokerWriterGroupTransportDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.BrokerWriterGroupTransportDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.BrokerWriterGroupTransportDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("QueueName", QueueName);
            encoder.WriteString("ResourceUri", ResourceUri);
            encoder.WriteString("AuthenticationProfileUri", AuthenticationProfileUri);
            encoder.WriteEnumerated("RequestedDeliveryGuarantee", RequestedDeliveryGuarantee);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            QueueName = decoder.ReadString("QueueName");
            ResourceUri = decoder.ReadString("ResourceUri");
            AuthenticationProfileUri = decoder.ReadString("AuthenticationProfileUri");
            RequestedDeliveryGuarantee = (BrokerTransportQualityOfService)decoder.ReadEnumerated("RequestedDeliveryGuarantee", typeof(BrokerTransportQualityOfService));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            BrokerWriterGroupTransportDataType value = encodeable as BrokerWriterGroupTransportDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_queueName, value.m_queueName)) return false;
            if (!Utils.IsEqual(m_resourceUri, value.m_resourceUri)) return false;
            if (!Utils.IsEqual(m_authenticationProfileUri, value.m_authenticationProfileUri)) return false;
            if (!Utils.IsEqual(m_requestedDeliveryGuarantee, value.m_requestedDeliveryGuarantee)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (BrokerWriterGroupTransportDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            BrokerWriterGroupTransportDataType clone = (BrokerWriterGroupTransportDataType)base.MemberwiseClone();

            clone.m_queueName = (string)Utils.Clone(this.m_queueName);
            clone.m_resourceUri = (string)Utils.Clone(this.m_resourceUri);
            clone.m_authenticationProfileUri = (string)Utils.Clone(this.m_authenticationProfileUri);
            clone.m_requestedDeliveryGuarantee = (BrokerTransportQualityOfService)Utils.Clone(this.m_requestedDeliveryGuarantee);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_queueName;
        private string m_resourceUri;
        private string m_authenticationProfileUri;
        private BrokerTransportQualityOfService m_requestedDeliveryGuarantee;
        #endregion
    }

    #region BrokerWriterGroupTransportDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfBrokerWriterGroupTransportDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "BrokerWriterGroupTransportDataType")]
    #if !NET_STANDARD
    public partial class BrokerWriterGroupTransportDataTypeCollection : List<BrokerWriterGroupTransportDataType>, ICloneable
    #else
    public partial class BrokerWriterGroupTransportDataTypeCollection : List<BrokerWriterGroupTransportDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public BrokerWriterGroupTransportDataTypeCollection() {}

        /// <remarks />
        public BrokerWriterGroupTransportDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public BrokerWriterGroupTransportDataTypeCollection(IEnumerable<BrokerWriterGroupTransportDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator BrokerWriterGroupTransportDataTypeCollection(BrokerWriterGroupTransportDataType[] values)
        {
            if (values != null)
            {
                return new BrokerWriterGroupTransportDataTypeCollection(values);
            }

            return new BrokerWriterGroupTransportDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator BrokerWriterGroupTransportDataType[](BrokerWriterGroupTransportDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (BrokerWriterGroupTransportDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            BrokerWriterGroupTransportDataTypeCollection clone = new BrokerWriterGroupTransportDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((BrokerWriterGroupTransportDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region BrokerDataSetWriterTransportDataType Class
    #if (!OPCUA_EXCLUDE_BrokerDataSetWriterTransportDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class BrokerDataSetWriterTransportDataType : DataSetWriterTransportDataType
    {
        #region Constructors
        /// <remarks />
        public BrokerDataSetWriterTransportDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_queueName = null;
            m_resourceUri = null;
            m_authenticationProfileUri = null;
            m_requestedDeliveryGuarantee = BrokerTransportQualityOfService.NotSpecified;
            m_metaDataQueueName = null;
            m_metaDataUpdateTime = (double)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "QueueName", IsRequired = false, Order = 1)]
        public string QueueName
        {
            get { return m_queueName;  }
            set { m_queueName = value; }
        }

        /// <remarks />
        [DataMember(Name = "ResourceUri", IsRequired = false, Order = 2)]
        public string ResourceUri
        {
            get { return m_resourceUri;  }
            set { m_resourceUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "AuthenticationProfileUri", IsRequired = false, Order = 3)]
        public string AuthenticationProfileUri
        {
            get { return m_authenticationProfileUri;  }
            set { m_authenticationProfileUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "RequestedDeliveryGuarantee", IsRequired = false, Order = 4)]
        public BrokerTransportQualityOfService RequestedDeliveryGuarantee
        {
            get { return m_requestedDeliveryGuarantee;  }
            set { m_requestedDeliveryGuarantee = value; }
        }

        /// <remarks />
        [DataMember(Name = "MetaDataQueueName", IsRequired = false, Order = 5)]
        public string MetaDataQueueName
        {
            get { return m_metaDataQueueName;  }
            set { m_metaDataQueueName = value; }
        }

        /// <remarks />
        [DataMember(Name = "MetaDataUpdateTime", IsRequired = false, Order = 6)]
        public double MetaDataUpdateTime
        {
            get { return m_metaDataUpdateTime;  }
            set { m_metaDataUpdateTime = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.BrokerDataSetWriterTransportDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.BrokerDataSetWriterTransportDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.BrokerDataSetWriterTransportDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.BrokerDataSetWriterTransportDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("QueueName", QueueName);
            encoder.WriteString("ResourceUri", ResourceUri);
            encoder.WriteString("AuthenticationProfileUri", AuthenticationProfileUri);
            encoder.WriteEnumerated("RequestedDeliveryGuarantee", RequestedDeliveryGuarantee);
            encoder.WriteString("MetaDataQueueName", MetaDataQueueName);
            encoder.WriteDouble("MetaDataUpdateTime", MetaDataUpdateTime);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            QueueName = decoder.ReadString("QueueName");
            ResourceUri = decoder.ReadString("ResourceUri");
            AuthenticationProfileUri = decoder.ReadString("AuthenticationProfileUri");
            RequestedDeliveryGuarantee = (BrokerTransportQualityOfService)decoder.ReadEnumerated("RequestedDeliveryGuarantee", typeof(BrokerTransportQualityOfService));
            MetaDataQueueName = decoder.ReadString("MetaDataQueueName");
            MetaDataUpdateTime = decoder.ReadDouble("MetaDataUpdateTime");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            BrokerDataSetWriterTransportDataType value = encodeable as BrokerDataSetWriterTransportDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_queueName, value.m_queueName)) return false;
            if (!Utils.IsEqual(m_resourceUri, value.m_resourceUri)) return false;
            if (!Utils.IsEqual(m_authenticationProfileUri, value.m_authenticationProfileUri)) return false;
            if (!Utils.IsEqual(m_requestedDeliveryGuarantee, value.m_requestedDeliveryGuarantee)) return false;
            if (!Utils.IsEqual(m_metaDataQueueName, value.m_metaDataQueueName)) return false;
            if (!Utils.IsEqual(m_metaDataUpdateTime, value.m_metaDataUpdateTime)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (BrokerDataSetWriterTransportDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            BrokerDataSetWriterTransportDataType clone = (BrokerDataSetWriterTransportDataType)base.MemberwiseClone();

            clone.m_queueName = (string)Utils.Clone(this.m_queueName);
            clone.m_resourceUri = (string)Utils.Clone(this.m_resourceUri);
            clone.m_authenticationProfileUri = (string)Utils.Clone(this.m_authenticationProfileUri);
            clone.m_requestedDeliveryGuarantee = (BrokerTransportQualityOfService)Utils.Clone(this.m_requestedDeliveryGuarantee);
            clone.m_metaDataQueueName = (string)Utils.Clone(this.m_metaDataQueueName);
            clone.m_metaDataUpdateTime = (double)Utils.Clone(this.m_metaDataUpdateTime);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_queueName;
        private string m_resourceUri;
        private string m_authenticationProfileUri;
        private BrokerTransportQualityOfService m_requestedDeliveryGuarantee;
        private string m_metaDataQueueName;
        private double m_metaDataUpdateTime;
        #endregion
    }

    #region BrokerDataSetWriterTransportDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfBrokerDataSetWriterTransportDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "BrokerDataSetWriterTransportDataType")]
    #if !NET_STANDARD
    public partial class BrokerDataSetWriterTransportDataTypeCollection : List<BrokerDataSetWriterTransportDataType>, ICloneable
    #else
    public partial class BrokerDataSetWriterTransportDataTypeCollection : List<BrokerDataSetWriterTransportDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public BrokerDataSetWriterTransportDataTypeCollection() {}

        /// <remarks />
        public BrokerDataSetWriterTransportDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public BrokerDataSetWriterTransportDataTypeCollection(IEnumerable<BrokerDataSetWriterTransportDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator BrokerDataSetWriterTransportDataTypeCollection(BrokerDataSetWriterTransportDataType[] values)
        {
            if (values != null)
            {
                return new BrokerDataSetWriterTransportDataTypeCollection(values);
            }

            return new BrokerDataSetWriterTransportDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator BrokerDataSetWriterTransportDataType[](BrokerDataSetWriterTransportDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (BrokerDataSetWriterTransportDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            BrokerDataSetWriterTransportDataTypeCollection clone = new BrokerDataSetWriterTransportDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((BrokerDataSetWriterTransportDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region BrokerDataSetReaderTransportDataType Class
    #if (!OPCUA_EXCLUDE_BrokerDataSetReaderTransportDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class BrokerDataSetReaderTransportDataType : DataSetReaderTransportDataType
    {
        #region Constructors
        /// <remarks />
        public BrokerDataSetReaderTransportDataType()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_queueName = null;
            m_resourceUri = null;
            m_authenticationProfileUri = null;
            m_requestedDeliveryGuarantee = BrokerTransportQualityOfService.NotSpecified;
            m_metaDataQueueName = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "QueueName", IsRequired = false, Order = 1)]
        public string QueueName
        {
            get { return m_queueName;  }
            set { m_queueName = value; }
        }

        /// <remarks />
        [DataMember(Name = "ResourceUri", IsRequired = false, Order = 2)]
        public string ResourceUri
        {
            get { return m_resourceUri;  }
            set { m_resourceUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "AuthenticationProfileUri", IsRequired = false, Order = 3)]
        public string AuthenticationProfileUri
        {
            get { return m_authenticationProfileUri;  }
            set { m_authenticationProfileUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "RequestedDeliveryGuarantee", IsRequired = false, Order = 4)]
        public BrokerTransportQualityOfService RequestedDeliveryGuarantee
        {
            get { return m_requestedDeliveryGuarantee;  }
            set { m_requestedDeliveryGuarantee = value; }
        }

        /// <remarks />
        [DataMember(Name = "MetaDataQueueName", IsRequired = false, Order = 5)]
        public string MetaDataQueueName
        {
            get { return m_metaDataQueueName;  }
            set { m_metaDataQueueName = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.BrokerDataSetReaderTransportDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.BrokerDataSetReaderTransportDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.BrokerDataSetReaderTransportDataType_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.BrokerDataSetReaderTransportDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("QueueName", QueueName);
            encoder.WriteString("ResourceUri", ResourceUri);
            encoder.WriteString("AuthenticationProfileUri", AuthenticationProfileUri);
            encoder.WriteEnumerated("RequestedDeliveryGuarantee", RequestedDeliveryGuarantee);
            encoder.WriteString("MetaDataQueueName", MetaDataQueueName);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            QueueName = decoder.ReadString("QueueName");
            ResourceUri = decoder.ReadString("ResourceUri");
            AuthenticationProfileUri = decoder.ReadString("AuthenticationProfileUri");
            RequestedDeliveryGuarantee = (BrokerTransportQualityOfService)decoder.ReadEnumerated("RequestedDeliveryGuarantee", typeof(BrokerTransportQualityOfService));
            MetaDataQueueName = decoder.ReadString("MetaDataQueueName");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            BrokerDataSetReaderTransportDataType value = encodeable as BrokerDataSetReaderTransportDataType;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_queueName, value.m_queueName)) return false;
            if (!Utils.IsEqual(m_resourceUri, value.m_resourceUri)) return false;
            if (!Utils.IsEqual(m_authenticationProfileUri, value.m_authenticationProfileUri)) return false;
            if (!Utils.IsEqual(m_requestedDeliveryGuarantee, value.m_requestedDeliveryGuarantee)) return false;
            if (!Utils.IsEqual(m_metaDataQueueName, value.m_metaDataQueueName)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (BrokerDataSetReaderTransportDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            BrokerDataSetReaderTransportDataType clone = (BrokerDataSetReaderTransportDataType)base.MemberwiseClone();

            clone.m_queueName = (string)Utils.Clone(this.m_queueName);
            clone.m_resourceUri = (string)Utils.Clone(this.m_resourceUri);
            clone.m_authenticationProfileUri = (string)Utils.Clone(this.m_authenticationProfileUri);
            clone.m_requestedDeliveryGuarantee = (BrokerTransportQualityOfService)Utils.Clone(this.m_requestedDeliveryGuarantee);
            clone.m_metaDataQueueName = (string)Utils.Clone(this.m_metaDataQueueName);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_queueName;
        private string m_resourceUri;
        private string m_authenticationProfileUri;
        private BrokerTransportQualityOfService m_requestedDeliveryGuarantee;
        private string m_metaDataQueueName;
        #endregion
    }

    #region BrokerDataSetReaderTransportDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfBrokerDataSetReaderTransportDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "BrokerDataSetReaderTransportDataType")]
    #if !NET_STANDARD
    public partial class BrokerDataSetReaderTransportDataTypeCollection : List<BrokerDataSetReaderTransportDataType>, ICloneable
    #else
    public partial class BrokerDataSetReaderTransportDataTypeCollection : List<BrokerDataSetReaderTransportDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public BrokerDataSetReaderTransportDataTypeCollection() {}

        /// <remarks />
        public BrokerDataSetReaderTransportDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public BrokerDataSetReaderTransportDataTypeCollection(IEnumerable<BrokerDataSetReaderTransportDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator BrokerDataSetReaderTransportDataTypeCollection(BrokerDataSetReaderTransportDataType[] values)
        {
            if (values != null)
            {
                return new BrokerDataSetReaderTransportDataTypeCollection(values);
            }

            return new BrokerDataSetReaderTransportDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator BrokerDataSetReaderTransportDataType[](BrokerDataSetReaderTransportDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (BrokerDataSetReaderTransportDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            BrokerDataSetReaderTransportDataTypeCollection clone = new BrokerDataSetReaderTransportDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((BrokerDataSetReaderTransportDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PubSubConfigurationRefMask Enumeration
    #if (!OPCUA_EXCLUDE_PubSubConfigurationRefMask)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum PubSubConfigurationRefMask : UInt32
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "ElementAdd_1")]
        ElementAdd = 1,

        /// <remarks />
        [EnumMember(Value = "ElementMatch_2")]
        ElementMatch = 2,

        /// <remarks />
        [EnumMember(Value = "ElementModify_4")]
        ElementModify = 4,

        /// <remarks />
        [EnumMember(Value = "ElementRemove_8")]
        ElementRemove = 8,

        /// <remarks />
        [EnumMember(Value = "ReferenceWriter_16")]
        ReferenceWriter = 16,

        /// <remarks />
        [EnumMember(Value = "ReferenceReader_32")]
        ReferenceReader = 32,

        /// <remarks />
        [EnumMember(Value = "ReferenceWriterGroup_64")]
        ReferenceWriterGroup = 64,

        /// <remarks />
        [EnumMember(Value = "ReferenceReaderGroup_128")]
        ReferenceReaderGroup = 128,

        /// <remarks />
        [EnumMember(Value = "ReferenceConnection_256")]
        ReferenceConnection = 256,

        /// <remarks />
        [EnumMember(Value = "ReferencePubDataset_512")]
        ReferencePubDataset = 512,

        /// <remarks />
        [EnumMember(Value = "ReferenceSubDataset_1024")]
        ReferenceSubDataset = 1024,

        /// <remarks />
        [EnumMember(Value = "ReferenceSecurityGroup_2048")]
        ReferenceSecurityGroup = 2048,

        /// <remarks />
        [EnumMember(Value = "ReferencePushTarget_4096")]
        ReferencePushTarget = 4096,
    }

    #region PubSubConfigurationRefMaskCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPubSubConfigurationRefMask", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PubSubConfigurationRefMask")]
    #if !NET_STANDARD
    public partial class PubSubConfigurationRefMaskCollection : List<PubSubConfigurationRefMask>, ICloneable
    #else
    public partial class PubSubConfigurationRefMaskCollection : List<PubSubConfigurationRefMask>
    #endif
    {
        #region Constructors
        /// <remarks />
        public PubSubConfigurationRefMaskCollection() {}

        /// <remarks />
        public PubSubConfigurationRefMaskCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public PubSubConfigurationRefMaskCollection(IEnumerable<PubSubConfigurationRefMask> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator PubSubConfigurationRefMaskCollection(PubSubConfigurationRefMask[] values)
        {
            if (values != null)
            {
                return new PubSubConfigurationRefMaskCollection(values);
            }

            return new PubSubConfigurationRefMaskCollection();
        }

        /// <remarks />
        public static explicit operator PubSubConfigurationRefMask[](PubSubConfigurationRefMaskCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (PubSubConfigurationRefMaskCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PubSubConfigurationRefMaskCollection clone = new PubSubConfigurationRefMaskCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PubSubConfigurationRefMask)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PubSubConfigurationRefDataType Class
    #if (!OPCUA_EXCLUDE_PubSubConfigurationRefDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class PubSubConfigurationRefDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public PubSubConfigurationRefDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_configurationMask = 0;
            m_elementIndex = (ushort)0;
            m_connectionIndex = (ushort)0;
            m_groupIndex = (ushort)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "ConfigurationMask", IsRequired = false, Order = 1)]
        public uint ConfigurationMask
        {
            get { return m_configurationMask;  }
            set { m_configurationMask = value; }
        }

        /// <remarks />
        [DataMember(Name = "ElementIndex", IsRequired = false, Order = 2)]
        public ushort ElementIndex
        {
            get { return m_elementIndex;  }
            set { m_elementIndex = value; }
        }

        /// <remarks />
        [DataMember(Name = "ConnectionIndex", IsRequired = false, Order = 3)]
        public ushort ConnectionIndex
        {
            get { return m_connectionIndex;  }
            set { m_connectionIndex = value; }
        }

        /// <remarks />
        [DataMember(Name = "GroupIndex", IsRequired = false, Order = 4)]
        public ushort GroupIndex
        {
            get { return m_groupIndex;  }
            set { m_groupIndex = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.PubSubConfigurationRefDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.PubSubConfigurationRefDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.PubSubConfigurationRefDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.PubSubConfigurationRefDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt32("ConfigurationMask", ConfigurationMask);
            encoder.WriteUInt16("ElementIndex", ElementIndex);
            encoder.WriteUInt16("ConnectionIndex", ConnectionIndex);
            encoder.WriteUInt16("GroupIndex", GroupIndex);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ConfigurationMask = decoder.ReadUInt32("ConfigurationMask");
            ElementIndex = decoder.ReadUInt16("ElementIndex");
            ConnectionIndex = decoder.ReadUInt16("ConnectionIndex");
            GroupIndex = decoder.ReadUInt16("GroupIndex");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            PubSubConfigurationRefDataType value = encodeable as PubSubConfigurationRefDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_configurationMask, value.m_configurationMask)) return false;
            if (!Utils.IsEqual(m_elementIndex, value.m_elementIndex)) return false;
            if (!Utils.IsEqual(m_connectionIndex, value.m_connectionIndex)) return false;
            if (!Utils.IsEqual(m_groupIndex, value.m_groupIndex)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (PubSubConfigurationRefDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PubSubConfigurationRefDataType clone = (PubSubConfigurationRefDataType)base.MemberwiseClone();

            clone.m_configurationMask = (uint)Utils.Clone(this.m_configurationMask);
            clone.m_elementIndex = (ushort)Utils.Clone(this.m_elementIndex);
            clone.m_connectionIndex = (ushort)Utils.Clone(this.m_connectionIndex);
            clone.m_groupIndex = (ushort)Utils.Clone(this.m_groupIndex);

            return clone;
        }
        #endregion

        #region Private Fields
        private uint m_configurationMask;
        private ushort m_elementIndex;
        private ushort m_connectionIndex;
        private ushort m_groupIndex;
        #endregion
    }

    #region PubSubConfigurationRefDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPubSubConfigurationRefDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PubSubConfigurationRefDataType")]
    #if !NET_STANDARD
    public partial class PubSubConfigurationRefDataTypeCollection : List<PubSubConfigurationRefDataType>, ICloneable
    #else
    public partial class PubSubConfigurationRefDataTypeCollection : List<PubSubConfigurationRefDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public PubSubConfigurationRefDataTypeCollection() {}

        /// <remarks />
        public PubSubConfigurationRefDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public PubSubConfigurationRefDataTypeCollection(IEnumerable<PubSubConfigurationRefDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator PubSubConfigurationRefDataTypeCollection(PubSubConfigurationRefDataType[] values)
        {
            if (values != null)
            {
                return new PubSubConfigurationRefDataTypeCollection(values);
            }

            return new PubSubConfigurationRefDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator PubSubConfigurationRefDataType[](PubSubConfigurationRefDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (PubSubConfigurationRefDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PubSubConfigurationRefDataTypeCollection clone = new PubSubConfigurationRefDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PubSubConfigurationRefDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PubSubConfigurationValueDataType Class
    #if (!OPCUA_EXCLUDE_PubSubConfigurationValueDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class PubSubConfigurationValueDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public PubSubConfigurationValueDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_configurationElement = new PubSubConfigurationRefDataType();
            m_name = null;
            m_identifier = Variant.Null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "ConfigurationElement", IsRequired = false, Order = 1)]
        public PubSubConfigurationRefDataType ConfigurationElement
        {
            get
            {
                return m_configurationElement;
            }

            set
            {
                m_configurationElement = value;

                if (value == null)
                {
                    m_configurationElement = new PubSubConfigurationRefDataType();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 2)]
        public string Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }

        /// <remarks />
        [DataMember(Name = "Identifier", IsRequired = false, Order = 3)]
        public Variant Identifier
        {
            get { return m_identifier;  }
            set { m_identifier = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.PubSubConfigurationValueDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.PubSubConfigurationValueDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.PubSubConfigurationValueDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.PubSubConfigurationValueDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("ConfigurationElement", ConfigurationElement, typeof(PubSubConfigurationRefDataType));
            encoder.WriteString("Name", Name);
            encoder.WriteVariant("Identifier", Identifier);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ConfigurationElement = (PubSubConfigurationRefDataType)decoder.ReadEncodeable("ConfigurationElement", typeof(PubSubConfigurationRefDataType));
            Name = decoder.ReadString("Name");
            Identifier = decoder.ReadVariant("Identifier");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            PubSubConfigurationValueDataType value = encodeable as PubSubConfigurationValueDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_configurationElement, value.m_configurationElement)) return false;
            if (!Utils.IsEqual(m_name, value.m_name)) return false;
            if (!Utils.IsEqual(m_identifier, value.m_identifier)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (PubSubConfigurationValueDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PubSubConfigurationValueDataType clone = (PubSubConfigurationValueDataType)base.MemberwiseClone();

            clone.m_configurationElement = (PubSubConfigurationRefDataType)Utils.Clone(this.m_configurationElement);
            clone.m_name = (string)Utils.Clone(this.m_name);
            clone.m_identifier = (Variant)Utils.Clone(this.m_identifier);

            return clone;
        }
        #endregion

        #region Private Fields
        private PubSubConfigurationRefDataType m_configurationElement;
        private string m_name;
        private Variant m_identifier;
        #endregion
    }

    #region PubSubConfigurationValueDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPubSubConfigurationValueDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PubSubConfigurationValueDataType")]
    #if !NET_STANDARD
    public partial class PubSubConfigurationValueDataTypeCollection : List<PubSubConfigurationValueDataType>, ICloneable
    #else
    public partial class PubSubConfigurationValueDataTypeCollection : List<PubSubConfigurationValueDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public PubSubConfigurationValueDataTypeCollection() {}

        /// <remarks />
        public PubSubConfigurationValueDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public PubSubConfigurationValueDataTypeCollection(IEnumerable<PubSubConfigurationValueDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator PubSubConfigurationValueDataTypeCollection(PubSubConfigurationValueDataType[] values)
        {
            if (values != null)
            {
                return new PubSubConfigurationValueDataTypeCollection(values);
            }

            return new PubSubConfigurationValueDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator PubSubConfigurationValueDataType[](PubSubConfigurationValueDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (PubSubConfigurationValueDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PubSubConfigurationValueDataTypeCollection clone = new PubSubConfigurationValueDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PubSubConfigurationValueDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DiagnosticsLevel Enumeration
    #if (!OPCUA_EXCLUDE_DiagnosticsLevel)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum DiagnosticsLevel
    {
        /// <remarks />
        [EnumMember(Value = "Basic_0")]
        Basic = 0,

        /// <remarks />
        [EnumMember(Value = "Advanced_1")]
        Advanced = 1,

        /// <remarks />
        [EnumMember(Value = "Info_2")]
        Info = 2,

        /// <remarks />
        [EnumMember(Value = "Log_3")]
        Log = 3,

        /// <remarks />
        [EnumMember(Value = "Debug_4")]
        Debug = 4,
    }

    #region DiagnosticsLevelCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDiagnosticsLevel", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DiagnosticsLevel")]
    #if !NET_STANDARD
    public partial class DiagnosticsLevelCollection : List<DiagnosticsLevel>, ICloneable
    #else
    public partial class DiagnosticsLevelCollection : List<DiagnosticsLevel>
    #endif
    {
        #region Constructors
        /// <remarks />
        public DiagnosticsLevelCollection() {}

        /// <remarks />
        public DiagnosticsLevelCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public DiagnosticsLevelCollection(IEnumerable<DiagnosticsLevel> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator DiagnosticsLevelCollection(DiagnosticsLevel[] values)
        {
            if (values != null)
            {
                return new DiagnosticsLevelCollection(values);
            }

            return new DiagnosticsLevelCollection();
        }

        /// <remarks />
        public static explicit operator DiagnosticsLevel[](DiagnosticsLevelCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (DiagnosticsLevelCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DiagnosticsLevelCollection clone = new DiagnosticsLevelCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DiagnosticsLevel)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PubSubDiagnosticsCounterClassification Enumeration
    #if (!OPCUA_EXCLUDE_PubSubDiagnosticsCounterClassification)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum PubSubDiagnosticsCounterClassification
    {
        /// <remarks />
        [EnumMember(Value = "Information_0")]
        Information = 0,

        /// <remarks />
        [EnumMember(Value = "Error_1")]
        Error = 1,
    }

    #region PubSubDiagnosticsCounterClassificationCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPubSubDiagnosticsCounterClassification", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PubSubDiagnosticsCounterClassification")]
    #if !NET_STANDARD
    public partial class PubSubDiagnosticsCounterClassificationCollection : List<PubSubDiagnosticsCounterClassification>, ICloneable
    #else
    public partial class PubSubDiagnosticsCounterClassificationCollection : List<PubSubDiagnosticsCounterClassification>
    #endif
    {
        #region Constructors
        /// <remarks />
        public PubSubDiagnosticsCounterClassificationCollection() {}

        /// <remarks />
        public PubSubDiagnosticsCounterClassificationCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public PubSubDiagnosticsCounterClassificationCollection(IEnumerable<PubSubDiagnosticsCounterClassification> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator PubSubDiagnosticsCounterClassificationCollection(PubSubDiagnosticsCounterClassification[] values)
        {
            if (values != null)
            {
                return new PubSubDiagnosticsCounterClassificationCollection(values);
            }

            return new PubSubDiagnosticsCounterClassificationCollection();
        }

        /// <remarks />
        public static explicit operator PubSubDiagnosticsCounterClassification[](PubSubDiagnosticsCounterClassificationCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (PubSubDiagnosticsCounterClassificationCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PubSubDiagnosticsCounterClassificationCollection clone = new PubSubDiagnosticsCounterClassificationCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PubSubDiagnosticsCounterClassification)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region AliasNameDataType Class
    #if (!OPCUA_EXCLUDE_AliasNameDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class AliasNameDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public AliasNameDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_aliasName = null;
            m_referencedNodes = new ExpandedNodeIdCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "AliasName", IsRequired = false, Order = 1)]
        public QualifiedName AliasName
        {
            get { return m_aliasName;  }
            set { m_aliasName = value; }
        }

        /// <remarks />
        [DataMember(Name = "ReferencedNodes", IsRequired = false, Order = 2)]
        public ExpandedNodeIdCollection ReferencedNodes
        {
            get
            {
                return m_referencedNodes;
            }

            set
            {
                m_referencedNodes = value;

                if (value == null)
                {
                    m_referencedNodes = new ExpandedNodeIdCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.AliasNameDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.AliasNameDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.AliasNameDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.AliasNameDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteQualifiedName("AliasName", AliasName);
            encoder.WriteExpandedNodeIdArray("ReferencedNodes", ReferencedNodes);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            AliasName = decoder.ReadQualifiedName("AliasName");
            ReferencedNodes = decoder.ReadExpandedNodeIdArray("ReferencedNodes");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            AliasNameDataType value = encodeable as AliasNameDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_aliasName, value.m_aliasName)) return false;
            if (!Utils.IsEqual(m_referencedNodes, value.m_referencedNodes)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (AliasNameDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            AliasNameDataType clone = (AliasNameDataType)base.MemberwiseClone();

            clone.m_aliasName = (QualifiedName)Utils.Clone(this.m_aliasName);
            clone.m_referencedNodes = (ExpandedNodeIdCollection)Utils.Clone(this.m_referencedNodes);

            return clone;
        }
        #endregion

        #region Private Fields
        private QualifiedName m_aliasName;
        private ExpandedNodeIdCollection m_referencedNodes;
        #endregion
    }

    #region AliasNameDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfAliasNameDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "AliasNameDataType")]
    #if !NET_STANDARD
    public partial class AliasNameDataTypeCollection : List<AliasNameDataType>, ICloneable
    #else
    public partial class AliasNameDataTypeCollection : List<AliasNameDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public AliasNameDataTypeCollection() {}

        /// <remarks />
        public AliasNameDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public AliasNameDataTypeCollection(IEnumerable<AliasNameDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator AliasNameDataTypeCollection(AliasNameDataType[] values)
        {
            if (values != null)
            {
                return new AliasNameDataTypeCollection(values);
            }

            return new AliasNameDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator AliasNameDataType[](AliasNameDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (AliasNameDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            AliasNameDataTypeCollection clone = new AliasNameDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((AliasNameDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PasswordOptionsMask Enumeration
    #if (!OPCUA_EXCLUDE_PasswordOptionsMask)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum PasswordOptionsMask : UInt32
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "SupportInitialPasswordChange_1")]
        SupportInitialPasswordChange = 1,

        /// <remarks />
        [EnumMember(Value = "SupportDisableUser_2")]
        SupportDisableUser = 2,

        /// <remarks />
        [EnumMember(Value = "SupportDisableDeleteForUser_4")]
        SupportDisableDeleteForUser = 4,

        /// <remarks />
        [EnumMember(Value = "SupportNoChangeForUser_8")]
        SupportNoChangeForUser = 8,

        /// <remarks />
        [EnumMember(Value = "SupportDescriptionForUser_16")]
        SupportDescriptionForUser = 16,

        /// <remarks />
        [EnumMember(Value = "RequiresUpperCaseCharacters_32")]
        RequiresUpperCaseCharacters = 32,

        /// <remarks />
        [EnumMember(Value = "RequiresLowerCaseCharacters_64")]
        RequiresLowerCaseCharacters = 64,

        /// <remarks />
        [EnumMember(Value = "RequiresDigitCharacters_128")]
        RequiresDigitCharacters = 128,

        /// <remarks />
        [EnumMember(Value = "RequiresSpecialCharacters_256")]
        RequiresSpecialCharacters = 256,
    }

    #region PasswordOptionsMaskCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPasswordOptionsMask", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PasswordOptionsMask")]
    #if !NET_STANDARD
    public partial class PasswordOptionsMaskCollection : List<PasswordOptionsMask>, ICloneable
    #else
    public partial class PasswordOptionsMaskCollection : List<PasswordOptionsMask>
    #endif
    {
        #region Constructors
        /// <remarks />
        public PasswordOptionsMaskCollection() {}

        /// <remarks />
        public PasswordOptionsMaskCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public PasswordOptionsMaskCollection(IEnumerable<PasswordOptionsMask> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator PasswordOptionsMaskCollection(PasswordOptionsMask[] values)
        {
            if (values != null)
            {
                return new PasswordOptionsMaskCollection(values);
            }

            return new PasswordOptionsMaskCollection();
        }

        /// <remarks />
        public static explicit operator PasswordOptionsMask[](PasswordOptionsMaskCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (PasswordOptionsMaskCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PasswordOptionsMaskCollection clone = new PasswordOptionsMaskCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PasswordOptionsMask)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region UserConfigurationMask Enumeration
    #if (!OPCUA_EXCLUDE_UserConfigurationMask)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum UserConfigurationMask : UInt32
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "NoDelete_1")]
        NoDelete = 1,

        /// <remarks />
        [EnumMember(Value = "Disabled_2")]
        Disabled = 2,

        /// <remarks />
        [EnumMember(Value = "NoChangeByUser_4")]
        NoChangeByUser = 4,

        /// <remarks />
        [EnumMember(Value = "MustChangePassword_8")]
        MustChangePassword = 8,
    }

    #region UserConfigurationMaskCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfUserConfigurationMask", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "UserConfigurationMask")]
    #if !NET_STANDARD
    public partial class UserConfigurationMaskCollection : List<UserConfigurationMask>, ICloneable
    #else
    public partial class UserConfigurationMaskCollection : List<UserConfigurationMask>
    #endif
    {
        #region Constructors
        /// <remarks />
        public UserConfigurationMaskCollection() {}

        /// <remarks />
        public UserConfigurationMaskCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public UserConfigurationMaskCollection(IEnumerable<UserConfigurationMask> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator UserConfigurationMaskCollection(UserConfigurationMask[] values)
        {
            if (values != null)
            {
                return new UserConfigurationMaskCollection(values);
            }

            return new UserConfigurationMaskCollection();
        }

        /// <remarks />
        public static explicit operator UserConfigurationMask[](UserConfigurationMaskCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (UserConfigurationMaskCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UserConfigurationMaskCollection clone = new UserConfigurationMaskCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((UserConfigurationMask)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region UserManagementDataType Class
    #if (!OPCUA_EXCLUDE_UserManagementDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class UserManagementDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public UserManagementDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_userName = null;
            m_userConfiguration = 0;
            m_description = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "UserName", IsRequired = false, Order = 1)]
        public string UserName
        {
            get { return m_userName;  }
            set { m_userName = value; }
        }

        /// <remarks />
        [DataMember(Name = "UserConfiguration", IsRequired = false, Order = 2)]
        public uint UserConfiguration
        {
            get { return m_userConfiguration;  }
            set { m_userConfiguration = value; }
        }

        /// <remarks />
        [DataMember(Name = "Description", IsRequired = false, Order = 3)]
        public string Description
        {
            get { return m_description;  }
            set { m_description = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.UserManagementDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.UserManagementDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.UserManagementDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.UserManagementDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("UserName", UserName);
            encoder.WriteUInt32("UserConfiguration", UserConfiguration);
            encoder.WriteString("Description", Description);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            UserName = decoder.ReadString("UserName");
            UserConfiguration = decoder.ReadUInt32("UserConfiguration");
            Description = decoder.ReadString("Description");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            UserManagementDataType value = encodeable as UserManagementDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_userName, value.m_userName)) return false;
            if (!Utils.IsEqual(m_userConfiguration, value.m_userConfiguration)) return false;
            if (!Utils.IsEqual(m_description, value.m_description)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (UserManagementDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UserManagementDataType clone = (UserManagementDataType)base.MemberwiseClone();

            clone.m_userName = (string)Utils.Clone(this.m_userName);
            clone.m_userConfiguration = (uint)Utils.Clone(this.m_userConfiguration);
            clone.m_description = (string)Utils.Clone(this.m_description);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_userName;
        private uint m_userConfiguration;
        private string m_description;
        #endregion
    }

    #region UserManagementDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfUserManagementDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "UserManagementDataType")]
    #if !NET_STANDARD
    public partial class UserManagementDataTypeCollection : List<UserManagementDataType>, ICloneable
    #else
    public partial class UserManagementDataTypeCollection : List<UserManagementDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public UserManagementDataTypeCollection() {}

        /// <remarks />
        public UserManagementDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public UserManagementDataTypeCollection(IEnumerable<UserManagementDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator UserManagementDataTypeCollection(UserManagementDataType[] values)
        {
            if (values != null)
            {
                return new UserManagementDataTypeCollection(values);
            }

            return new UserManagementDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator UserManagementDataType[](UserManagementDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (UserManagementDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UserManagementDataTypeCollection clone = new UserManagementDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((UserManagementDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region Duplex Enumeration
    #if (!OPCUA_EXCLUDE_Duplex)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum Duplex
    {
        /// <remarks />
        [EnumMember(Value = "Full_0")]
        Full = 0,

        /// <remarks />
        [EnumMember(Value = "Half_1")]
        Half = 1,

        /// <remarks />
        [EnumMember(Value = "Unknown_2")]
        Unknown = 2,
    }

    #region DuplexCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDuplex", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "Duplex")]
    #if !NET_STANDARD
    public partial class DuplexCollection : List<Duplex>, ICloneable
    #else
    public partial class DuplexCollection : List<Duplex>
    #endif
    {
        #region Constructors
        /// <remarks />
        public DuplexCollection() {}

        /// <remarks />
        public DuplexCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public DuplexCollection(IEnumerable<Duplex> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator DuplexCollection(Duplex[] values)
        {
            if (values != null)
            {
                return new DuplexCollection(values);
            }

            return new DuplexCollection();
        }

        /// <remarks />
        public static explicit operator Duplex[](DuplexCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (DuplexCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DuplexCollection clone = new DuplexCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((Duplex)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region InterfaceAdminStatus Enumeration
    #if (!OPCUA_EXCLUDE_InterfaceAdminStatus)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum InterfaceAdminStatus
    {
        /// <remarks />
        [EnumMember(Value = "Up_0")]
        Up = 0,

        /// <remarks />
        [EnumMember(Value = "Down_1")]
        Down = 1,

        /// <remarks />
        [EnumMember(Value = "Testing_2")]
        Testing = 2,
    }

    #region InterfaceAdminStatusCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfInterfaceAdminStatus", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "InterfaceAdminStatus")]
    #if !NET_STANDARD
    public partial class InterfaceAdminStatusCollection : List<InterfaceAdminStatus>, ICloneable
    #else
    public partial class InterfaceAdminStatusCollection : List<InterfaceAdminStatus>
    #endif
    {
        #region Constructors
        /// <remarks />
        public InterfaceAdminStatusCollection() {}

        /// <remarks />
        public InterfaceAdminStatusCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public InterfaceAdminStatusCollection(IEnumerable<InterfaceAdminStatus> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator InterfaceAdminStatusCollection(InterfaceAdminStatus[] values)
        {
            if (values != null)
            {
                return new InterfaceAdminStatusCollection(values);
            }

            return new InterfaceAdminStatusCollection();
        }

        /// <remarks />
        public static explicit operator InterfaceAdminStatus[](InterfaceAdminStatusCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (InterfaceAdminStatusCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            InterfaceAdminStatusCollection clone = new InterfaceAdminStatusCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((InterfaceAdminStatus)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region InterfaceOperStatus Enumeration
    #if (!OPCUA_EXCLUDE_InterfaceOperStatus)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum InterfaceOperStatus
    {
        /// <remarks />
        [EnumMember(Value = "Up_0")]
        Up = 0,

        /// <remarks />
        [EnumMember(Value = "Down_1")]
        Down = 1,

        /// <remarks />
        [EnumMember(Value = "Testing_2")]
        Testing = 2,

        /// <remarks />
        [EnumMember(Value = "Unknown_3")]
        Unknown = 3,

        /// <remarks />
        [EnumMember(Value = "Dormant_4")]
        Dormant = 4,

        /// <remarks />
        [EnumMember(Value = "NotPresent_5")]
        NotPresent = 5,

        /// <remarks />
        [EnumMember(Value = "LowerLayerDown_6")]
        LowerLayerDown = 6,
    }

    #region InterfaceOperStatusCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfInterfaceOperStatus", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "InterfaceOperStatus")]
    #if !NET_STANDARD
    public partial class InterfaceOperStatusCollection : List<InterfaceOperStatus>, ICloneable
    #else
    public partial class InterfaceOperStatusCollection : List<InterfaceOperStatus>
    #endif
    {
        #region Constructors
        /// <remarks />
        public InterfaceOperStatusCollection() {}

        /// <remarks />
        public InterfaceOperStatusCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public InterfaceOperStatusCollection(IEnumerable<InterfaceOperStatus> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator InterfaceOperStatusCollection(InterfaceOperStatus[] values)
        {
            if (values != null)
            {
                return new InterfaceOperStatusCollection(values);
            }

            return new InterfaceOperStatusCollection();
        }

        /// <remarks />
        public static explicit operator InterfaceOperStatus[](InterfaceOperStatusCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (InterfaceOperStatusCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            InterfaceOperStatusCollection clone = new InterfaceOperStatusCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((InterfaceOperStatus)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region NegotiationStatus Enumeration
    #if (!OPCUA_EXCLUDE_NegotiationStatus)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum NegotiationStatus
    {
        /// <remarks />
        [EnumMember(Value = "InProgress_0")]
        InProgress = 0,

        /// <remarks />
        [EnumMember(Value = "Complete_1")]
        Complete = 1,

        /// <remarks />
        [EnumMember(Value = "Failed_2")]
        Failed = 2,

        /// <remarks />
        [EnumMember(Value = "Unknown_3")]
        Unknown = 3,

        /// <remarks />
        [EnumMember(Value = "NoNegotiation_4")]
        NoNegotiation = 4,
    }

    #region NegotiationStatusCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfNegotiationStatus", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "NegotiationStatus")]
    #if !NET_STANDARD
    public partial class NegotiationStatusCollection : List<NegotiationStatus>, ICloneable
    #else
    public partial class NegotiationStatusCollection : List<NegotiationStatus>
    #endif
    {
        #region Constructors
        /// <remarks />
        public NegotiationStatusCollection() {}

        /// <remarks />
        public NegotiationStatusCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public NegotiationStatusCollection(IEnumerable<NegotiationStatus> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator NegotiationStatusCollection(NegotiationStatus[] values)
        {
            if (values != null)
            {
                return new NegotiationStatusCollection(values);
            }

            return new NegotiationStatusCollection();
        }

        /// <remarks />
        public static explicit operator NegotiationStatus[](NegotiationStatusCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (NegotiationStatusCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            NegotiationStatusCollection clone = new NegotiationStatusCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((NegotiationStatus)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region TsnFailureCode Enumeration
    #if (!OPCUA_EXCLUDE_TsnFailureCode)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum TsnFailureCode
    {
        /// <remarks />
        [EnumMember(Value = "NoFailure_0")]
        NoFailure = 0,

        /// <remarks />
        [EnumMember(Value = "InsufficientBandwidth_1")]
        InsufficientBandwidth = 1,

        /// <remarks />
        [EnumMember(Value = "InsufficientResources_2")]
        InsufficientResources = 2,

        /// <remarks />
        [EnumMember(Value = "InsufficientTrafficClassBandwidth_3")]
        InsufficientTrafficClassBandwidth = 3,

        /// <remarks />
        [EnumMember(Value = "StreamIdInUse_4")]
        StreamIdInUse = 4,

        /// <remarks />
        [EnumMember(Value = "StreamDestinationAddressInUse_5")]
        StreamDestinationAddressInUse = 5,

        /// <remarks />
        [EnumMember(Value = "StreamPreemptedByHigherRank_6")]
        StreamPreemptedByHigherRank = 6,

        /// <remarks />
        [EnumMember(Value = "LatencyHasChanged_7")]
        LatencyHasChanged = 7,

        /// <remarks />
        [EnumMember(Value = "EgressPortNotAvbCapable_8")]
        EgressPortNotAvbCapable = 8,

        /// <remarks />
        [EnumMember(Value = "UseDifferentDestinationAddress_9")]
        UseDifferentDestinationAddress = 9,

        /// <remarks />
        [EnumMember(Value = "OutOfMsrpResources_10")]
        OutOfMsrpResources = 10,

        /// <remarks />
        [EnumMember(Value = "OutOfMmrpResources_11")]
        OutOfMmrpResources = 11,

        /// <remarks />
        [EnumMember(Value = "CannotStoreDestinationAddress_12")]
        CannotStoreDestinationAddress = 12,

        /// <remarks />
        [EnumMember(Value = "PriorityIsNotAnSrcClass_13")]
        PriorityIsNotAnSrcClass = 13,

        /// <remarks />
        [EnumMember(Value = "MaxFrameSizeTooLarge_14")]
        MaxFrameSizeTooLarge = 14,

        /// <remarks />
        [EnumMember(Value = "MaxFanInPortsLimitReached_15")]
        MaxFanInPortsLimitReached = 15,

        /// <remarks />
        [EnumMember(Value = "FirstValueChangedForStreamId_16")]
        FirstValueChangedForStreamId = 16,

        /// <remarks />
        [EnumMember(Value = "VlanBlockedOnEgress_17")]
        VlanBlockedOnEgress = 17,

        /// <remarks />
        [EnumMember(Value = "VlanTaggingDisabledOnEgress_18")]
        VlanTaggingDisabledOnEgress = 18,

        /// <remarks />
        [EnumMember(Value = "SrClassPriorityMismatch_19")]
        SrClassPriorityMismatch = 19,

        /// <remarks />
        [EnumMember(Value = "FeatureNotPropagated_20")]
        FeatureNotPropagated = 20,

        /// <remarks />
        [EnumMember(Value = "MaxLatencyExceeded_21")]
        MaxLatencyExceeded = 21,

        /// <remarks />
        [EnumMember(Value = "BridgeDoesNotProvideNetworkId_22")]
        BridgeDoesNotProvideNetworkId = 22,

        /// <remarks />
        [EnumMember(Value = "StreamTransformNotSupported_23")]
        StreamTransformNotSupported = 23,

        /// <remarks />
        [EnumMember(Value = "StreamIdTypeNotSupported_24")]
        StreamIdTypeNotSupported = 24,

        /// <remarks />
        [EnumMember(Value = "FeatureNotSupported_25")]
        FeatureNotSupported = 25,
    }

    #region TsnFailureCodeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfTsnFailureCode", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "TsnFailureCode")]
    #if !NET_STANDARD
    public partial class TsnFailureCodeCollection : List<TsnFailureCode>, ICloneable
    #else
    public partial class TsnFailureCodeCollection : List<TsnFailureCode>
    #endif
    {
        #region Constructors
        /// <remarks />
        public TsnFailureCodeCollection() {}

        /// <remarks />
        public TsnFailureCodeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public TsnFailureCodeCollection(IEnumerable<TsnFailureCode> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator TsnFailureCodeCollection(TsnFailureCode[] values)
        {
            if (values != null)
            {
                return new TsnFailureCodeCollection(values);
            }

            return new TsnFailureCodeCollection();
        }

        /// <remarks />
        public static explicit operator TsnFailureCode[](TsnFailureCodeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (TsnFailureCodeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TsnFailureCodeCollection clone = new TsnFailureCodeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((TsnFailureCode)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region TsnStreamState Enumeration
    #if (!OPCUA_EXCLUDE_TsnStreamState)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum TsnStreamState
    {
        /// <remarks />
        [EnumMember(Value = "Disabled_0")]
        Disabled = 0,

        /// <remarks />
        [EnumMember(Value = "Configuring_1")]
        Configuring = 1,

        /// <remarks />
        [EnumMember(Value = "Ready_2")]
        Ready = 2,

        /// <remarks />
        [EnumMember(Value = "Operational_3")]
        Operational = 3,

        /// <remarks />
        [EnumMember(Value = "Error_4")]
        Error = 4,
    }

    #region TsnStreamStateCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfTsnStreamState", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "TsnStreamState")]
    #if !NET_STANDARD
    public partial class TsnStreamStateCollection : List<TsnStreamState>, ICloneable
    #else
    public partial class TsnStreamStateCollection : List<TsnStreamState>
    #endif
    {
        #region Constructors
        /// <remarks />
        public TsnStreamStateCollection() {}

        /// <remarks />
        public TsnStreamStateCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public TsnStreamStateCollection(IEnumerable<TsnStreamState> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator TsnStreamStateCollection(TsnStreamState[] values)
        {
            if (values != null)
            {
                return new TsnStreamStateCollection(values);
            }

            return new TsnStreamStateCollection();
        }

        /// <remarks />
        public static explicit operator TsnStreamState[](TsnStreamStateCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (TsnStreamStateCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TsnStreamStateCollection clone = new TsnStreamStateCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((TsnStreamState)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region TsnTalkerStatus Enumeration
    #if (!OPCUA_EXCLUDE_TsnTalkerStatus)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum TsnTalkerStatus
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "Ready_1")]
        Ready = 1,

        /// <remarks />
        [EnumMember(Value = "Failed_2")]
        Failed = 2,
    }

    #region TsnTalkerStatusCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfTsnTalkerStatus", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "TsnTalkerStatus")]
    #if !NET_STANDARD
    public partial class TsnTalkerStatusCollection : List<TsnTalkerStatus>, ICloneable
    #else
    public partial class TsnTalkerStatusCollection : List<TsnTalkerStatus>
    #endif
    {
        #region Constructors
        /// <remarks />
        public TsnTalkerStatusCollection() {}

        /// <remarks />
        public TsnTalkerStatusCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public TsnTalkerStatusCollection(IEnumerable<TsnTalkerStatus> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator TsnTalkerStatusCollection(TsnTalkerStatus[] values)
        {
            if (values != null)
            {
                return new TsnTalkerStatusCollection(values);
            }

            return new TsnTalkerStatusCollection();
        }

        /// <remarks />
        public static explicit operator TsnTalkerStatus[](TsnTalkerStatusCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (TsnTalkerStatusCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TsnTalkerStatusCollection clone = new TsnTalkerStatusCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((TsnTalkerStatus)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region TsnListenerStatus Enumeration
    #if (!OPCUA_EXCLUDE_TsnListenerStatus)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum TsnListenerStatus
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "Ready_1")]
        Ready = 1,

        /// <remarks />
        [EnumMember(Value = "PartialFailed_2")]
        PartialFailed = 2,

        /// <remarks />
        [EnumMember(Value = "Failed_3")]
        Failed = 3,
    }

    #region TsnListenerStatusCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfTsnListenerStatus", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "TsnListenerStatus")]
    #if !NET_STANDARD
    public partial class TsnListenerStatusCollection : List<TsnListenerStatus>, ICloneable
    #else
    public partial class TsnListenerStatusCollection : List<TsnListenerStatus>
    #endif
    {
        #region Constructors
        /// <remarks />
        public TsnListenerStatusCollection() {}

        /// <remarks />
        public TsnListenerStatusCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public TsnListenerStatusCollection(IEnumerable<TsnListenerStatus> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator TsnListenerStatusCollection(TsnListenerStatus[] values)
        {
            if (values != null)
            {
                return new TsnListenerStatusCollection(values);
            }

            return new TsnListenerStatusCollection();
        }

        /// <remarks />
        public static explicit operator TsnListenerStatus[](TsnListenerStatusCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (TsnListenerStatusCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TsnListenerStatusCollection clone = new TsnListenerStatusCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((TsnListenerStatus)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region PriorityMappingEntryType Class
    #if (!OPCUA_EXCLUDE_PriorityMappingEntryType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class PriorityMappingEntryType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public PriorityMappingEntryType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_mappingUri = null;
            m_priorityLabel = null;
            m_priorityValue_PCP = (byte)0;
            m_priorityValue_DSCP = (uint)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "MappingUri", IsRequired = false, Order = 1)]
        public string MappingUri
        {
            get { return m_mappingUri;  }
            set { m_mappingUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "PriorityLabel", IsRequired = false, Order = 2)]
        public string PriorityLabel
        {
            get { return m_priorityLabel;  }
            set { m_priorityLabel = value; }
        }

        /// <remarks />
        [DataMember(Name = "PriorityValue_PCP", IsRequired = false, Order = 3)]
        public byte PriorityValue_PCP
        {
            get { return m_priorityValue_PCP;  }
            set { m_priorityValue_PCP = value; }
        }

        /// <remarks />
        [DataMember(Name = "PriorityValue_DSCP", IsRequired = false, Order = 4)]
        public uint PriorityValue_DSCP
        {
            get { return m_priorityValue_DSCP;  }
            set { m_priorityValue_DSCP = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.PriorityMappingEntryType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.PriorityMappingEntryType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.PriorityMappingEntryType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.PriorityMappingEntryType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("MappingUri", MappingUri);
            encoder.WriteString("PriorityLabel", PriorityLabel);
            encoder.WriteByte("PriorityValue_PCP", PriorityValue_PCP);
            encoder.WriteUInt32("PriorityValue_DSCP", PriorityValue_DSCP);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            MappingUri = decoder.ReadString("MappingUri");
            PriorityLabel = decoder.ReadString("PriorityLabel");
            PriorityValue_PCP = decoder.ReadByte("PriorityValue_PCP");
            PriorityValue_DSCP = decoder.ReadUInt32("PriorityValue_DSCP");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            PriorityMappingEntryType value = encodeable as PriorityMappingEntryType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_mappingUri, value.m_mappingUri)) return false;
            if (!Utils.IsEqual(m_priorityLabel, value.m_priorityLabel)) return false;
            if (!Utils.IsEqual(m_priorityValue_PCP, value.m_priorityValue_PCP)) return false;
            if (!Utils.IsEqual(m_priorityValue_DSCP, value.m_priorityValue_DSCP)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (PriorityMappingEntryType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PriorityMappingEntryType clone = (PriorityMappingEntryType)base.MemberwiseClone();

            clone.m_mappingUri = (string)Utils.Clone(this.m_mappingUri);
            clone.m_priorityLabel = (string)Utils.Clone(this.m_priorityLabel);
            clone.m_priorityValue_PCP = (byte)Utils.Clone(this.m_priorityValue_PCP);
            clone.m_priorityValue_DSCP = (uint)Utils.Clone(this.m_priorityValue_DSCP);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_mappingUri;
        private string m_priorityLabel;
        private byte m_priorityValue_PCP;
        private uint m_priorityValue_DSCP;
        #endregion
    }

    #region PriorityMappingEntryTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfPriorityMappingEntryType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "PriorityMappingEntryType")]
    #if !NET_STANDARD
    public partial class PriorityMappingEntryTypeCollection : List<PriorityMappingEntryType>, ICloneable
    #else
    public partial class PriorityMappingEntryTypeCollection : List<PriorityMappingEntryType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public PriorityMappingEntryTypeCollection() {}

        /// <remarks />
        public PriorityMappingEntryTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public PriorityMappingEntryTypeCollection(IEnumerable<PriorityMappingEntryType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator PriorityMappingEntryTypeCollection(PriorityMappingEntryType[] values)
        {
            if (values != null)
            {
                return new PriorityMappingEntryTypeCollection(values);
            }

            return new PriorityMappingEntryTypeCollection();
        }

        /// <remarks />
        public static explicit operator PriorityMappingEntryType[](PriorityMappingEntryTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (PriorityMappingEntryTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            PriorityMappingEntryTypeCollection clone = new PriorityMappingEntryTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((PriorityMappingEntryType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region ReferenceDescriptionDataType Class
    #if (!OPCUA_EXCLUDE_ReferenceDescriptionDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ReferenceDescriptionDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public ReferenceDescriptionDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_sourceNode = null;
            m_referenceType = null;
            m_isForward = true;
            m_targetNode = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "SourceNode", IsRequired = false, Order = 1)]
        public NodeId SourceNode
        {
            get { return m_sourceNode;  }
            set { m_sourceNode = value; }
        }

        /// <remarks />
        [DataMember(Name = "ReferenceType", IsRequired = false, Order = 2)]
        public NodeId ReferenceType
        {
            get { return m_referenceType;  }
            set { m_referenceType = value; }
        }

        /// <remarks />
        [DataMember(Name = "IsForward", IsRequired = false, Order = 3)]
        public bool IsForward
        {
            get { return m_isForward;  }
            set { m_isForward = value; }
        }

        /// <remarks />
        [DataMember(Name = "TargetNode", IsRequired = false, Order = 4)]
        public ExpandedNodeId TargetNode
        {
            get { return m_targetNode;  }
            set { m_targetNode = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.ReferenceDescriptionDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.ReferenceDescriptionDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.ReferenceDescriptionDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.ReferenceDescriptionDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteNodeId("SourceNode", SourceNode);
            encoder.WriteNodeId("ReferenceType", ReferenceType);
            encoder.WriteBoolean("IsForward", IsForward);
            encoder.WriteExpandedNodeId("TargetNode", TargetNode);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            SourceNode = decoder.ReadNodeId("SourceNode");
            ReferenceType = decoder.ReadNodeId("ReferenceType");
            IsForward = decoder.ReadBoolean("IsForward");
            TargetNode = decoder.ReadExpandedNodeId("TargetNode");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ReferenceDescriptionDataType value = encodeable as ReferenceDescriptionDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_sourceNode, value.m_sourceNode)) return false;
            if (!Utils.IsEqual(m_referenceType, value.m_referenceType)) return false;
            if (!Utils.IsEqual(m_isForward, value.m_isForward)) return false;
            if (!Utils.IsEqual(m_targetNode, value.m_targetNode)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (ReferenceDescriptionDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReferenceDescriptionDataType clone = (ReferenceDescriptionDataType)base.MemberwiseClone();

            clone.m_sourceNode = (NodeId)Utils.Clone(this.m_sourceNode);
            clone.m_referenceType = (NodeId)Utils.Clone(this.m_referenceType);
            clone.m_isForward = (bool)Utils.Clone(this.m_isForward);
            clone.m_targetNode = (ExpandedNodeId)Utils.Clone(this.m_targetNode);

            return clone;
        }
        #endregion

        #region Private Fields
        private NodeId m_sourceNode;
        private NodeId m_referenceType;
        private bool m_isForward;
        private ExpandedNodeId m_targetNode;
        #endregion
    }

    #region ReferenceDescriptionDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfReferenceDescriptionDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ReferenceDescriptionDataType")]
    #if !NET_STANDARD
    public partial class ReferenceDescriptionDataTypeCollection : List<ReferenceDescriptionDataType>, ICloneable
    #else
    public partial class ReferenceDescriptionDataTypeCollection : List<ReferenceDescriptionDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public ReferenceDescriptionDataTypeCollection() {}

        /// <remarks />
        public ReferenceDescriptionDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public ReferenceDescriptionDataTypeCollection(IEnumerable<ReferenceDescriptionDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator ReferenceDescriptionDataTypeCollection(ReferenceDescriptionDataType[] values)
        {
            if (values != null)
            {
                return new ReferenceDescriptionDataTypeCollection(values);
            }

            return new ReferenceDescriptionDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator ReferenceDescriptionDataType[](ReferenceDescriptionDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (ReferenceDescriptionDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReferenceDescriptionDataTypeCollection clone = new ReferenceDescriptionDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ReferenceDescriptionDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region ReferenceListEntryDataType Class
    #if (!OPCUA_EXCLUDE_ReferenceListEntryDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ReferenceListEntryDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public ReferenceListEntryDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_referenceType = null;
            m_isForward = true;
            m_targetNode = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "ReferenceType", IsRequired = false, Order = 1)]
        public NodeId ReferenceType
        {
            get { return m_referenceType;  }
            set { m_referenceType = value; }
        }

        /// <remarks />
        [DataMember(Name = "IsForward", IsRequired = false, Order = 2)]
        public bool IsForward
        {
            get { return m_isForward;  }
            set { m_isForward = value; }
        }

        /// <remarks />
        [DataMember(Name = "TargetNode", IsRequired = false, Order = 3)]
        public ExpandedNodeId TargetNode
        {
            get { return m_targetNode;  }
            set { m_targetNode = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.ReferenceListEntryDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.ReferenceListEntryDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.ReferenceListEntryDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.ReferenceListEntryDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteNodeId("ReferenceType", ReferenceType);
            encoder.WriteBoolean("IsForward", IsForward);
            encoder.WriteExpandedNodeId("TargetNode", TargetNode);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ReferenceType = decoder.ReadNodeId("ReferenceType");
            IsForward = decoder.ReadBoolean("IsForward");
            TargetNode = decoder.ReadExpandedNodeId("TargetNode");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ReferenceListEntryDataType value = encodeable as ReferenceListEntryDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_referenceType, value.m_referenceType)) return false;
            if (!Utils.IsEqual(m_isForward, value.m_isForward)) return false;
            if (!Utils.IsEqual(m_targetNode, value.m_targetNode)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (ReferenceListEntryDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReferenceListEntryDataType clone = (ReferenceListEntryDataType)base.MemberwiseClone();

            clone.m_referenceType = (NodeId)Utils.Clone(this.m_referenceType);
            clone.m_isForward = (bool)Utils.Clone(this.m_isForward);
            clone.m_targetNode = (ExpandedNodeId)Utils.Clone(this.m_targetNode);

            return clone;
        }
        #endregion

        #region Private Fields
        private NodeId m_referenceType;
        private bool m_isForward;
        private ExpandedNodeId m_targetNode;
        #endregion
    }

    #region ReferenceListEntryDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfReferenceListEntryDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ReferenceListEntryDataType")]
    #if !NET_STANDARD
    public partial class ReferenceListEntryDataTypeCollection : List<ReferenceListEntryDataType>, ICloneable
    #else
    public partial class ReferenceListEntryDataTypeCollection : List<ReferenceListEntryDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public ReferenceListEntryDataTypeCollection() {}

        /// <remarks />
        public ReferenceListEntryDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public ReferenceListEntryDataTypeCollection(IEnumerable<ReferenceListEntryDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator ReferenceListEntryDataTypeCollection(ReferenceListEntryDataType[] values)
        {
            if (values != null)
            {
                return new ReferenceListEntryDataTypeCollection(values);
            }

            return new ReferenceListEntryDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator ReferenceListEntryDataType[](ReferenceListEntryDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (ReferenceListEntryDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReferenceListEntryDataTypeCollection clone = new ReferenceListEntryDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ReferenceListEntryDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region IdType Enumeration
    #if (!OPCUA_EXCLUDE_IdType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum IdType
    {
        /// <remarks />
        [EnumMember(Value = "Numeric_0")]
        Numeric = 0,

        /// <remarks />
        [EnumMember(Value = "String_1")]
        String = 1,

        /// <remarks />
        [EnumMember(Value = "Guid_2")]
        Guid = 2,

        /// <remarks />
        [EnumMember(Value = "Opaque_3")]
        Opaque = 3,
    }

    #region IdTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfIdType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "IdType")]
    #if !NET_STANDARD
    public partial class IdTypeCollection : List<IdType>, ICloneable
    #else
    public partial class IdTypeCollection : List<IdType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public IdTypeCollection() {}

        /// <remarks />
        public IdTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public IdTypeCollection(IEnumerable<IdType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator IdTypeCollection(IdType[] values)
        {
            if (values != null)
            {
                return new IdTypeCollection(values);
            }

            return new IdTypeCollection();
        }

        /// <remarks />
        public static explicit operator IdType[](IdTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (IdTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            IdTypeCollection clone = new IdTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((IdType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region NodeClass Enumeration
    #if (!OPCUA_EXCLUDE_NodeClass)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum NodeClass
    {
        /// <remarks />
        [EnumMember(Value = "Unspecified_0")]
        Unspecified = 0,

        /// <remarks />
        [EnumMember(Value = "Object_1")]
        Object = 1,

        /// <remarks />
        [EnumMember(Value = "Variable_2")]
        Variable = 2,

        /// <remarks />
        [EnumMember(Value = "Method_4")]
        Method = 4,

        /// <remarks />
        [EnumMember(Value = "ObjectType_8")]
        ObjectType = 8,

        /// <remarks />
        [EnumMember(Value = "VariableType_16")]
        VariableType = 16,

        /// <remarks />
        [EnumMember(Value = "ReferenceType_32")]
        ReferenceType = 32,

        /// <remarks />
        [EnumMember(Value = "DataType_64")]
        DataType = 64,

        /// <remarks />
        [EnumMember(Value = "View_128")]
        View = 128,
    }
    #endif
    #endregion

    #region PermissionType Enumeration
    #if (!OPCUA_EXCLUDE_PermissionType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum PermissionType : UInt32
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "Browse_1")]
        Browse = 1,

        /// <remarks />
        [EnumMember(Value = "ReadRolePermissions_2")]
        ReadRolePermissions = 2,

        /// <remarks />
        [EnumMember(Value = "WriteAttribute_4")]
        WriteAttribute = 4,

        /// <remarks />
        [EnumMember(Value = "WriteRolePermissions_8")]
        WriteRolePermissions = 8,

        /// <remarks />
        [EnumMember(Value = "WriteHistorizing_16")]
        WriteHistorizing = 16,

        /// <remarks />
        [EnumMember(Value = "Read_32")]
        Read = 32,

        /// <remarks />
        [EnumMember(Value = "Write_64")]
        Write = 64,

        /// <remarks />
        [EnumMember(Value = "ReadHistory_128")]
        ReadHistory = 128,

        /// <remarks />
        [EnumMember(Value = "InsertHistory_256")]
        InsertHistory = 256,

        /// <remarks />
        [EnumMember(Value = "ModifyHistory_512")]
        ModifyHistory = 512,

        /// <remarks />
        [EnumMember(Value = "DeleteHistory_1024")]
        DeleteHistory = 1024,

        /// <remarks />
        [EnumMember(Value = "ReceiveEvents_2048")]
        ReceiveEvents = 2048,

        /// <remarks />
        [EnumMember(Value = "Call_4096")]
        Call = 4096,

        /// <remarks />
        [EnumMember(Value = "AddReference_8192")]
        AddReference = 8192,

        /// <remarks />
        [EnumMember(Value = "RemoveReference_16384")]
        RemoveReference = 16384,

        /// <remarks />
        [EnumMember(Value = "DeleteNode_32768")]
        DeleteNode = 32768,

        /// <remarks />
        [EnumMember(Value = "AddNode_65536")]
        AddNode = 65536,
    }
    #endif
    #endregion

    #region AccessLevelType Enumeration
    #if (!OPCUA_EXCLUDE_AccessLevelType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum AccessLevelType : Byte
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "CurrentRead_1")]
        CurrentRead = 1,

        /// <remarks />
        [EnumMember(Value = "CurrentWrite_2")]
        CurrentWrite = 2,

        /// <remarks />
        [EnumMember(Value = "HistoryRead_4")]
        HistoryRead = 4,

        /// <remarks />
        [EnumMember(Value = "HistoryWrite_8")]
        HistoryWrite = 8,

        /// <remarks />
        [EnumMember(Value = "SemanticChange_16")]
        SemanticChange = 16,

        /// <remarks />
        [EnumMember(Value = "StatusWrite_32")]
        StatusWrite = 32,

        /// <remarks />
        [EnumMember(Value = "TimestampWrite_64")]
        TimestampWrite = 64,
    }
    #endif
    #endregion

    #region AccessLevelExType Enumeration
    #if (!OPCUA_EXCLUDE_AccessLevelExType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum AccessLevelExType : UInt32
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "CurrentRead_1")]
        CurrentRead = 1,

        /// <remarks />
        [EnumMember(Value = "CurrentWrite_2")]
        CurrentWrite = 2,

        /// <remarks />
        [EnumMember(Value = "HistoryRead_4")]
        HistoryRead = 4,

        /// <remarks />
        [EnumMember(Value = "HistoryWrite_8")]
        HistoryWrite = 8,

        /// <remarks />
        [EnumMember(Value = "SemanticChange_16")]
        SemanticChange = 16,

        /// <remarks />
        [EnumMember(Value = "StatusWrite_32")]
        StatusWrite = 32,

        /// <remarks />
        [EnumMember(Value = "TimestampWrite_64")]
        TimestampWrite = 64,

        /// <remarks />
        [EnumMember(Value = "NonatomicRead_256")]
        NonatomicRead = 256,

        /// <remarks />
        [EnumMember(Value = "NonatomicWrite_512")]
        NonatomicWrite = 512,

        /// <remarks />
        [EnumMember(Value = "WriteFullArrayOnly_1024")]
        WriteFullArrayOnly = 1024,

        /// <remarks />
        [EnumMember(Value = "NoSubDataTypes_2048")]
        NoSubDataTypes = 2048,

        /// <remarks />
        [EnumMember(Value = "NonVolatile_4096")]
        NonVolatile = 4096,

        /// <remarks />
        [EnumMember(Value = "Constant_8192")]
        Constant = 8192,
    }
    #endif
    #endregion

    #region EventNotifierType Enumeration
    #if (!OPCUA_EXCLUDE_EventNotifierType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum EventNotifierType : Byte
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "SubscribeToEvents_1")]
        SubscribeToEvents = 1,

        /// <remarks />
        [EnumMember(Value = "HistoryRead_4")]
        HistoryRead = 4,

        /// <remarks />
        [EnumMember(Value = "HistoryWrite_8")]
        HistoryWrite = 8,
    }
    #endif
    #endregion

    #region AccessRestrictionType Enumeration
    #if (!OPCUA_EXCLUDE_AccessRestrictionType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)][Flags]
    public enum AccessRestrictionType : UInt16
    {
        /// <remarks />
        [EnumMember(Value = "None_0")]
        None = 0,

        /// <remarks />
        [EnumMember(Value = "SigningRequired_1")]
        SigningRequired = 1,

        /// <remarks />
        [EnumMember(Value = "EncryptionRequired_2")]
        EncryptionRequired = 2,

        /// <remarks />
        [EnumMember(Value = "SessionRequired_4")]
        SessionRequired = 4,

        /// <remarks />
        [EnumMember(Value = "ApplyRestrictionsToBrowse_8")]
        ApplyRestrictionsToBrowse = 8,
    }
    #endif
    #endregion

    #region RolePermissionType Class
    #if (!OPCUA_EXCLUDE_RolePermissionType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class RolePermissionType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public RolePermissionType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_roleId = null;
            m_permissions = 0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "RoleId", IsRequired = false, Order = 1)]
        public NodeId RoleId
        {
            get { return m_roleId;  }
            set { m_roleId = value; }
        }

        /// <remarks />
        [DataMember(Name = "Permissions", IsRequired = false, Order = 2)]
        public uint Permissions
        {
            get { return m_permissions;  }
            set { m_permissions = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.RolePermissionType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.RolePermissionType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.RolePermissionType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.RolePermissionType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteNodeId("RoleId", RoleId);
            encoder.WriteUInt32("Permissions", Permissions);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            RoleId = decoder.ReadNodeId("RoleId");
            Permissions = decoder.ReadUInt32("Permissions");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            RolePermissionType value = encodeable as RolePermissionType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_roleId, value.m_roleId)) return false;
            if (!Utils.IsEqual(m_permissions, value.m_permissions)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (RolePermissionType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            RolePermissionType clone = (RolePermissionType)base.MemberwiseClone();

            clone.m_roleId = (NodeId)Utils.Clone(this.m_roleId);
            clone.m_permissions = (uint)Utils.Clone(this.m_permissions);

            return clone;
        }
        #endregion

        #region Private Fields
        private NodeId m_roleId;
        private uint m_permissions;
        #endregion
    }

    #region RolePermissionTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfRolePermissionType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "RolePermissionType")]
    #if !NET_STANDARD
    public partial class RolePermissionTypeCollection : List<RolePermissionType>, ICloneable
    #else
    public partial class RolePermissionTypeCollection : List<RolePermissionType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public RolePermissionTypeCollection() {}

        /// <remarks />
        public RolePermissionTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public RolePermissionTypeCollection(IEnumerable<RolePermissionType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator RolePermissionTypeCollection(RolePermissionType[] values)
        {
            if (values != null)
            {
                return new RolePermissionTypeCollection(values);
            }

            return new RolePermissionTypeCollection();
        }

        /// <remarks />
        public static explicit operator RolePermissionType[](RolePermissionTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (RolePermissionTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            RolePermissionTypeCollection clone = new RolePermissionTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((RolePermissionType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region DataTypeDefinition Class
    #if (!OPCUA_EXCLUDE_DataTypeDefinition)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DataTypeDefinition : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public DataTypeDefinition()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.DataTypeDefinition; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.DataTypeDefinition_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.DataTypeDefinition_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.DataTypeDefinition_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DataTypeDefinition value = encodeable as DataTypeDefinition;

            if (value == null)
            {
                return false;
            }


            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (DataTypeDefinition)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataTypeDefinition clone = (DataTypeDefinition)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }

    #region DataTypeDefinitionCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfDataTypeDefinition", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "DataTypeDefinition")]
    #if !NET_STANDARD
    public partial class DataTypeDefinitionCollection : List<DataTypeDefinition>, ICloneable
    #else
    public partial class DataTypeDefinitionCollection : List<DataTypeDefinition>
    #endif
    {
        #region Constructors
        /// <remarks />
        public DataTypeDefinitionCollection() {}

        /// <remarks />
        public DataTypeDefinitionCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public DataTypeDefinitionCollection(IEnumerable<DataTypeDefinition> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator DataTypeDefinitionCollection(DataTypeDefinition[] values)
        {
            if (values != null)
            {
                return new DataTypeDefinitionCollection(values);
            }

            return new DataTypeDefinitionCollection();
        }

        /// <remarks />
        public static explicit operator DataTypeDefinition[](DataTypeDefinitionCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (DataTypeDefinitionCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataTypeDefinitionCollection clone = new DataTypeDefinitionCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((DataTypeDefinition)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region StructureType Enumeration
    #if (!OPCUA_EXCLUDE_StructureType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum StructureType
    {
        /// <remarks />
        [EnumMember(Value = "Structure_0")]
        Structure = 0,

        /// <remarks />
        [EnumMember(Value = "StructureWithOptionalFields_1")]
        StructureWithOptionalFields = 1,

        /// <remarks />
        [EnumMember(Value = "Union_2")]
        Union = 2,

        /// <remarks />
        [EnumMember(Value = "StructureWithSubtypedValues_3")]
        StructureWithSubtypedValues = 3,

        /// <remarks />
        [EnumMember(Value = "UnionWithSubtypedValues_4")]
        UnionWithSubtypedValues = 4,
    }
    #endif
    #endregion

    #region StructureField Class
    #if (!OPCUA_EXCLUDE_StructureField)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class StructureField : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public StructureField()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_name = null;
            m_description = null;
            m_dataType = null;
            m_valueRank = (int)0;
            m_arrayDimensions = new UInt32Collection();
            m_maxStringLength = (uint)0;
            m_isOptional = false;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 1)]
        public string Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }

        /// <remarks />
        [DataMember(Name = "Description", IsRequired = false, Order = 2)]
        public LocalizedText Description
        {
            get { return m_description;  }
            set { m_description = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataType", IsRequired = false, Order = 3)]
        public NodeId DataType
        {
            get { return m_dataType;  }
            set { m_dataType = value; }
        }

        /// <remarks />
        [DataMember(Name = "ValueRank", IsRequired = false, Order = 4)]
        public int ValueRank
        {
            get { return m_valueRank;  }
            set { m_valueRank = value; }
        }

        /// <remarks />
        [DataMember(Name = "ArrayDimensions", IsRequired = false, Order = 5)]
        public UInt32Collection ArrayDimensions
        {
            get
            {
                return m_arrayDimensions;
            }

            set
            {
                m_arrayDimensions = value;

                if (value == null)
                {
                    m_arrayDimensions = new UInt32Collection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "MaxStringLength", IsRequired = false, Order = 6)]
        public uint MaxStringLength
        {
            get { return m_maxStringLength;  }
            set { m_maxStringLength = value; }
        }

        /// <remarks />
        [DataMember(Name = "IsOptional", IsRequired = false, Order = 7)]
        public bool IsOptional
        {
            get { return m_isOptional;  }
            set { m_isOptional = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.StructureField; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.StructureField_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.StructureField_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.StructureField_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("Name", Name);
            encoder.WriteLocalizedText("Description", Description);
            encoder.WriteNodeId("DataType", DataType);
            encoder.WriteInt32("ValueRank", ValueRank);
            encoder.WriteUInt32Array("ArrayDimensions", ArrayDimensions);
            encoder.WriteUInt32("MaxStringLength", MaxStringLength);
            encoder.WriteBoolean("IsOptional", IsOptional);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Name = decoder.ReadString("Name");
            Description = decoder.ReadLocalizedText("Description");
            DataType = decoder.ReadNodeId("DataType");
            ValueRank = decoder.ReadInt32("ValueRank");
            ArrayDimensions = decoder.ReadUInt32Array("ArrayDimensions");
            MaxStringLength = decoder.ReadUInt32("MaxStringLength");
            IsOptional = decoder.ReadBoolean("IsOptional");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            StructureField value = encodeable as StructureField;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_name, value.m_name)) return false;
            if (!Utils.IsEqual(m_description, value.m_description)) return false;
            if (!Utils.IsEqual(m_dataType, value.m_dataType)) return false;
            if (!Utils.IsEqual(m_valueRank, value.m_valueRank)) return false;
            if (!Utils.IsEqual(m_arrayDimensions, value.m_arrayDimensions)) return false;
            if (!Utils.IsEqual(m_maxStringLength, value.m_maxStringLength)) return false;
            if (!Utils.IsEqual(m_isOptional, value.m_isOptional)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (StructureField)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            StructureField clone = (StructureField)base.MemberwiseClone();

            clone.m_name = (string)Utils.Clone(this.m_name);
            clone.m_description = (LocalizedText)Utils.Clone(this.m_description);
            clone.m_dataType = (NodeId)Utils.Clone(this.m_dataType);
            clone.m_valueRank = (int)Utils.Clone(this.m_valueRank);
            clone.m_arrayDimensions = (UInt32Collection)Utils.Clone(this.m_arrayDimensions);
            clone.m_maxStringLength = (uint)Utils.Clone(this.m_maxStringLength);
            clone.m_isOptional = (bool)Utils.Clone(this.m_isOptional);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_name;
        private LocalizedText m_description;
        private NodeId m_dataType;
        private int m_valueRank;
        private UInt32Collection m_arrayDimensions;
        private uint m_maxStringLength;
        private bool m_isOptional;
        #endregion
    }

    #region StructureFieldCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfStructureField", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "StructureField")]
    #if !NET_STANDARD
    public partial class StructureFieldCollection : List<StructureField>, ICloneable
    #else
    public partial class StructureFieldCollection : List<StructureField>
    #endif
    {
        #region Constructors
        /// <remarks />
        public StructureFieldCollection() {}

        /// <remarks />
        public StructureFieldCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public StructureFieldCollection(IEnumerable<StructureField> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator StructureFieldCollection(StructureField[] values)
        {
            if (values != null)
            {
                return new StructureFieldCollection(values);
            }

            return new StructureFieldCollection();
        }

        /// <remarks />
        public static explicit operator StructureField[](StructureFieldCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (StructureFieldCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            StructureFieldCollection clone = new StructureFieldCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((StructureField)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region StructureDefinition Class
    #if (!OPCUA_EXCLUDE_StructureDefinition)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class StructureDefinition : DataTypeDefinition
    {
        #region Constructors
        /// <remarks />
        public StructureDefinition()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_defaultEncodingId = null;
            m_baseDataType = null;
            m_structureType = StructureType.Structure;
            m_fields = new StructureFieldCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "DefaultEncodingId", IsRequired = false, Order = 1)]
        public NodeId DefaultEncodingId
        {
            get { return m_defaultEncodingId;  }
            set { m_defaultEncodingId = value; }
        }

        /// <remarks />
        [DataMember(Name = "BaseDataType", IsRequired = false, Order = 2)]
        public NodeId BaseDataType
        {
            get { return m_baseDataType;  }
            set { m_baseDataType = value; }
        }

        /// <remarks />
        [DataMember(Name = "StructureType", IsRequired = false, Order = 3)]
        public StructureType StructureType
        {
            get { return m_structureType;  }
            set { m_structureType = value; }
        }

        /// <remarks />
        [DataMember(Name = "Fields", IsRequired = false, Order = 4)]
        public StructureFieldCollection Fields
        {
            get
            {
                return m_fields;
            }

            set
            {
                m_fields = value;

                if (value == null)
                {
                    m_fields = new StructureFieldCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.StructureDefinition; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.StructureDefinition_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.StructureDefinition_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.StructureDefinition_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteNodeId("DefaultEncodingId", DefaultEncodingId);
            encoder.WriteNodeId("BaseDataType", BaseDataType);
            encoder.WriteEnumerated("StructureType", StructureType);
            encoder.WriteEncodeableArray("Fields", Fields.ToArray(), typeof(StructureField));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            DefaultEncodingId = decoder.ReadNodeId("DefaultEncodingId");
            BaseDataType = decoder.ReadNodeId("BaseDataType");
            StructureType = (StructureType)decoder.ReadEnumerated("StructureType", typeof(StructureType));
            Fields = (StructureFieldCollection)decoder.ReadEncodeableArray("Fields", typeof(StructureField));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            StructureDefinition value = encodeable as StructureDefinition;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_defaultEncodingId, value.m_defaultEncodingId)) return false;
            if (!Utils.IsEqual(m_baseDataType, value.m_baseDataType)) return false;
            if (!Utils.IsEqual(m_structureType, value.m_structureType)) return false;
            if (!Utils.IsEqual(m_fields, value.m_fields)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (StructureDefinition)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            StructureDefinition clone = (StructureDefinition)base.MemberwiseClone();

            clone.m_defaultEncodingId = (NodeId)Utils.Clone(this.m_defaultEncodingId);
            clone.m_baseDataType = (NodeId)Utils.Clone(this.m_baseDataType);
            clone.m_structureType = (StructureType)Utils.Clone(this.m_structureType);
            clone.m_fields = (StructureFieldCollection)Utils.Clone(this.m_fields);

            return clone;
        }
        #endregion

        #region Private Fields
        private NodeId m_defaultEncodingId;
        private NodeId m_baseDataType;
        private StructureType m_structureType;
        private StructureFieldCollection m_fields;
        #endregion
    }

    #region StructureDefinitionCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfStructureDefinition", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "StructureDefinition")]
    #if !NET_STANDARD
    public partial class StructureDefinitionCollection : List<StructureDefinition>, ICloneable
    #else
    public partial class StructureDefinitionCollection : List<StructureDefinition>
    #endif
    {
        #region Constructors
        /// <remarks />
        public StructureDefinitionCollection() {}

        /// <remarks />
        public StructureDefinitionCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public StructureDefinitionCollection(IEnumerable<StructureDefinition> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator StructureDefinitionCollection(StructureDefinition[] values)
        {
            if (values != null)
            {
                return new StructureDefinitionCollection(values);
            }

            return new StructureDefinitionCollection();
        }

        /// <remarks />
        public static explicit operator StructureDefinition[](StructureDefinitionCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (StructureDefinitionCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            StructureDefinitionCollection clone = new StructureDefinitionCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((StructureDefinition)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region EnumDefinition Class
    #if (!OPCUA_EXCLUDE_EnumDefinition)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class EnumDefinition : DataTypeDefinition
    {
        #region Constructors
        /// <remarks />
        public EnumDefinition()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_fields = new EnumFieldCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Fields", IsRequired = false, Order = 1)]
        public EnumFieldCollection Fields
        {
            get
            {
                return m_fields;
            }

            set
            {
                m_fields = value;

                if (value == null)
                {
                    m_fields = new EnumFieldCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.EnumDefinition; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.EnumDefinition_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.EnumDefinition_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.EnumDefinition_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeableArray("Fields", Fields.ToArray(), typeof(EnumField));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Fields = (EnumFieldCollection)decoder.ReadEncodeableArray("Fields", typeof(EnumField));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            EnumDefinition value = encodeable as EnumDefinition;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_fields, value.m_fields)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (EnumDefinition)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EnumDefinition clone = (EnumDefinition)base.MemberwiseClone();

            clone.m_fields = (EnumFieldCollection)Utils.Clone(this.m_fields);

            return clone;
        }
        #endregion

        #region Private Fields
        private EnumFieldCollection m_fields;
        #endregion
    }

    #region EnumDefinitionCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfEnumDefinition", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "EnumDefinition")]
    #if !NET_STANDARD
    public partial class EnumDefinitionCollection : List<EnumDefinition>, ICloneable
    #else
    public partial class EnumDefinitionCollection : List<EnumDefinition>
    #endif
    {
        #region Constructors
        /// <remarks />
        public EnumDefinitionCollection() {}

        /// <remarks />
        public EnumDefinitionCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public EnumDefinitionCollection(IEnumerable<EnumDefinition> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator EnumDefinitionCollection(EnumDefinition[] values)
        {
            if (values != null)
            {
                return new EnumDefinitionCollection(values);
            }

            return new EnumDefinitionCollection();
        }

        /// <remarks />
        public static explicit operator EnumDefinition[](EnumDefinitionCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (EnumDefinitionCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EnumDefinitionCollection clone = new EnumDefinitionCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((EnumDefinition)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region Node Class
    #if (!OPCUA_EXCLUDE_Node)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class Node : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public Node()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_nodeId = null;
            m_nodeClass = NodeClass.Unspecified;
            m_browseName = null;
            m_displayName = null;
            m_description = null;
            m_writeMask = (uint)0;
            m_userWriteMask = (uint)0;
            m_rolePermissions = new RolePermissionTypeCollection();
            m_userRolePermissions = new RolePermissionTypeCollection();
            m_accessRestrictions = (ushort)0;
            m_references = new ReferenceNodeCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "NodeId", IsRequired = false, Order = 1)]
        public NodeId NodeId
        {
            get { return m_nodeId;  }
            set { m_nodeId = value; }
        }

        /// <remarks />
        [DataMember(Name = "NodeClass", IsRequired = false, Order = 2)]
        public NodeClass NodeClass
        {
            get { return m_nodeClass;  }
            set { m_nodeClass = value; }
        }

        /// <remarks />
        [DataMember(Name = "BrowseName", IsRequired = false, Order = 3)]
        public QualifiedName BrowseName
        {
            get { return m_browseName;  }
            set { m_browseName = value; }
        }

        /// <remarks />
        [DataMember(Name = "DisplayName", IsRequired = false, Order = 4)]
        public LocalizedText DisplayName
        {
            get { return m_displayName;  }
            set { m_displayName = value; }
        }

        /// <remarks />
        [DataMember(Name = "Description", IsRequired = false, Order = 5)]
        public LocalizedText Description
        {
            get { return m_description;  }
            set { m_description = value; }
        }

        /// <remarks />
        [DataMember(Name = "WriteMask", IsRequired = false, Order = 6)]
        public uint WriteMask
        {
            get { return m_writeMask;  }
            set { m_writeMask = value; }
        }

        /// <remarks />
        [DataMember(Name = "UserWriteMask", IsRequired = false, Order = 7)]
        public uint UserWriteMask
        {
            get { return m_userWriteMask;  }
            set { m_userWriteMask = value; }
        }

        /// <remarks />
        [DataMember(Name = "RolePermissions", IsRequired = false, Order = 8)]
        public RolePermissionTypeCollection RolePermissions
        {
            get
            {
                return m_rolePermissions;
            }

            set
            {
                m_rolePermissions = value;

                if (value == null)
                {
                    m_rolePermissions = new RolePermissionTypeCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "UserRolePermissions", IsRequired = false, Order = 9)]
        public RolePermissionTypeCollection UserRolePermissions
        {
            get
            {
                return m_userRolePermissions;
            }

            set
            {
                m_userRolePermissions = value;

                if (value == null)
                {
                    m_userRolePermissions = new RolePermissionTypeCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "AccessRestrictions", IsRequired = false, Order = 10)]
        public ushort AccessRestrictions
        {
            get { return m_accessRestrictions;  }
            set { m_accessRestrictions = value; }
        }

        /// <remarks />
        [DataMember(Name = "References", IsRequired = false, Order = 11)]
        public ReferenceNodeCollection References
        {
            get
            {
                return m_references;
            }

            set
            {
                m_references = value;

                if (value == null)
                {
                    m_references = new ReferenceNodeCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.Node; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.Node_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.Node_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.Node_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteNodeId("NodeId", NodeId);
            encoder.WriteEnumerated("NodeClass", NodeClass);
            encoder.WriteQualifiedName("BrowseName", BrowseName);
            encoder.WriteLocalizedText("DisplayName", DisplayName);
            encoder.WriteLocalizedText("Description", Description);
            encoder.WriteUInt32("WriteMask", WriteMask);
            encoder.WriteUInt32("UserWriteMask", UserWriteMask);
            encoder.WriteEncodeableArray("RolePermissions", RolePermissions.ToArray(), typeof(RolePermissionType));
            encoder.WriteEncodeableArray("UserRolePermissions", UserRolePermissions.ToArray(), typeof(RolePermissionType));
            encoder.WriteUInt16("AccessRestrictions", AccessRestrictions);
            encoder.WriteEncodeableArray("References", References.ToArray(), typeof(ReferenceNode));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            NodeId = decoder.ReadNodeId("NodeId");
            NodeClass = (NodeClass)decoder.ReadEnumerated("NodeClass", typeof(NodeClass));
            BrowseName = decoder.ReadQualifiedName("BrowseName");
            DisplayName = decoder.ReadLocalizedText("DisplayName");
            Description = decoder.ReadLocalizedText("Description");
            WriteMask = decoder.ReadUInt32("WriteMask");
            UserWriteMask = decoder.ReadUInt32("UserWriteMask");
            RolePermissions = (RolePermissionTypeCollection)decoder.ReadEncodeableArray("RolePermissions", typeof(RolePermissionType));
            UserRolePermissions = (RolePermissionTypeCollection)decoder.ReadEncodeableArray("UserRolePermissions", typeof(RolePermissionType));
            AccessRestrictions = decoder.ReadUInt16("AccessRestrictions");
            References = (ReferenceNodeCollection)decoder.ReadEncodeableArray("References", typeof(ReferenceNode));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            Node value = encodeable as Node;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_nodeId, value.m_nodeId)) return false;
            if (!Utils.IsEqual(m_nodeClass, value.m_nodeClass)) return false;
            if (!Utils.IsEqual(m_browseName, value.m_browseName)) return false;
            if (!Utils.IsEqual(m_displayName, value.m_displayName)) return false;
            if (!Utils.IsEqual(m_description, value.m_description)) return false;
            if (!Utils.IsEqual(m_writeMask, value.m_writeMask)) return false;
            if (!Utils.IsEqual(m_userWriteMask, value.m_userWriteMask)) return false;
            if (!Utils.IsEqual(m_rolePermissions, value.m_rolePermissions)) return false;
            if (!Utils.IsEqual(m_userRolePermissions, value.m_userRolePermissions)) return false;
            if (!Utils.IsEqual(m_accessRestrictions, value.m_accessRestrictions)) return false;
            if (!Utils.IsEqual(m_references, value.m_references)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (Node)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            Node clone = (Node)base.MemberwiseClone();

            clone.m_nodeId = (NodeId)Utils.Clone(this.m_nodeId);
            clone.m_nodeClass = (NodeClass)Utils.Clone(this.m_nodeClass);
            clone.m_browseName = (QualifiedName)Utils.Clone(this.m_browseName);
            clone.m_displayName = (LocalizedText)Utils.Clone(this.m_displayName);
            clone.m_description = (LocalizedText)Utils.Clone(this.m_description);
            clone.m_writeMask = (uint)Utils.Clone(this.m_writeMask);
            clone.m_userWriteMask = (uint)Utils.Clone(this.m_userWriteMask);
            clone.m_rolePermissions = (RolePermissionTypeCollection)Utils.Clone(this.m_rolePermissions);
            clone.m_userRolePermissions = (RolePermissionTypeCollection)Utils.Clone(this.m_userRolePermissions);
            clone.m_accessRestrictions = (ushort)Utils.Clone(this.m_accessRestrictions);
            clone.m_references = (ReferenceNodeCollection)Utils.Clone(this.m_references);

            return clone;
        }
        #endregion

        #region Private Fields
        private NodeId m_nodeId;
        private NodeClass m_nodeClass;
        private QualifiedName m_browseName;
        private LocalizedText m_displayName;
        private LocalizedText m_description;
        private uint m_writeMask;
        private uint m_userWriteMask;
        private RolePermissionTypeCollection m_rolePermissions;
        private RolePermissionTypeCollection m_userRolePermissions;
        private ushort m_accessRestrictions;
        private ReferenceNodeCollection m_references;
        #endregion
    }

    #region NodeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfNode", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "Node")]
    #if !NET_STANDARD
    public partial class NodeCollection : List<Node>, ICloneable
    #else
    public partial class NodeCollection : List<Node>
    #endif
    {
        #region Constructors
        /// <remarks />
        public NodeCollection() {}

        /// <remarks />
        public NodeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public NodeCollection(IEnumerable<Node> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator NodeCollection(Node[] values)
        {
            if (values != null)
            {
                return new NodeCollection(values);
            }

            return new NodeCollection();
        }

        /// <remarks />
        public static explicit operator Node[](NodeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (NodeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            NodeCollection clone = new NodeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((Node)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region InstanceNode Class
    #if (!OPCUA_EXCLUDE_InstanceNode)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class InstanceNode : Node
    {
        #region Constructors
        /// <remarks />
        public InstanceNode()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.InstanceNode; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.InstanceNode_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.InstanceNode_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.InstanceNode_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            InstanceNode value = encodeable as InstanceNode;

            if (value == null)
            {
                return false;
            }


            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (InstanceNode)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            InstanceNode clone = (InstanceNode)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region TypeNode Class
    #if (!OPCUA_EXCLUDE_TypeNode)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class TypeNode : Node
    {
        #region Constructors
        /// <remarks />
        public TypeNode()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
        }
        #endregion

        #region Public Properties
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.TypeNode; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.TypeNode_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.TypeNode_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.TypeNode_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);


            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            TypeNode value = encodeable as TypeNode;

            if (value == null)
            {
                return false;
            }


            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (TypeNode)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TypeNode clone = (TypeNode)base.MemberwiseClone();


            return clone;
        }
        #endregion

        #region Private Fields
        #endregion
    }
    #endif
    #endregion

    #region ObjectNode Class
    #if (!OPCUA_EXCLUDE_ObjectNode)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ObjectNode : InstanceNode
    {
        #region Constructors
        /// <remarks />
        public ObjectNode()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_eventNotifier = (byte)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "EventNotifier", IsRequired = false, Order = 1)]
        public byte EventNotifier
        {
            get { return m_eventNotifier;  }
            set { m_eventNotifier = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.ObjectNode; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.ObjectNode_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.ObjectNode_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.ObjectNode_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteByte("EventNotifier", EventNotifier);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            EventNotifier = decoder.ReadByte("EventNotifier");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ObjectNode value = encodeable as ObjectNode;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_eventNotifier, value.m_eventNotifier)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (ObjectNode)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ObjectNode clone = (ObjectNode)base.MemberwiseClone();

            clone.m_eventNotifier = (byte)Utils.Clone(this.m_eventNotifier);

            return clone;
        }
        #endregion

        #region Private Fields
        private byte m_eventNotifier;
        #endregion
    }
    #endif
    #endregion

    #region ObjectTypeNode Class
    #if (!OPCUA_EXCLUDE_ObjectTypeNode)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ObjectTypeNode : TypeNode
    {
        #region Constructors
        /// <remarks />
        public ObjectTypeNode()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_isAbstract = true;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "IsAbstract", IsRequired = false, Order = 1)]
        public bool IsAbstract
        {
            get { return m_isAbstract;  }
            set { m_isAbstract = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.ObjectTypeNode; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.ObjectTypeNode_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.ObjectTypeNode_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.ObjectTypeNode_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteBoolean("IsAbstract", IsAbstract);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            IsAbstract = decoder.ReadBoolean("IsAbstract");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ObjectTypeNode value = encodeable as ObjectTypeNode;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_isAbstract, value.m_isAbstract)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (ObjectTypeNode)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ObjectTypeNode clone = (ObjectTypeNode)base.MemberwiseClone();

            clone.m_isAbstract = (bool)Utils.Clone(this.m_isAbstract);

            return clone;
        }
        #endregion

        #region Private Fields
        private bool m_isAbstract;
        #endregion
    }
    #endif
    #endregion

    #region VariableNode Class
    #if (!OPCUA_EXCLUDE_VariableNode)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class VariableNode : InstanceNode
    {
        #region Constructors
        /// <remarks />
        public VariableNode()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_value = Variant.Null;
            m_dataType = null;
            m_valueRank = (int)0;
            m_arrayDimensions = new UInt32Collection();
            m_accessLevel = (byte)0;
            m_userAccessLevel = (byte)0;
            m_minimumSamplingInterval = (double)0;
            m_historizing = true;
            m_accessLevelEx = (uint)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Value", IsRequired = false, Order = 1)]
        public Variant Value
        {
            get { return m_value;  }
            set { m_value = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataType", IsRequired = false, Order = 2)]
        public NodeId DataType
        {
            get { return m_dataType;  }
            set { m_dataType = value; }
        }

        /// <remarks />
        [DataMember(Name = "ValueRank", IsRequired = false, Order = 3)]
        public int ValueRank
        {
            get { return m_valueRank;  }
            set { m_valueRank = value; }
        }

        /// <remarks />
        [DataMember(Name = "ArrayDimensions", IsRequired = false, Order = 4)]
        public UInt32Collection ArrayDimensions
        {
            get
            {
                return m_arrayDimensions;
            }

            set
            {
                m_arrayDimensions = value;

                if (value == null)
                {
                    m_arrayDimensions = new UInt32Collection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "AccessLevel", IsRequired = false, Order = 5)]
        public byte AccessLevel
        {
            get { return m_accessLevel;  }
            set { m_accessLevel = value; }
        }

        /// <remarks />
        [DataMember(Name = "UserAccessLevel", IsRequired = false, Order = 6)]
        public byte UserAccessLevel
        {
            get { return m_userAccessLevel;  }
            set { m_userAccessLevel = value; }
        }

        /// <remarks />
        [DataMember(Name = "MinimumSamplingInterval", IsRequired = false, Order = 7)]
        public double MinimumSamplingInterval
        {
            get { return m_minimumSamplingInterval;  }
            set { m_minimumSamplingInterval = value; }
        }

        /// <remarks />
        [DataMember(Name = "Historizing", IsRequired = false, Order = 8)]
        public bool Historizing
        {
            get { return m_historizing;  }
            set { m_historizing = value; }
        }

        /// <remarks />
        [DataMember(Name = "AccessLevelEx", IsRequired = false, Order = 9)]
        public uint AccessLevelEx
        {
            get { return m_accessLevelEx;  }
            set { m_accessLevelEx = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.VariableNode; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.VariableNode_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.VariableNode_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.VariableNode_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteVariant("Value", Value);
            encoder.WriteNodeId("DataType", DataType);
            encoder.WriteInt32("ValueRank", ValueRank);
            encoder.WriteUInt32Array("ArrayDimensions", ArrayDimensions);
            encoder.WriteByte("AccessLevel", AccessLevel);
            encoder.WriteByte("UserAccessLevel", UserAccessLevel);
            encoder.WriteDouble("MinimumSamplingInterval", MinimumSamplingInterval);
            encoder.WriteBoolean("Historizing", Historizing);
            encoder.WriteUInt32("AccessLevelEx", AccessLevelEx);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Value = decoder.ReadVariant("Value");
            DataType = decoder.ReadNodeId("DataType");
            ValueRank = decoder.ReadInt32("ValueRank");
            ArrayDimensions = decoder.ReadUInt32Array("ArrayDimensions");
            AccessLevel = decoder.ReadByte("AccessLevel");
            UserAccessLevel = decoder.ReadByte("UserAccessLevel");
            MinimumSamplingInterval = decoder.ReadDouble("MinimumSamplingInterval");
            Historizing = decoder.ReadBoolean("Historizing");
            AccessLevelEx = decoder.ReadUInt32("AccessLevelEx");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            VariableNode value = encodeable as VariableNode;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_value, value.m_value)) return false;
            if (!Utils.IsEqual(m_dataType, value.m_dataType)) return false;
            if (!Utils.IsEqual(m_valueRank, value.m_valueRank)) return false;
            if (!Utils.IsEqual(m_arrayDimensions, value.m_arrayDimensions)) return false;
            if (!Utils.IsEqual(m_accessLevel, value.m_accessLevel)) return false;
            if (!Utils.IsEqual(m_userAccessLevel, value.m_userAccessLevel)) return false;
            if (!Utils.IsEqual(m_minimumSamplingInterval, value.m_minimumSamplingInterval)) return false;
            if (!Utils.IsEqual(m_historizing, value.m_historizing)) return false;
            if (!Utils.IsEqual(m_accessLevelEx, value.m_accessLevelEx)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (VariableNode)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            VariableNode clone = (VariableNode)base.MemberwiseClone();

            clone.m_value = (Variant)Utils.Clone(this.m_value);
            clone.m_dataType = (NodeId)Utils.Clone(this.m_dataType);
            clone.m_valueRank = (int)Utils.Clone(this.m_valueRank);
            clone.m_arrayDimensions = (UInt32Collection)Utils.Clone(this.m_arrayDimensions);
            clone.m_accessLevel = (byte)Utils.Clone(this.m_accessLevel);
            clone.m_userAccessLevel = (byte)Utils.Clone(this.m_userAccessLevel);
            clone.m_minimumSamplingInterval = (double)Utils.Clone(this.m_minimumSamplingInterval);
            clone.m_historizing = (bool)Utils.Clone(this.m_historizing);
            clone.m_accessLevelEx = (uint)Utils.Clone(this.m_accessLevelEx);

            return clone;
        }
        #endregion

        #region Private Fields
        private Variant m_value;
        private NodeId m_dataType;
        private int m_valueRank;
        private UInt32Collection m_arrayDimensions;
        private byte m_accessLevel;
        private byte m_userAccessLevel;
        private double m_minimumSamplingInterval;
        private bool m_historizing;
        private uint m_accessLevelEx;
        #endregion
    }
    #endif
    #endregion

    #region VariableTypeNode Class
    #if (!OPCUA_EXCLUDE_VariableTypeNode)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class VariableTypeNode : TypeNode
    {
        #region Constructors
        /// <remarks />
        public VariableTypeNode()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_value = Variant.Null;
            m_dataType = null;
            m_valueRank = (int)0;
            m_arrayDimensions = new UInt32Collection();
            m_isAbstract = true;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Value", IsRequired = false, Order = 1)]
        public Variant Value
        {
            get { return m_value;  }
            set { m_value = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataType", IsRequired = false, Order = 2)]
        public NodeId DataType
        {
            get { return m_dataType;  }
            set { m_dataType = value; }
        }

        /// <remarks />
        [DataMember(Name = "ValueRank", IsRequired = false, Order = 3)]
        public int ValueRank
        {
            get { return m_valueRank;  }
            set { m_valueRank = value; }
        }

        /// <remarks />
        [DataMember(Name = "ArrayDimensions", IsRequired = false, Order = 4)]
        public UInt32Collection ArrayDimensions
        {
            get
            {
                return m_arrayDimensions;
            }

            set
            {
                m_arrayDimensions = value;

                if (value == null)
                {
                    m_arrayDimensions = new UInt32Collection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "IsAbstract", IsRequired = false, Order = 5)]
        public bool IsAbstract
        {
            get { return m_isAbstract;  }
            set { m_isAbstract = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.VariableTypeNode; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.VariableTypeNode_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.VariableTypeNode_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.VariableTypeNode_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteVariant("Value", Value);
            encoder.WriteNodeId("DataType", DataType);
            encoder.WriteInt32("ValueRank", ValueRank);
            encoder.WriteUInt32Array("ArrayDimensions", ArrayDimensions);
            encoder.WriteBoolean("IsAbstract", IsAbstract);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Value = decoder.ReadVariant("Value");
            DataType = decoder.ReadNodeId("DataType");
            ValueRank = decoder.ReadInt32("ValueRank");
            ArrayDimensions = decoder.ReadUInt32Array("ArrayDimensions");
            IsAbstract = decoder.ReadBoolean("IsAbstract");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            VariableTypeNode value = encodeable as VariableTypeNode;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_value, value.m_value)) return false;
            if (!Utils.IsEqual(m_dataType, value.m_dataType)) return false;
            if (!Utils.IsEqual(m_valueRank, value.m_valueRank)) return false;
            if (!Utils.IsEqual(m_arrayDimensions, value.m_arrayDimensions)) return false;
            if (!Utils.IsEqual(m_isAbstract, value.m_isAbstract)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (VariableTypeNode)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            VariableTypeNode clone = (VariableTypeNode)base.MemberwiseClone();

            clone.m_value = (Variant)Utils.Clone(this.m_value);
            clone.m_dataType = (NodeId)Utils.Clone(this.m_dataType);
            clone.m_valueRank = (int)Utils.Clone(this.m_valueRank);
            clone.m_arrayDimensions = (UInt32Collection)Utils.Clone(this.m_arrayDimensions);
            clone.m_isAbstract = (bool)Utils.Clone(this.m_isAbstract);

            return clone;
        }
        #endregion

        #region Private Fields
        private Variant m_value;
        private NodeId m_dataType;
        private int m_valueRank;
        private UInt32Collection m_arrayDimensions;
        private bool m_isAbstract;
        #endregion
    }
    #endif
    #endregion

    #region ReferenceTypeNode Class
    #if (!OPCUA_EXCLUDE_ReferenceTypeNode)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ReferenceTypeNode : TypeNode
    {
        #region Constructors
        /// <remarks />
        public ReferenceTypeNode()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_isAbstract = true;
            m_symmetric = true;
            m_inverseName = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "IsAbstract", IsRequired = false, Order = 1)]
        public bool IsAbstract
        {
            get { return m_isAbstract;  }
            set { m_isAbstract = value; }
        }

        /// <remarks />
        [DataMember(Name = "Symmetric", IsRequired = false, Order = 2)]
        public bool Symmetric
        {
            get { return m_symmetric;  }
            set { m_symmetric = value; }
        }

        /// <remarks />
        [DataMember(Name = "InverseName", IsRequired = false, Order = 3)]
        public LocalizedText InverseName
        {
            get { return m_inverseName;  }
            set { m_inverseName = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.ReferenceTypeNode; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.ReferenceTypeNode_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.ReferenceTypeNode_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.ReferenceTypeNode_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteBoolean("IsAbstract", IsAbstract);
            encoder.WriteBoolean("Symmetric", Symmetric);
            encoder.WriteLocalizedText("InverseName", InverseName);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            IsAbstract = decoder.ReadBoolean("IsAbstract");
            Symmetric = decoder.ReadBoolean("Symmetric");
            InverseName = decoder.ReadLocalizedText("InverseName");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ReferenceTypeNode value = encodeable as ReferenceTypeNode;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_isAbstract, value.m_isAbstract)) return false;
            if (!Utils.IsEqual(m_symmetric, value.m_symmetric)) return false;
            if (!Utils.IsEqual(m_inverseName, value.m_inverseName)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (ReferenceTypeNode)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReferenceTypeNode clone = (ReferenceTypeNode)base.MemberwiseClone();

            clone.m_isAbstract = (bool)Utils.Clone(this.m_isAbstract);
            clone.m_symmetric = (bool)Utils.Clone(this.m_symmetric);
            clone.m_inverseName = (LocalizedText)Utils.Clone(this.m_inverseName);

            return clone;
        }
        #endregion

        #region Private Fields
        private bool m_isAbstract;
        private bool m_symmetric;
        private LocalizedText m_inverseName;
        #endregion
    }
    #endif
    #endregion

    #region MethodNode Class
    #if (!OPCUA_EXCLUDE_MethodNode)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class MethodNode : InstanceNode
    {
        #region Constructors
        /// <remarks />
        public MethodNode()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_executable = true;
            m_userExecutable = true;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Executable", IsRequired = false, Order = 1)]
        public bool Executable
        {
            get { return m_executable;  }
            set { m_executable = value; }
        }

        /// <remarks />
        [DataMember(Name = "UserExecutable", IsRequired = false, Order = 2)]
        public bool UserExecutable
        {
            get { return m_userExecutable;  }
            set { m_userExecutable = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.MethodNode; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.MethodNode_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.MethodNode_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.MethodNode_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteBoolean("Executable", Executable);
            encoder.WriteBoolean("UserExecutable", UserExecutable);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Executable = decoder.ReadBoolean("Executable");
            UserExecutable = decoder.ReadBoolean("UserExecutable");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            MethodNode value = encodeable as MethodNode;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_executable, value.m_executable)) return false;
            if (!Utils.IsEqual(m_userExecutable, value.m_userExecutable)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (MethodNode)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            MethodNode clone = (MethodNode)base.MemberwiseClone();

            clone.m_executable = (bool)Utils.Clone(this.m_executable);
            clone.m_userExecutable = (bool)Utils.Clone(this.m_userExecutable);

            return clone;
        }
        #endregion

        #region Private Fields
        private bool m_executable;
        private bool m_userExecutable;
        #endregion
    }
    #endif
    #endregion

    #region ViewNode Class
    #if (!OPCUA_EXCLUDE_ViewNode)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ViewNode : InstanceNode
    {
        #region Constructors
        /// <remarks />
        public ViewNode()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_containsNoLoops = true;
            m_eventNotifier = (byte)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "ContainsNoLoops", IsRequired = false, Order = 1)]
        public bool ContainsNoLoops
        {
            get { return m_containsNoLoops;  }
            set { m_containsNoLoops = value; }
        }

        /// <remarks />
        [DataMember(Name = "EventNotifier", IsRequired = false, Order = 2)]
        public byte EventNotifier
        {
            get { return m_eventNotifier;  }
            set { m_eventNotifier = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.ViewNode; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.ViewNode_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.ViewNode_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.ViewNode_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteBoolean("ContainsNoLoops", ContainsNoLoops);
            encoder.WriteByte("EventNotifier", EventNotifier);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ContainsNoLoops = decoder.ReadBoolean("ContainsNoLoops");
            EventNotifier = decoder.ReadByte("EventNotifier");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ViewNode value = encodeable as ViewNode;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_containsNoLoops, value.m_containsNoLoops)) return false;
            if (!Utils.IsEqual(m_eventNotifier, value.m_eventNotifier)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (ViewNode)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ViewNode clone = (ViewNode)base.MemberwiseClone();

            clone.m_containsNoLoops = (bool)Utils.Clone(this.m_containsNoLoops);
            clone.m_eventNotifier = (byte)Utils.Clone(this.m_eventNotifier);

            return clone;
        }
        #endregion

        #region Private Fields
        private bool m_containsNoLoops;
        private byte m_eventNotifier;
        #endregion
    }
    #endif
    #endregion

    #region DataTypeNode Class
    #if (!OPCUA_EXCLUDE_DataTypeNode)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class DataTypeNode : TypeNode
    {
        #region Constructors
        /// <remarks />
        public DataTypeNode()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_isAbstract = true;
            m_dataTypeDefinition = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "IsAbstract", IsRequired = false, Order = 1)]
        public bool IsAbstract
        {
            get { return m_isAbstract;  }
            set { m_isAbstract = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataTypeDefinition", IsRequired = false, Order = 2)]
        public ExtensionObject DataTypeDefinition
        {
            get { return m_dataTypeDefinition;  }
            set { m_dataTypeDefinition = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.DataTypeNode; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.DataTypeNode_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.DataTypeNode_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.DataTypeNode_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteBoolean("IsAbstract", IsAbstract);
            encoder.WriteExtensionObject("DataTypeDefinition", DataTypeDefinition);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            IsAbstract = decoder.ReadBoolean("IsAbstract");
            DataTypeDefinition = decoder.ReadExtensionObject("DataTypeDefinition");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            DataTypeNode value = encodeable as DataTypeNode;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_isAbstract, value.m_isAbstract)) return false;
            if (!Utils.IsEqual(m_dataTypeDefinition, value.m_dataTypeDefinition)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (DataTypeNode)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            DataTypeNode clone = (DataTypeNode)base.MemberwiseClone();

            clone.m_isAbstract = (bool)Utils.Clone(this.m_isAbstract);
            clone.m_dataTypeDefinition = (ExtensionObject)Utils.Clone(this.m_dataTypeDefinition);

            return clone;
        }
        #endregion

        #region Private Fields
        private bool m_isAbstract;
        private ExtensionObject m_dataTypeDefinition;
        #endregion
    }
    #endif
    #endregion

    #region ReferenceNode Class
    #if (!OPCUA_EXCLUDE_ReferenceNode)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ReferenceNode : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public ReferenceNode()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_referenceTypeId = null;
            m_isInverse = true;
            m_targetId = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "ReferenceTypeId", IsRequired = false, Order = 1)]
        public NodeId ReferenceTypeId
        {
            get { return m_referenceTypeId;  }
            set { m_referenceTypeId = value; }
        }

        /// <remarks />
        [DataMember(Name = "IsInverse", IsRequired = false, Order = 2)]
        public bool IsInverse
        {
            get { return m_isInverse;  }
            set { m_isInverse = value; }
        }

        /// <remarks />
        [DataMember(Name = "TargetId", IsRequired = false, Order = 3)]
        public ExpandedNodeId TargetId
        {
            get { return m_targetId;  }
            set { m_targetId = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.ReferenceNode; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.ReferenceNode_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.ReferenceNode_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.ReferenceNode_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteNodeId("ReferenceTypeId", ReferenceTypeId);
            encoder.WriteBoolean("IsInverse", IsInverse);
            encoder.WriteExpandedNodeId("TargetId", TargetId);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ReferenceTypeId = decoder.ReadNodeId("ReferenceTypeId");
            IsInverse = decoder.ReadBoolean("IsInverse");
            TargetId = decoder.ReadExpandedNodeId("TargetId");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ReferenceNode value = encodeable as ReferenceNode;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_referenceTypeId, value.m_referenceTypeId)) return false;
            if (!Utils.IsEqual(m_isInverse, value.m_isInverse)) return false;
            if (!Utils.IsEqual(m_targetId, value.m_targetId)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (ReferenceNode)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReferenceNode clone = (ReferenceNode)base.MemberwiseClone();

            clone.m_referenceTypeId = (NodeId)Utils.Clone(this.m_referenceTypeId);
            clone.m_isInverse = (bool)Utils.Clone(this.m_isInverse);
            clone.m_targetId = (ExpandedNodeId)Utils.Clone(this.m_targetId);

            return clone;
        }
        #endregion

        #region Private Fields
        private NodeId m_referenceTypeId;
        private bool m_isInverse;
        private ExpandedNodeId m_targetId;
        #endregion
    }

    #region ReferenceNodeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfReferenceNode", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ReferenceNode")]
    #if !NET_STANDARD
    public partial class ReferenceNodeCollection : List<ReferenceNode>, ICloneable
    #else
    public partial class ReferenceNodeCollection : List<ReferenceNode>
    #endif
    {
        #region Constructors
        /// <remarks />
        public ReferenceNodeCollection() {}

        /// <remarks />
        public ReferenceNodeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public ReferenceNodeCollection(IEnumerable<ReferenceNode> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator ReferenceNodeCollection(ReferenceNode[] values)
        {
            if (values != null)
            {
                return new ReferenceNodeCollection(values);
            }

            return new ReferenceNodeCollection();
        }

        /// <remarks />
        public static explicit operator ReferenceNode[](ReferenceNodeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (ReferenceNodeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ReferenceNodeCollection clone = new ReferenceNodeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ReferenceNode)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region Argument Class
    #if (!OPCUA_EXCLUDE_Argument)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class Argument : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public Argument()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_name = null;
            m_dataType = null;
            m_valueRank = (int)0;
            m_arrayDimensions = new UInt32Collection();
            m_description = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 1)]
        public string Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }

        /// <remarks />
        [DataMember(Name = "DataType", IsRequired = false, Order = 2)]
        public NodeId DataType
        {
            get { return m_dataType;  }
            set { m_dataType = value; }
        }

        /// <remarks />
        [DataMember(Name = "ValueRank", IsRequired = false, Order = 3)]
        public int ValueRank
        {
            get { return m_valueRank;  }
            set { m_valueRank = value; }
        }

        /// <remarks />
        [DataMember(Name = "ArrayDimensions", IsRequired = false, Order = 4)]
        public UInt32Collection ArrayDimensions
        {
            get
            {
                return m_arrayDimensions;
            }

            set
            {
                m_arrayDimensions = value;

                if (value == null)
                {
                    m_arrayDimensions = new UInt32Collection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "Description", IsRequired = false, Order = 5)]
        public LocalizedText Description
        {
            get { return m_description;  }
            set { m_description = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.Argument; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.Argument_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.Argument_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.Argument_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("Name", Name);
            encoder.WriteNodeId("DataType", DataType);
            encoder.WriteInt32("ValueRank", ValueRank);
            encoder.WriteUInt32Array("ArrayDimensions", ArrayDimensions);
            encoder.WriteLocalizedText("Description", Description);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Name = decoder.ReadString("Name");
            DataType = decoder.ReadNodeId("DataType");
            ValueRank = decoder.ReadInt32("ValueRank");
            ArrayDimensions = decoder.ReadUInt32Array("ArrayDimensions");
            Description = decoder.ReadLocalizedText("Description");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            Argument value = encodeable as Argument;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_name, value.m_name)) return false;
            if (!Utils.IsEqual(m_dataType, value.m_dataType)) return false;
            if (!Utils.IsEqual(m_valueRank, value.m_valueRank)) return false;
            if (!Utils.IsEqual(m_arrayDimensions, value.m_arrayDimensions)) return false;
            if (!Utils.IsEqual(m_description, value.m_description)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (Argument)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            Argument clone = (Argument)base.MemberwiseClone();

            clone.m_name = (string)Utils.Clone(this.m_name);
            clone.m_dataType = (NodeId)Utils.Clone(this.m_dataType);
            clone.m_valueRank = (int)Utils.Clone(this.m_valueRank);
            clone.m_arrayDimensions = (UInt32Collection)Utils.Clone(this.m_arrayDimensions);
            clone.m_description = (LocalizedText)Utils.Clone(this.m_description);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_name;
        private NodeId m_dataType;
        private int m_valueRank;
        private UInt32Collection m_arrayDimensions;
        private LocalizedText m_description;
        #endregion
    }

    #region ArgumentCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfArgument", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "Argument")]
    #if !NET_STANDARD
    public partial class ArgumentCollection : List<Argument>, ICloneable
    #else
    public partial class ArgumentCollection : List<Argument>
    #endif
    {
        #region Constructors
        /// <remarks />
        public ArgumentCollection() {}

        /// <remarks />
        public ArgumentCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public ArgumentCollection(IEnumerable<Argument> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator ArgumentCollection(Argument[] values)
        {
            if (values != null)
            {
                return new ArgumentCollection(values);
            }

            return new ArgumentCollection();
        }

        /// <remarks />
        public static explicit operator Argument[](ArgumentCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (ArgumentCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ArgumentCollection clone = new ArgumentCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((Argument)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region EnumValueType Class
    #if (!OPCUA_EXCLUDE_EnumValueType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class EnumValueType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public EnumValueType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_value = (long)0;
            m_displayName = null;
            m_description = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Value", IsRequired = false, Order = 1)]
        public long Value
        {
            get { return m_value;  }
            set { m_value = value; }
        }

        /// <remarks />
        [DataMember(Name = "DisplayName", IsRequired = false, Order = 2)]
        public LocalizedText DisplayName
        {
            get { return m_displayName;  }
            set { m_displayName = value; }
        }

        /// <remarks />
        [DataMember(Name = "Description", IsRequired = false, Order = 3)]
        public LocalizedText Description
        {
            get { return m_description;  }
            set { m_description = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.EnumValueType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.EnumValueType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.EnumValueType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.EnumValueType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteInt64("Value", Value);
            encoder.WriteLocalizedText("DisplayName", DisplayName);
            encoder.WriteLocalizedText("Description", Description);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Value = decoder.ReadInt64("Value");
            DisplayName = decoder.ReadLocalizedText("DisplayName");
            Description = decoder.ReadLocalizedText("Description");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            EnumValueType value = encodeable as EnumValueType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_value, value.m_value)) return false;
            if (!Utils.IsEqual(m_displayName, value.m_displayName)) return false;
            if (!Utils.IsEqual(m_description, value.m_description)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (EnumValueType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EnumValueType clone = (EnumValueType)base.MemberwiseClone();

            clone.m_value = (long)Utils.Clone(this.m_value);
            clone.m_displayName = (LocalizedText)Utils.Clone(this.m_displayName);
            clone.m_description = (LocalizedText)Utils.Clone(this.m_description);

            return clone;
        }
        #endregion

        #region Private Fields
        private long m_value;
        private LocalizedText m_displayName;
        private LocalizedText m_description;
        #endregion
    }

    #region EnumValueTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfEnumValueType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "EnumValueType")]
    #if !NET_STANDARD
    public partial class EnumValueTypeCollection : List<EnumValueType>, ICloneable
    #else
    public partial class EnumValueTypeCollection : List<EnumValueType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public EnumValueTypeCollection() {}

        /// <remarks />
        public EnumValueTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public EnumValueTypeCollection(IEnumerable<EnumValueType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator EnumValueTypeCollection(EnumValueType[] values)
        {
            if (values != null)
            {
                return new EnumValueTypeCollection(values);
            }

            return new EnumValueTypeCollection();
        }

        /// <remarks />
        public static explicit operator EnumValueType[](EnumValueTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (EnumValueTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EnumValueTypeCollection clone = new EnumValueTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((EnumValueType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region EnumField Class
    #if (!OPCUA_EXCLUDE_EnumField)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class EnumField : EnumValueType
    {
        #region Constructors
        /// <remarks />
        public EnumField()
        {
            Initialize();
        }

        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }

        private void Initialize()
        {
            m_name = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Name", IsRequired = false, Order = 1)]
        public string Name
        {
            get { return m_name;  }
            set { m_name = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public override ExpandedNodeId TypeId => DataTypeIds.EnumField; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public override ExpandedNodeId BinaryEncodingId => ObjectIds.EnumField_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public override ExpandedNodeId XmlEncodingId => ObjectIds.EnumField_Encoding_DefaultXml;
            
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public override ExpandedNodeId JsonEncodingId => ObjectIds.EnumField_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public override void Encode(IEncoder encoder)
        {
            base.Encode(encoder);

            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("Name", Name);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public override void Decode(IDecoder decoder)
        {
            base.Decode(decoder);

            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Name = decoder.ReadString("Name");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public override bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            EnumField value = encodeable as EnumField;

            if (value == null)
            {
                return false;
            }

            if (!base.IsEqual(encodeable)) return false;
            if (!Utils.IsEqual(m_name, value.m_name)) return false;

            return true;
        }    

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public override object Clone()
        {
            return (EnumField)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EnumField clone = (EnumField)base.MemberwiseClone();

            clone.m_name = (string)Utils.Clone(this.m_name);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_name;
        #endregion
    }

    #region EnumFieldCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfEnumField", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "EnumField")]
    #if !NET_STANDARD
    public partial class EnumFieldCollection : List<EnumField>, ICloneable
    #else
    public partial class EnumFieldCollection : List<EnumField>
    #endif
    {
        #region Constructors
        /// <remarks />
        public EnumFieldCollection() {}

        /// <remarks />
        public EnumFieldCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public EnumFieldCollection(IEnumerable<EnumField> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator EnumFieldCollection(EnumField[] values)
        {
            if (values != null)
            {
                return new EnumFieldCollection(values);
            }

            return new EnumFieldCollection();
        }

        /// <remarks />
        public static explicit operator EnumField[](EnumFieldCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (EnumFieldCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EnumFieldCollection clone = new EnumFieldCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((EnumField)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region OptionSet Class
    #if (!OPCUA_EXCLUDE_OptionSet)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class OptionSet : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public OptionSet()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_value = null;
            m_validBits = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Value", IsRequired = false, Order = 1)]
        public byte[] Value
        {
            get { return m_value;  }
            set { m_value = value; }
        }

        /// <remarks />
        [DataMember(Name = "ValidBits", IsRequired = false, Order = 2)]
        public byte[] ValidBits
        {
            get { return m_validBits;  }
            set { m_validBits = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.OptionSet; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.OptionSet_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.OptionSet_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.OptionSet_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteByteString("Value", Value);
            encoder.WriteByteString("ValidBits", ValidBits);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Value = decoder.ReadByteString("Value");
            ValidBits = decoder.ReadByteString("ValidBits");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            OptionSet value = encodeable as OptionSet;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_value, value.m_value)) return false;
            if (!Utils.IsEqual(m_validBits, value.m_validBits)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (OptionSet)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            OptionSet clone = (OptionSet)base.MemberwiseClone();

            clone.m_value = (byte[])Utils.Clone(this.m_value);
            clone.m_validBits = (byte[])Utils.Clone(this.m_validBits);

            return clone;
        }
        #endregion

        #region Private Fields
        private byte[] m_value;
        private byte[] m_validBits;
        #endregion
    }

    #region OptionSetCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfOptionSet", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "OptionSet")]
    #if !NET_STANDARD
    public partial class OptionSetCollection : List<OptionSet>, ICloneable
    #else
    public partial class OptionSetCollection : List<OptionSet>
    #endif
    {
        #region Constructors
        /// <remarks />
        public OptionSetCollection() {}

        /// <remarks />
        public OptionSetCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public OptionSetCollection(IEnumerable<OptionSet> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator OptionSetCollection(OptionSet[] values)
        {
            if (values != null)
            {
                return new OptionSetCollection(values);
            }

            return new OptionSetCollection();
        }

        /// <remarks />
        public static explicit operator OptionSet[](OptionSetCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (OptionSetCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            OptionSetCollection clone = new OptionSetCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((OptionSet)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region TimeZoneDataType Class
    #if (!OPCUA_EXCLUDE_TimeZoneDataType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class TimeZoneDataType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public TimeZoneDataType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_offset = (short)0;
            m_daylightSavingInOffset = true;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Offset", IsRequired = false, Order = 1)]
        public short Offset
        {
            get { return m_offset;  }
            set { m_offset = value; }
        }

        /// <remarks />
        [DataMember(Name = "DaylightSavingInOffset", IsRequired = false, Order = 2)]
        public bool DaylightSavingInOffset
        {
            get { return m_daylightSavingInOffset;  }
            set { m_daylightSavingInOffset = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.TimeZoneDataType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.TimeZoneDataType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.TimeZoneDataType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.TimeZoneDataType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteInt16("Offset", Offset);
            encoder.WriteBoolean("DaylightSavingInOffset", DaylightSavingInOffset);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Offset = decoder.ReadInt16("Offset");
            DaylightSavingInOffset = decoder.ReadBoolean("DaylightSavingInOffset");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            TimeZoneDataType value = encodeable as TimeZoneDataType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_offset, value.m_offset)) return false;
            if (!Utils.IsEqual(m_daylightSavingInOffset, value.m_daylightSavingInOffset)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (TimeZoneDataType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TimeZoneDataType clone = (TimeZoneDataType)base.MemberwiseClone();

            clone.m_offset = (short)Utils.Clone(this.m_offset);
            clone.m_daylightSavingInOffset = (bool)Utils.Clone(this.m_daylightSavingInOffset);

            return clone;
        }
        #endregion

        #region Private Fields
        private short m_offset;
        private bool m_daylightSavingInOffset;
        #endregion
    }

    #region TimeZoneDataTypeCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfTimeZoneDataType", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "TimeZoneDataType")]
    #if !NET_STANDARD
    public partial class TimeZoneDataTypeCollection : List<TimeZoneDataType>, ICloneable
    #else
    public partial class TimeZoneDataTypeCollection : List<TimeZoneDataType>
    #endif
    {
        #region Constructors
        /// <remarks />
        public TimeZoneDataTypeCollection() {}

        /// <remarks />
        public TimeZoneDataTypeCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public TimeZoneDataTypeCollection(IEnumerable<TimeZoneDataType> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator TimeZoneDataTypeCollection(TimeZoneDataType[] values)
        {
            if (values != null)
            {
                return new TimeZoneDataTypeCollection(values);
            }

            return new TimeZoneDataTypeCollection();
        }

        /// <remarks />
        public static explicit operator TimeZoneDataType[](TimeZoneDataTypeCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (TimeZoneDataTypeCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            TimeZoneDataTypeCollection clone = new TimeZoneDataTypeCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((TimeZoneDataType)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region ApplicationType Enumeration
    #if (!OPCUA_EXCLUDE_ApplicationType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum ApplicationType
    {
        /// <remarks />
        [EnumMember(Value = "Server_0")]
        Server = 0,

        /// <remarks />
        [EnumMember(Value = "Client_1")]
        Client = 1,

        /// <remarks />
        [EnumMember(Value = "ClientAndServer_2")]
        ClientAndServer = 2,

        /// <remarks />
        [EnumMember(Value = "DiscoveryServer_3")]
        DiscoveryServer = 3,
    }
    #endif
    #endregion

    #region ApplicationDescription Class
    #if (!OPCUA_EXCLUDE_ApplicationDescription)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ApplicationDescription : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public ApplicationDescription()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_applicationUri = null;
            m_productUri = null;
            m_applicationName = null;
            m_applicationType = ApplicationType.Server;
            m_gatewayServerUri = null;
            m_discoveryProfileUri = null;
            m_discoveryUrls = new StringCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "ApplicationUri", IsRequired = false, Order = 1)]
        public string ApplicationUri
        {
            get { return m_applicationUri;  }
            set { m_applicationUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "ProductUri", IsRequired = false, Order = 2)]
        public string ProductUri
        {
            get { return m_productUri;  }
            set { m_productUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "ApplicationName", IsRequired = false, Order = 3)]
        public LocalizedText ApplicationName
        {
            get { return m_applicationName;  }
            set { m_applicationName = value; }
        }

        /// <remarks />
        [DataMember(Name = "ApplicationType", IsRequired = false, Order = 4)]
        public ApplicationType ApplicationType
        {
            get { return m_applicationType;  }
            set { m_applicationType = value; }
        }

        /// <remarks />
        [DataMember(Name = "GatewayServerUri", IsRequired = false, Order = 5)]
        public string GatewayServerUri
        {
            get { return m_gatewayServerUri;  }
            set { m_gatewayServerUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "DiscoveryProfileUri", IsRequired = false, Order = 6)]
        public string DiscoveryProfileUri
        {
            get { return m_discoveryProfileUri;  }
            set { m_discoveryProfileUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "DiscoveryUrls", IsRequired = false, Order = 7)]
        public StringCollection DiscoveryUrls
        {
            get
            {
                return m_discoveryUrls;
            }

            set
            {
                m_discoveryUrls = value;

                if (value == null)
                {
                    m_discoveryUrls = new StringCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.ApplicationDescription; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.ApplicationDescription_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.ApplicationDescription_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.ApplicationDescription_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("ApplicationUri", ApplicationUri);
            encoder.WriteString("ProductUri", ProductUri);
            encoder.WriteLocalizedText("ApplicationName", ApplicationName);
            encoder.WriteEnumerated("ApplicationType", ApplicationType);
            encoder.WriteString("GatewayServerUri", GatewayServerUri);
            encoder.WriteString("DiscoveryProfileUri", DiscoveryProfileUri);
            encoder.WriteStringArray("DiscoveryUrls", DiscoveryUrls);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ApplicationUri = decoder.ReadString("ApplicationUri");
            ProductUri = decoder.ReadString("ProductUri");
            ApplicationName = decoder.ReadLocalizedText("ApplicationName");
            ApplicationType = (ApplicationType)decoder.ReadEnumerated("ApplicationType", typeof(ApplicationType));
            GatewayServerUri = decoder.ReadString("GatewayServerUri");
            DiscoveryProfileUri = decoder.ReadString("DiscoveryProfileUri");
            DiscoveryUrls = decoder.ReadStringArray("DiscoveryUrls");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ApplicationDescription value = encodeable as ApplicationDescription;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_applicationUri, value.m_applicationUri)) return false;
            if (!Utils.IsEqual(m_productUri, value.m_productUri)) return false;
            if (!Utils.IsEqual(m_applicationName, value.m_applicationName)) return false;
            if (!Utils.IsEqual(m_applicationType, value.m_applicationType)) return false;
            if (!Utils.IsEqual(m_gatewayServerUri, value.m_gatewayServerUri)) return false;
            if (!Utils.IsEqual(m_discoveryProfileUri, value.m_discoveryProfileUri)) return false;
            if (!Utils.IsEqual(m_discoveryUrls, value.m_discoveryUrls)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (ApplicationDescription)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ApplicationDescription clone = (ApplicationDescription)base.MemberwiseClone();

            clone.m_applicationUri = (string)Utils.Clone(this.m_applicationUri);
            clone.m_productUri = (string)Utils.Clone(this.m_productUri);
            clone.m_applicationName = (LocalizedText)Utils.Clone(this.m_applicationName);
            clone.m_applicationType = (ApplicationType)Utils.Clone(this.m_applicationType);
            clone.m_gatewayServerUri = (string)Utils.Clone(this.m_gatewayServerUri);
            clone.m_discoveryProfileUri = (string)Utils.Clone(this.m_discoveryProfileUri);
            clone.m_discoveryUrls = (StringCollection)Utils.Clone(this.m_discoveryUrls);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_applicationUri;
        private string m_productUri;
        private LocalizedText m_applicationName;
        private ApplicationType m_applicationType;
        private string m_gatewayServerUri;
        private string m_discoveryProfileUri;
        private StringCollection m_discoveryUrls;
        #endregion
    }

    #region ApplicationDescriptionCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfApplicationDescription", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ApplicationDescription")]
    #if !NET_STANDARD
    public partial class ApplicationDescriptionCollection : List<ApplicationDescription>, ICloneable
    #else
    public partial class ApplicationDescriptionCollection : List<ApplicationDescription>
    #endif
    {
        #region Constructors
        /// <remarks />
        public ApplicationDescriptionCollection() {}

        /// <remarks />
        public ApplicationDescriptionCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public ApplicationDescriptionCollection(IEnumerable<ApplicationDescription> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator ApplicationDescriptionCollection(ApplicationDescription[] values)
        {
            if (values != null)
            {
                return new ApplicationDescriptionCollection(values);
            }

            return new ApplicationDescriptionCollection();
        }

        /// <remarks />
        public static explicit operator ApplicationDescription[](ApplicationDescriptionCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (ApplicationDescriptionCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ApplicationDescriptionCollection clone = new ApplicationDescriptionCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ApplicationDescription)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region RequestHeader Class
    #if (!OPCUA_EXCLUDE_RequestHeader)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class RequestHeader : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public RequestHeader()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_authenticationToken = null;
            m_timestamp = DateTime.MinValue;
            m_requestHandle = (uint)0;
            m_returnDiagnostics = (uint)0;
            m_auditEntryId = null;
            m_timeoutHint = (uint)0;
            m_additionalHeader = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "AuthenticationToken", IsRequired = false, Order = 1)]
        public NodeId AuthenticationToken
        {
            get { return m_authenticationToken;  }
            set { m_authenticationToken = value; }
        }

        /// <remarks />
        [DataMember(Name = "Timestamp", IsRequired = false, Order = 2)]
        public DateTime Timestamp
        {
            get { return m_timestamp;  }
            set { m_timestamp = value; }
        }

        /// <remarks />
        [DataMember(Name = "RequestHandle", IsRequired = false, Order = 3)]
        public uint RequestHandle
        {
            get { return m_requestHandle;  }
            set { m_requestHandle = value; }
        }

        /// <remarks />
        [DataMember(Name = "ReturnDiagnostics", IsRequired = false, Order = 4)]
        public uint ReturnDiagnostics
        {
            get { return m_returnDiagnostics;  }
            set { m_returnDiagnostics = value; }
        }

        /// <remarks />
        [DataMember(Name = "AuditEntryId", IsRequired = false, Order = 5)]
        public string AuditEntryId
        {
            get { return m_auditEntryId;  }
            set { m_auditEntryId = value; }
        }

        /// <remarks />
        [DataMember(Name = "TimeoutHint", IsRequired = false, Order = 6)]
        public uint TimeoutHint
        {
            get { return m_timeoutHint;  }
            set { m_timeoutHint = value; }
        }

        /// <remarks />
        [DataMember(Name = "AdditionalHeader", IsRequired = false, Order = 7)]
        public ExtensionObject AdditionalHeader
        {
            get { return m_additionalHeader;  }
            set { m_additionalHeader = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.RequestHeader; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.RequestHeader_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.RequestHeader_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.RequestHeader_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteNodeId("AuthenticationToken", AuthenticationToken);
            encoder.WriteDateTime("Timestamp", Timestamp);
            encoder.WriteUInt32("RequestHandle", RequestHandle);
            encoder.WriteUInt32("ReturnDiagnostics", ReturnDiagnostics);
            encoder.WriteString("AuditEntryId", AuditEntryId);
            encoder.WriteUInt32("TimeoutHint", TimeoutHint);
            encoder.WriteExtensionObject("AdditionalHeader", AdditionalHeader);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            AuthenticationToken = decoder.ReadNodeId("AuthenticationToken");
            Timestamp = decoder.ReadDateTime("Timestamp");
            RequestHandle = decoder.ReadUInt32("RequestHandle");
            ReturnDiagnostics = decoder.ReadUInt32("ReturnDiagnostics");
            AuditEntryId = decoder.ReadString("AuditEntryId");
            TimeoutHint = decoder.ReadUInt32("TimeoutHint");
            AdditionalHeader = decoder.ReadExtensionObject("AdditionalHeader");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            RequestHeader value = encodeable as RequestHeader;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_authenticationToken, value.m_authenticationToken)) return false;
            if (!Utils.IsEqual(m_timestamp, value.m_timestamp)) return false;
            if (!Utils.IsEqual(m_requestHandle, value.m_requestHandle)) return false;
            if (!Utils.IsEqual(m_returnDiagnostics, value.m_returnDiagnostics)) return false;
            if (!Utils.IsEqual(m_auditEntryId, value.m_auditEntryId)) return false;
            if (!Utils.IsEqual(m_timeoutHint, value.m_timeoutHint)) return false;
            if (!Utils.IsEqual(m_additionalHeader, value.m_additionalHeader)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (RequestHeader)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            RequestHeader clone = (RequestHeader)base.MemberwiseClone();

            clone.m_authenticationToken = (NodeId)Utils.Clone(this.m_authenticationToken);
            clone.m_timestamp = (DateTime)Utils.Clone(this.m_timestamp);
            clone.m_requestHandle = (uint)Utils.Clone(this.m_requestHandle);
            clone.m_returnDiagnostics = (uint)Utils.Clone(this.m_returnDiagnostics);
            clone.m_auditEntryId = (string)Utils.Clone(this.m_auditEntryId);
            clone.m_timeoutHint = (uint)Utils.Clone(this.m_timeoutHint);
            clone.m_additionalHeader = (ExtensionObject)Utils.Clone(this.m_additionalHeader);

            return clone;
        }
        #endregion

        #region Private Fields
        private NodeId m_authenticationToken;
        private DateTime m_timestamp;
        private uint m_requestHandle;
        private uint m_returnDiagnostics;
        private string m_auditEntryId;
        private uint m_timeoutHint;
        private ExtensionObject m_additionalHeader;
        #endregion
    }
    #endif
    #endregion

    #region ResponseHeader Class
    #if (!OPCUA_EXCLUDE_ResponseHeader)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ResponseHeader : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public ResponseHeader()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_timestamp = DateTime.MinValue;
            m_requestHandle = (uint)0;
            m_serviceResult = StatusCodes.Good;
            m_serviceDiagnostics = null;
            m_stringTable = new StringCollection();
            m_additionalHeader = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "Timestamp", IsRequired = false, Order = 1)]
        public DateTime Timestamp
        {
            get { return m_timestamp;  }
            set { m_timestamp = value; }
        }

        /// <remarks />
        [DataMember(Name = "RequestHandle", IsRequired = false, Order = 2)]
        public uint RequestHandle
        {
            get { return m_requestHandle;  }
            set { m_requestHandle = value; }
        }

        /// <remarks />
        [DataMember(Name = "ServiceResult", IsRequired = false, Order = 3)]
        public StatusCode ServiceResult
        {
            get { return m_serviceResult;  }
            set { m_serviceResult = value; }
        }

        /// <remarks />
        [DataMember(Name = "ServiceDiagnostics", IsRequired = false, Order = 4)]
        public DiagnosticInfo ServiceDiagnostics
        {
            get { return m_serviceDiagnostics;  }
            set { m_serviceDiagnostics = value; }
        }

        /// <remarks />
        [DataMember(Name = "StringTable", IsRequired = false, Order = 5)]
        public StringCollection StringTable
        {
            get
            {
                return m_stringTable;
            }

            set
            {
                m_stringTable = value;

                if (value == null)
                {
                    m_stringTable = new StringCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "AdditionalHeader", IsRequired = false, Order = 6)]
        public ExtensionObject AdditionalHeader
        {
            get { return m_additionalHeader;  }
            set { m_additionalHeader = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.ResponseHeader; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.ResponseHeader_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.ResponseHeader_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.ResponseHeader_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteDateTime("Timestamp", Timestamp);
            encoder.WriteUInt32("RequestHandle", RequestHandle);
            encoder.WriteStatusCode("ServiceResult", ServiceResult);
            encoder.WriteDiagnosticInfo("ServiceDiagnostics", ServiceDiagnostics);
            encoder.WriteStringArray("StringTable", StringTable);
            encoder.WriteExtensionObject("AdditionalHeader", AdditionalHeader);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            Timestamp = decoder.ReadDateTime("Timestamp");
            RequestHandle = decoder.ReadUInt32("RequestHandle");
            ServiceResult = decoder.ReadStatusCode("ServiceResult");
            ServiceDiagnostics = decoder.ReadDiagnosticInfo("ServiceDiagnostics");
            StringTable = decoder.ReadStringArray("StringTable");
            AdditionalHeader = decoder.ReadExtensionObject("AdditionalHeader");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ResponseHeader value = encodeable as ResponseHeader;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_timestamp, value.m_timestamp)) return false;
            if (!Utils.IsEqual(m_requestHandle, value.m_requestHandle)) return false;
            if (!Utils.IsEqual(m_serviceResult, value.m_serviceResult)) return false;
            if (!Utils.IsEqual(m_serviceDiagnostics, value.m_serviceDiagnostics)) return false;
            if (!Utils.IsEqual(m_stringTable, value.m_stringTable)) return false;
            if (!Utils.IsEqual(m_additionalHeader, value.m_additionalHeader)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (ResponseHeader)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ResponseHeader clone = (ResponseHeader)base.MemberwiseClone();

            clone.m_timestamp = (DateTime)Utils.Clone(this.m_timestamp);
            clone.m_requestHandle = (uint)Utils.Clone(this.m_requestHandle);
            clone.m_serviceResult = (StatusCode)Utils.Clone(this.m_serviceResult);
            clone.m_serviceDiagnostics = (DiagnosticInfo)Utils.Clone(this.m_serviceDiagnostics);
            clone.m_stringTable = (StringCollection)Utils.Clone(this.m_stringTable);
            clone.m_additionalHeader = (ExtensionObject)Utils.Clone(this.m_additionalHeader);

            return clone;
        }
        #endregion

        #region Private Fields
        private DateTime m_timestamp;
        private uint m_requestHandle;
        private StatusCode m_serviceResult;
        private DiagnosticInfo m_serviceDiagnostics;
        private StringCollection m_stringTable;
        private ExtensionObject m_additionalHeader;
        #endregion
    }
    #endif
    #endregion

    #region ServiceFault Class
    #if (!OPCUA_EXCLUDE_ServiceFault)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ServiceFault : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public ServiceFault()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_responseHeader = new ResponseHeader();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "ResponseHeader", IsRequired = false, Order = 1)]
        public ResponseHeader ResponseHeader
        {
            get
            {
                return m_responseHeader;
            }

            set
            {
                m_responseHeader = value;

                if (value == null)
                {
                    m_responseHeader = new ResponseHeader();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.ServiceFault; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.ServiceFault_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.ServiceFault_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.ServiceFault_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("ResponseHeader", ResponseHeader, typeof(ResponseHeader));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ResponseHeader = (ResponseHeader)decoder.ReadEncodeable("ResponseHeader", typeof(ResponseHeader));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ServiceFault value = encodeable as ServiceFault;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_responseHeader, value.m_responseHeader)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (ServiceFault)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ServiceFault clone = (ServiceFault)base.MemberwiseClone();

            clone.m_responseHeader = (ResponseHeader)Utils.Clone(this.m_responseHeader);

            return clone;
        }
        #endregion

        #region Private Fields
        private ResponseHeader m_responseHeader;
        #endregion
    }
    #endif
    #endregion

    #region SessionlessInvokeRequestType Class
    #if (!OPCUA_EXCLUDE_SessionlessInvokeRequestType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class SessionlessInvokeRequestType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public SessionlessInvokeRequestType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_urisVersion = (uint)0;
            m_namespaceUris = new StringCollection();
            m_serverUris = new StringCollection();
            m_localeIds = new StringCollection();
            m_serviceId = (uint)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "UrisVersion", IsRequired = false, Order = 1)]
        public uint UrisVersion
        {
            get { return m_urisVersion;  }
            set { m_urisVersion = value; }
        }

        /// <remarks />
        [DataMember(Name = "NamespaceUris", IsRequired = false, Order = 2)]
        public StringCollection NamespaceUris
        {
            get
            {
                return m_namespaceUris;
            }

            set
            {
                m_namespaceUris = value;

                if (value == null)
                {
                    m_namespaceUris = new StringCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "ServerUris", IsRequired = false, Order = 3)]
        public StringCollection ServerUris
        {
            get
            {
                return m_serverUris;
            }

            set
            {
                m_serverUris = value;

                if (value == null)
                {
                    m_serverUris = new StringCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "LocaleIds", IsRequired = false, Order = 4)]
        public StringCollection LocaleIds
        {
            get
            {
                return m_localeIds;
            }

            set
            {
                m_localeIds = value;

                if (value == null)
                {
                    m_localeIds = new StringCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "ServiceId", IsRequired = false, Order = 5)]
        public uint ServiceId
        {
            get { return m_serviceId;  }
            set { m_serviceId = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.SessionlessInvokeRequestType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.SessionlessInvokeRequestType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.SessionlessInvokeRequestType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.SessionlessInvokeRequestType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt32("UrisVersion", UrisVersion);
            encoder.WriteStringArray("NamespaceUris", NamespaceUris);
            encoder.WriteStringArray("ServerUris", ServerUris);
            encoder.WriteStringArray("LocaleIds", LocaleIds);
            encoder.WriteUInt32("ServiceId", ServiceId);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            UrisVersion = decoder.ReadUInt32("UrisVersion");
            NamespaceUris = decoder.ReadStringArray("NamespaceUris");
            ServerUris = decoder.ReadStringArray("ServerUris");
            LocaleIds = decoder.ReadStringArray("LocaleIds");
            ServiceId = decoder.ReadUInt32("ServiceId");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            SessionlessInvokeRequestType value = encodeable as SessionlessInvokeRequestType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_urisVersion, value.m_urisVersion)) return false;
            if (!Utils.IsEqual(m_namespaceUris, value.m_namespaceUris)) return false;
            if (!Utils.IsEqual(m_serverUris, value.m_serverUris)) return false;
            if (!Utils.IsEqual(m_localeIds, value.m_localeIds)) return false;
            if (!Utils.IsEqual(m_serviceId, value.m_serviceId)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (SessionlessInvokeRequestType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            SessionlessInvokeRequestType clone = (SessionlessInvokeRequestType)base.MemberwiseClone();

            clone.m_urisVersion = (uint)Utils.Clone(this.m_urisVersion);
            clone.m_namespaceUris = (StringCollection)Utils.Clone(this.m_namespaceUris);
            clone.m_serverUris = (StringCollection)Utils.Clone(this.m_serverUris);
            clone.m_localeIds = (StringCollection)Utils.Clone(this.m_localeIds);
            clone.m_serviceId = (uint)Utils.Clone(this.m_serviceId);

            return clone;
        }
        #endregion

        #region Private Fields
        private uint m_urisVersion;
        private StringCollection m_namespaceUris;
        private StringCollection m_serverUris;
        private StringCollection m_localeIds;
        private uint m_serviceId;
        #endregion
    }
    #endif
    #endregion

    #region SessionlessInvokeResponseType Class
    #if (!OPCUA_EXCLUDE_SessionlessInvokeResponseType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class SessionlessInvokeResponseType : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public SessionlessInvokeResponseType()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_namespaceUris = new StringCollection();
            m_serverUris = new StringCollection();
            m_serviceId = (uint)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "NamespaceUris", IsRequired = false, Order = 1)]
        public StringCollection NamespaceUris
        {
            get
            {
                return m_namespaceUris;
            }

            set
            {
                m_namespaceUris = value;

                if (value == null)
                {
                    m_namespaceUris = new StringCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "ServerUris", IsRequired = false, Order = 2)]
        public StringCollection ServerUris
        {
            get
            {
                return m_serverUris;
            }

            set
            {
                m_serverUris = value;

                if (value == null)
                {
                    m_serverUris = new StringCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "ServiceId", IsRequired = false, Order = 3)]
        public uint ServiceId
        {
            get { return m_serviceId;  }
            set { m_serviceId = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.SessionlessInvokeResponseType; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.SessionlessInvokeResponseType_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.SessionlessInvokeResponseType_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.SessionlessInvokeResponseType_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteStringArray("NamespaceUris", NamespaceUris);
            encoder.WriteStringArray("ServerUris", ServerUris);
            encoder.WriteUInt32("ServiceId", ServiceId);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            NamespaceUris = decoder.ReadStringArray("NamespaceUris");
            ServerUris = decoder.ReadStringArray("ServerUris");
            ServiceId = decoder.ReadUInt32("ServiceId");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            SessionlessInvokeResponseType value = encodeable as SessionlessInvokeResponseType;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_namespaceUris, value.m_namespaceUris)) return false;
            if (!Utils.IsEqual(m_serverUris, value.m_serverUris)) return false;
            if (!Utils.IsEqual(m_serviceId, value.m_serviceId)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (SessionlessInvokeResponseType)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            SessionlessInvokeResponseType clone = (SessionlessInvokeResponseType)base.MemberwiseClone();

            clone.m_namespaceUris = (StringCollection)Utils.Clone(this.m_namespaceUris);
            clone.m_serverUris = (StringCollection)Utils.Clone(this.m_serverUris);
            clone.m_serviceId = (uint)Utils.Clone(this.m_serviceId);

            return clone;
        }
        #endregion

        #region Private Fields
        private StringCollection m_namespaceUris;
        private StringCollection m_serverUris;
        private uint m_serviceId;
        #endregion
    }
    #endif
    #endregion

    #region FindServersRequest Class
    #if (!OPCUA_EXCLUDE_FindServersRequest)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class FindServersRequest : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public FindServersRequest()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_requestHeader = new RequestHeader();
            m_endpointUrl = null;
            m_localeIds = new StringCollection();
            m_serverUris = new StringCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "RequestHeader", IsRequired = false, Order = 1)]
        public RequestHeader RequestHeader
        {
            get
            {
                return m_requestHeader;
            }

            set
            {
                m_requestHeader = value;

                if (value == null)
                {
                    m_requestHeader = new RequestHeader();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "EndpointUrl", IsRequired = false, Order = 2)]
        public string EndpointUrl
        {
            get { return m_endpointUrl;  }
            set { m_endpointUrl = value; }
        }

        /// <remarks />
        [DataMember(Name = "LocaleIds", IsRequired = false, Order = 3)]
        public StringCollection LocaleIds
        {
            get
            {
                return m_localeIds;
            }

            set
            {
                m_localeIds = value;

                if (value == null)
                {
                    m_localeIds = new StringCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "ServerUris", IsRequired = false, Order = 4)]
        public StringCollection ServerUris
        {
            get
            {
                return m_serverUris;
            }

            set
            {
                m_serverUris = value;

                if (value == null)
                {
                    m_serverUris = new StringCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.FindServersRequest; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.FindServersRequest_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.FindServersRequest_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.FindServersRequest_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("RequestHeader", RequestHeader, typeof(RequestHeader));
            encoder.WriteString("EndpointUrl", EndpointUrl);
            encoder.WriteStringArray("LocaleIds", LocaleIds);
            encoder.WriteStringArray("ServerUris", ServerUris);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            RequestHeader = (RequestHeader)decoder.ReadEncodeable("RequestHeader", typeof(RequestHeader));
            EndpointUrl = decoder.ReadString("EndpointUrl");
            LocaleIds = decoder.ReadStringArray("LocaleIds");
            ServerUris = decoder.ReadStringArray("ServerUris");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            FindServersRequest value = encodeable as FindServersRequest;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_requestHeader, value.m_requestHeader)) return false;
            if (!Utils.IsEqual(m_endpointUrl, value.m_endpointUrl)) return false;
            if (!Utils.IsEqual(m_localeIds, value.m_localeIds)) return false;
            if (!Utils.IsEqual(m_serverUris, value.m_serverUris)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (FindServersRequest)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            FindServersRequest clone = (FindServersRequest)base.MemberwiseClone();

            clone.m_requestHeader = (RequestHeader)Utils.Clone(this.m_requestHeader);
            clone.m_endpointUrl = (string)Utils.Clone(this.m_endpointUrl);
            clone.m_localeIds = (StringCollection)Utils.Clone(this.m_localeIds);
            clone.m_serverUris = (StringCollection)Utils.Clone(this.m_serverUris);

            return clone;
        }
        #endregion

        #region Private Fields
        private RequestHeader m_requestHeader;
        private string m_endpointUrl;
        private StringCollection m_localeIds;
        private StringCollection m_serverUris;
        #endregion
    }
    #endif
    #endregion

    #region FindServersResponse Class
    #if (!OPCUA_EXCLUDE_FindServersResponse)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class FindServersResponse : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public FindServersResponse()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_responseHeader = new ResponseHeader();
            m_servers = new ApplicationDescriptionCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "ResponseHeader", IsRequired = false, Order = 1)]
        public ResponseHeader ResponseHeader
        {
            get
            {
                return m_responseHeader;
            }

            set
            {
                m_responseHeader = value;

                if (value == null)
                {
                    m_responseHeader = new ResponseHeader();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "Servers", IsRequired = false, Order = 2)]
        public ApplicationDescriptionCollection Servers
        {
            get
            {
                return m_servers;
            }

            set
            {
                m_servers = value;

                if (value == null)
                {
                    m_servers = new ApplicationDescriptionCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.FindServersResponse; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.FindServersResponse_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.FindServersResponse_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.FindServersResponse_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("ResponseHeader", ResponseHeader, typeof(ResponseHeader));
            encoder.WriteEncodeableArray("Servers", Servers.ToArray(), typeof(ApplicationDescription));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ResponseHeader = (ResponseHeader)decoder.ReadEncodeable("ResponseHeader", typeof(ResponseHeader));
            Servers = (ApplicationDescriptionCollection)decoder.ReadEncodeableArray("Servers", typeof(ApplicationDescription));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            FindServersResponse value = encodeable as FindServersResponse;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_responseHeader, value.m_responseHeader)) return false;
            if (!Utils.IsEqual(m_servers, value.m_servers)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (FindServersResponse)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            FindServersResponse clone = (FindServersResponse)base.MemberwiseClone();

            clone.m_responseHeader = (ResponseHeader)Utils.Clone(this.m_responseHeader);
            clone.m_servers = (ApplicationDescriptionCollection)Utils.Clone(this.m_servers);

            return clone;
        }
        #endregion

        #region Private Fields
        private ResponseHeader m_responseHeader;
        private ApplicationDescriptionCollection m_servers;
        #endregion
    }
    #endif
    #endregion

    #region ServerOnNetwork Class
    #if (!OPCUA_EXCLUDE_ServerOnNetwork)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class ServerOnNetwork : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public ServerOnNetwork()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_recordId = (uint)0;
            m_serverName = null;
            m_discoveryUrl = null;
            m_serverCapabilities = new StringCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "RecordId", IsRequired = false, Order = 1)]
        public uint RecordId
        {
            get { return m_recordId;  }
            set { m_recordId = value; }
        }

        /// <remarks />
        [DataMember(Name = "ServerName", IsRequired = false, Order = 2)]
        public string ServerName
        {
            get { return m_serverName;  }
            set { m_serverName = value; }
        }

        /// <remarks />
        [DataMember(Name = "DiscoveryUrl", IsRequired = false, Order = 3)]
        public string DiscoveryUrl
        {
            get { return m_discoveryUrl;  }
            set { m_discoveryUrl = value; }
        }

        /// <remarks />
        [DataMember(Name = "ServerCapabilities", IsRequired = false, Order = 4)]
        public StringCollection ServerCapabilities
        {
            get
            {
                return m_serverCapabilities;
            }

            set
            {
                m_serverCapabilities = value;

                if (value == null)
                {
                    m_serverCapabilities = new StringCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.ServerOnNetwork; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.ServerOnNetwork_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.ServerOnNetwork_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.ServerOnNetwork_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteUInt32("RecordId", RecordId);
            encoder.WriteString("ServerName", ServerName);
            encoder.WriteString("DiscoveryUrl", DiscoveryUrl);
            encoder.WriteStringArray("ServerCapabilities", ServerCapabilities);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            RecordId = decoder.ReadUInt32("RecordId");
            ServerName = decoder.ReadString("ServerName");
            DiscoveryUrl = decoder.ReadString("DiscoveryUrl");
            ServerCapabilities = decoder.ReadStringArray("ServerCapabilities");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            ServerOnNetwork value = encodeable as ServerOnNetwork;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_recordId, value.m_recordId)) return false;
            if (!Utils.IsEqual(m_serverName, value.m_serverName)) return false;
            if (!Utils.IsEqual(m_discoveryUrl, value.m_discoveryUrl)) return false;
            if (!Utils.IsEqual(m_serverCapabilities, value.m_serverCapabilities)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (ServerOnNetwork)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ServerOnNetwork clone = (ServerOnNetwork)base.MemberwiseClone();

            clone.m_recordId = (uint)Utils.Clone(this.m_recordId);
            clone.m_serverName = (string)Utils.Clone(this.m_serverName);
            clone.m_discoveryUrl = (string)Utils.Clone(this.m_discoveryUrl);
            clone.m_serverCapabilities = (StringCollection)Utils.Clone(this.m_serverCapabilities);

            return clone;
        }
        #endregion

        #region Private Fields
        private uint m_recordId;
        private string m_serverName;
        private string m_discoveryUrl;
        private StringCollection m_serverCapabilities;
        #endregion
    }

    #region ServerOnNetworkCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfServerOnNetwork", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "ServerOnNetwork")]
    #if !NET_STANDARD
    public partial class ServerOnNetworkCollection : List<ServerOnNetwork>, ICloneable
    #else
    public partial class ServerOnNetworkCollection : List<ServerOnNetwork>
    #endif
    {
        #region Constructors
        /// <remarks />
        public ServerOnNetworkCollection() {}

        /// <remarks />
        public ServerOnNetworkCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public ServerOnNetworkCollection(IEnumerable<ServerOnNetwork> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator ServerOnNetworkCollection(ServerOnNetwork[] values)
        {
            if (values != null)
            {
                return new ServerOnNetworkCollection(values);
            }

            return new ServerOnNetworkCollection();
        }

        /// <remarks />
        public static explicit operator ServerOnNetwork[](ServerOnNetworkCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (ServerOnNetworkCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            ServerOnNetworkCollection clone = new ServerOnNetworkCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((ServerOnNetwork)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region FindServersOnNetworkRequest Class
    #if (!OPCUA_EXCLUDE_FindServersOnNetworkRequest)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class FindServersOnNetworkRequest : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public FindServersOnNetworkRequest()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_requestHeader = new RequestHeader();
            m_startingRecordId = (uint)0;
            m_maxRecordsToReturn = (uint)0;
            m_serverCapabilityFilter = new StringCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "RequestHeader", IsRequired = false, Order = 1)]
        public RequestHeader RequestHeader
        {
            get
            {
                return m_requestHeader;
            }

            set
            {
                m_requestHeader = value;

                if (value == null)
                {
                    m_requestHeader = new RequestHeader();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "StartingRecordId", IsRequired = false, Order = 2)]
        public uint StartingRecordId
        {
            get { return m_startingRecordId;  }
            set { m_startingRecordId = value; }
        }

        /// <remarks />
        [DataMember(Name = "MaxRecordsToReturn", IsRequired = false, Order = 3)]
        public uint MaxRecordsToReturn
        {
            get { return m_maxRecordsToReturn;  }
            set { m_maxRecordsToReturn = value; }
        }

        /// <remarks />
        [DataMember(Name = "ServerCapabilityFilter", IsRequired = false, Order = 4)]
        public StringCollection ServerCapabilityFilter
        {
            get
            {
                return m_serverCapabilityFilter;
            }

            set
            {
                m_serverCapabilityFilter = value;

                if (value == null)
                {
                    m_serverCapabilityFilter = new StringCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.FindServersOnNetworkRequest; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.FindServersOnNetworkRequest_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.FindServersOnNetworkRequest_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.FindServersOnNetworkRequest_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("RequestHeader", RequestHeader, typeof(RequestHeader));
            encoder.WriteUInt32("StartingRecordId", StartingRecordId);
            encoder.WriteUInt32("MaxRecordsToReturn", MaxRecordsToReturn);
            encoder.WriteStringArray("ServerCapabilityFilter", ServerCapabilityFilter);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            RequestHeader = (RequestHeader)decoder.ReadEncodeable("RequestHeader", typeof(RequestHeader));
            StartingRecordId = decoder.ReadUInt32("StartingRecordId");
            MaxRecordsToReturn = decoder.ReadUInt32("MaxRecordsToReturn");
            ServerCapabilityFilter = decoder.ReadStringArray("ServerCapabilityFilter");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            FindServersOnNetworkRequest value = encodeable as FindServersOnNetworkRequest;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_requestHeader, value.m_requestHeader)) return false;
            if (!Utils.IsEqual(m_startingRecordId, value.m_startingRecordId)) return false;
            if (!Utils.IsEqual(m_maxRecordsToReturn, value.m_maxRecordsToReturn)) return false;
            if (!Utils.IsEqual(m_serverCapabilityFilter, value.m_serverCapabilityFilter)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (FindServersOnNetworkRequest)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            FindServersOnNetworkRequest clone = (FindServersOnNetworkRequest)base.MemberwiseClone();

            clone.m_requestHeader = (RequestHeader)Utils.Clone(this.m_requestHeader);
            clone.m_startingRecordId = (uint)Utils.Clone(this.m_startingRecordId);
            clone.m_maxRecordsToReturn = (uint)Utils.Clone(this.m_maxRecordsToReturn);
            clone.m_serverCapabilityFilter = (StringCollection)Utils.Clone(this.m_serverCapabilityFilter);

            return clone;
        }
        #endregion

        #region Private Fields
        private RequestHeader m_requestHeader;
        private uint m_startingRecordId;
        private uint m_maxRecordsToReturn;
        private StringCollection m_serverCapabilityFilter;
        #endregion
    }
    #endif
    #endregion

    #region FindServersOnNetworkResponse Class
    #if (!OPCUA_EXCLUDE_FindServersOnNetworkResponse)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class FindServersOnNetworkResponse : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public FindServersOnNetworkResponse()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_responseHeader = new ResponseHeader();
            m_lastCounterResetTime = DateTime.MinValue;
            m_servers = new ServerOnNetworkCollection();
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "ResponseHeader", IsRequired = false, Order = 1)]
        public ResponseHeader ResponseHeader
        {
            get
            {
                return m_responseHeader;
            }

            set
            {
                m_responseHeader = value;

                if (value == null)
                {
                    m_responseHeader = new ResponseHeader();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "LastCounterResetTime", IsRequired = false, Order = 2)]
        public DateTime LastCounterResetTime
        {
            get { return m_lastCounterResetTime;  }
            set { m_lastCounterResetTime = value; }
        }

        /// <remarks />
        [DataMember(Name = "Servers", IsRequired = false, Order = 3)]
        public ServerOnNetworkCollection Servers
        {
            get
            {
                return m_servers;
            }

            set
            {
                m_servers = value;

                if (value == null)
                {
                    m_servers = new ServerOnNetworkCollection();
                }
            }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.FindServersOnNetworkResponse; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.FindServersOnNetworkResponse_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.FindServersOnNetworkResponse_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.FindServersOnNetworkResponse_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteEncodeable("ResponseHeader", ResponseHeader, typeof(ResponseHeader));
            encoder.WriteDateTime("LastCounterResetTime", LastCounterResetTime);
            encoder.WriteEncodeableArray("Servers", Servers.ToArray(), typeof(ServerOnNetwork));

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            ResponseHeader = (ResponseHeader)decoder.ReadEncodeable("ResponseHeader", typeof(ResponseHeader));
            LastCounterResetTime = decoder.ReadDateTime("LastCounterResetTime");
            Servers = (ServerOnNetworkCollection)decoder.ReadEncodeableArray("Servers", typeof(ServerOnNetwork));

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            FindServersOnNetworkResponse value = encodeable as FindServersOnNetworkResponse;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_responseHeader, value.m_responseHeader)) return false;
            if (!Utils.IsEqual(m_lastCounterResetTime, value.m_lastCounterResetTime)) return false;
            if (!Utils.IsEqual(m_servers, value.m_servers)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (FindServersOnNetworkResponse)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            FindServersOnNetworkResponse clone = (FindServersOnNetworkResponse)base.MemberwiseClone();

            clone.m_responseHeader = (ResponseHeader)Utils.Clone(this.m_responseHeader);
            clone.m_lastCounterResetTime = (DateTime)Utils.Clone(this.m_lastCounterResetTime);
            clone.m_servers = (ServerOnNetworkCollection)Utils.Clone(this.m_servers);

            return clone;
        }
        #endregion

        #region Private Fields
        private ResponseHeader m_responseHeader;
        private DateTime m_lastCounterResetTime;
        private ServerOnNetworkCollection m_servers;
        #endregion
    }
    #endif
    #endregion

    #region MessageSecurityMode Enumeration
    #if (!OPCUA_EXCLUDE_MessageSecurityMode)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum MessageSecurityMode
    {
        /// <remarks />
        [EnumMember(Value = "Invalid_0")]
        Invalid = 0,

        /// <remarks />
        [EnumMember(Value = "None_1")]
        None = 1,

        /// <remarks />
        [EnumMember(Value = "Sign_2")]
        Sign = 2,

        /// <remarks />
        [EnumMember(Value = "SignAndEncrypt_3")]
        SignAndEncrypt = 3,
    }
    #endif
    #endregion

    #region UserTokenType Enumeration
    #if (!OPCUA_EXCLUDE_UserTokenType)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public enum UserTokenType
    {
        /// <remarks />
        [EnumMember(Value = "Anonymous_0")]
        Anonymous = 0,

        /// <remarks />
        [EnumMember(Value = "UserName_1")]
        UserName = 1,

        /// <remarks />
        [EnumMember(Value = "Certificate_2")]
        Certificate = 2,

        /// <remarks />
        [EnumMember(Value = "IssuedToken_3")]
        IssuedToken = 3,
    }
    #endif
    #endregion

    #region UserTokenPolicy Class
    #if (!OPCUA_EXCLUDE_UserTokenPolicy)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class UserTokenPolicy : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public UserTokenPolicy()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_policyId = null;
            m_tokenType = UserTokenType.Anonymous;
            m_issuedTokenType = null;
            m_issuerEndpointUrl = null;
            m_securityPolicyUri = null;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "PolicyId", IsRequired = false, Order = 1)]
        public string PolicyId
        {
            get { return m_policyId;  }
            set { m_policyId = value; }
        }

        /// <remarks />
        [DataMember(Name = "TokenType", IsRequired = false, Order = 2)]
        public UserTokenType TokenType
        {
            get { return m_tokenType;  }
            set { m_tokenType = value; }
        }

        /// <remarks />
        [DataMember(Name = "IssuedTokenType", IsRequired = false, Order = 3)]
        public string IssuedTokenType
        {
            get { return m_issuedTokenType;  }
            set { m_issuedTokenType = value; }
        }

        /// <remarks />
        [DataMember(Name = "IssuerEndpointUrl", IsRequired = false, Order = 4)]
        public string IssuerEndpointUrl
        {
            get { return m_issuerEndpointUrl;  }
            set { m_issuerEndpointUrl = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityPolicyUri", IsRequired = false, Order = 5)]
        public string SecurityPolicyUri
        {
            get { return m_securityPolicyUri;  }
            set { m_securityPolicyUri = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.UserTokenPolicy; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.UserTokenPolicy_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.UserTokenPolicy_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.UserTokenPolicy_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("PolicyId", PolicyId);
            encoder.WriteEnumerated("TokenType", TokenType);
            encoder.WriteString("IssuedTokenType", IssuedTokenType);
            encoder.WriteString("IssuerEndpointUrl", IssuerEndpointUrl);
            encoder.WriteString("SecurityPolicyUri", SecurityPolicyUri);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            PolicyId = decoder.ReadString("PolicyId");
            TokenType = (UserTokenType)decoder.ReadEnumerated("TokenType", typeof(UserTokenType));
            IssuedTokenType = decoder.ReadString("IssuedTokenType");
            IssuerEndpointUrl = decoder.ReadString("IssuerEndpointUrl");
            SecurityPolicyUri = decoder.ReadString("SecurityPolicyUri");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            UserTokenPolicy value = encodeable as UserTokenPolicy;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_policyId, value.m_policyId)) return false;
            if (!Utils.IsEqual(m_tokenType, value.m_tokenType)) return false;
            if (!Utils.IsEqual(m_issuedTokenType, value.m_issuedTokenType)) return false;
            if (!Utils.IsEqual(m_issuerEndpointUrl, value.m_issuerEndpointUrl)) return false;
            if (!Utils.IsEqual(m_securityPolicyUri, value.m_securityPolicyUri)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (UserTokenPolicy)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UserTokenPolicy clone = (UserTokenPolicy)base.MemberwiseClone();

            clone.m_policyId = (string)Utils.Clone(this.m_policyId);
            clone.m_tokenType = (UserTokenType)Utils.Clone(this.m_tokenType);
            clone.m_issuedTokenType = (string)Utils.Clone(this.m_issuedTokenType);
            clone.m_issuerEndpointUrl = (string)Utils.Clone(this.m_issuerEndpointUrl);
            clone.m_securityPolicyUri = (string)Utils.Clone(this.m_securityPolicyUri);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_policyId;
        private UserTokenType m_tokenType;
        private string m_issuedTokenType;
        private string m_issuerEndpointUrl;
        private string m_securityPolicyUri;
        #endregion
    }

    #region UserTokenPolicyCollection Class
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [CollectionDataContract(Name = "ListOfUserTokenPolicy", Namespace = Opc.Ua.Namespaces.OpcUaXsd, ItemName = "UserTokenPolicy")]
    #if !NET_STANDARD
    public partial class UserTokenPolicyCollection : List<UserTokenPolicy>, ICloneable
    #else
    public partial class UserTokenPolicyCollection : List<UserTokenPolicy>
    #endif
    {
        #region Constructors
        /// <remarks />
        public UserTokenPolicyCollection() {}

        /// <remarks />
        public UserTokenPolicyCollection(int capacity) : base(capacity) {}

        /// <remarks />
        public UserTokenPolicyCollection(IEnumerable<UserTokenPolicy> collection) : base(collection) {}
        #endregion

        #region Static Operators
        /// <remarks />
        public static implicit operator UserTokenPolicyCollection(UserTokenPolicy[] values)
        {
            if (values != null)
            {
                return new UserTokenPolicyCollection(values);
            }

            return new UserTokenPolicyCollection();
        }

        /// <remarks />
        public static explicit operator UserTokenPolicy[](UserTokenPolicyCollection values)
        {
            if (values != null)
            {
                return values.ToArray();
            }

            return null;
        }
        #endregion

        #if !NET_STANDARD
        #region ICloneable Methods
        /// <remarks />
        public object Clone()
        {
            return (UserTokenPolicyCollection)this.MemberwiseClone();
        }
        #endregion
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            UserTokenPolicyCollection clone = new UserTokenPolicyCollection(this.Count);

            for (int ii = 0; ii < this.Count; ii++)
            {
                clone.Add((UserTokenPolicy)Utils.Clone(this[ii]));
            }

            return clone;
        }
    }
    #endregion
    #endif
    #endregion

    #region EndpointDescription Class
    #if (!OPCUA_EXCLUDE_EndpointDescription)
    /// <remarks />
    /// <exclude />
    [System.CodeDom.Compiler.GeneratedCodeAttribute("Opc.Ua.ModelCompiler", "1.0.0.0")]
    [DataContract(Namespace = Opc.Ua.Namespaces.OpcUaXsd)]
    public partial class EndpointDescription : IEncodeable, IJsonEncodeable
    {
        #region Constructors
        /// <remarks />
        public EndpointDescription()
        {
            Initialize();
        }
            
        [OnDeserializing]
        private void Initialize(StreamingContext context)
        {
            Initialize();
        }
            
        private void Initialize()
        {
            m_endpointUrl = null;
            m_server = new ApplicationDescription();
            m_serverCertificate = null;
            m_securityMode = MessageSecurityMode.Invalid;
            m_securityPolicyUri = null;
            m_userIdentityTokens = new UserTokenPolicyCollection();
            m_transportProfileUri = null;
            m_securityLevel = (byte)0;
        }
        #endregion

        #region Public Properties
        /// <remarks />
        [DataMember(Name = "EndpointUrl", IsRequired = false, Order = 1)]
        public string EndpointUrl
        {
            get { return m_endpointUrl;  }
            set { m_endpointUrl = value; }
        }

        /// <remarks />
        [DataMember(Name = "Server", IsRequired = false, Order = 2)]
        public ApplicationDescription Server
        {
            get
            {
                return m_server;
            }

            set
            {
                m_server = value;

                if (value == null)
                {
                    m_server = new ApplicationDescription();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "ServerCertificate", IsRequired = false, Order = 3)]
        public byte[] ServerCertificate
        {
            get { return m_serverCertificate;  }
            set { m_serverCertificate = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityMode", IsRequired = false, Order = 4)]
        public MessageSecurityMode SecurityMode
        {
            get { return m_securityMode;  }
            set { m_securityMode = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityPolicyUri", IsRequired = false, Order = 5)]
        public string SecurityPolicyUri
        {
            get { return m_securityPolicyUri;  }
            set { m_securityPolicyUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "UserIdentityTokens", IsRequired = false, Order = 6)]
        public UserTokenPolicyCollection UserIdentityTokens
        {
            get
            {
                return m_userIdentityTokens;
            }

            set
            {
                m_userIdentityTokens = value;

                if (value == null)
                {
                    m_userIdentityTokens = new UserTokenPolicyCollection();
                }
            }
        }

        /// <remarks />
        [DataMember(Name = "TransportProfileUri", IsRequired = false, Order = 7)]
        public string TransportProfileUri
        {
            get { return m_transportProfileUri;  }
            set { m_transportProfileUri = value; }
        }

        /// <remarks />
        [DataMember(Name = "SecurityLevel", IsRequired = false, Order = 8)]
        public byte SecurityLevel
        {
            get { return m_securityLevel;  }
            set { m_securityLevel = value; }
        }
        #endregion

        #region IEncodeable Members
        /// <summary cref="IEncodeable.TypeId" />
        public virtual ExpandedNodeId TypeId => DataTypeIds.EndpointDescription; 

        /// <summary cref="IEncodeable.BinaryEncodingId" />
        public virtual ExpandedNodeId BinaryEncodingId => ObjectIds.EndpointDescription_Encoding_DefaultBinary;

        /// <summary cref="IEncodeable.XmlEncodingId" />
        public virtual ExpandedNodeId XmlEncodingId => ObjectIds.EndpointDescription_Encoding_DefaultXml;
                    
        /// <summary cref="IJsonEncodeable.JsonEncodingId" />
        public virtual ExpandedNodeId JsonEncodingId => ObjectIds.EndpointDescription_Encoding_DefaultJson; 

        /// <summary cref="IEncodeable.Encode(IEncoder)" />
        public virtual void Encode(IEncoder encoder)
        {
            encoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            encoder.WriteString("EndpointUrl", EndpointUrl);
            encoder.WriteEncodeable("Server", Server, typeof(ApplicationDescription));
            encoder.WriteByteString("ServerCertificate", ServerCertificate);
            encoder.WriteEnumerated("SecurityMode", SecurityMode);
            encoder.WriteString("SecurityPolicyUri", SecurityPolicyUri);
            encoder.WriteEncodeableArray("UserIdentityTokens", UserIdentityTokens.ToArray(), typeof(UserTokenPolicy));
            encoder.WriteString("TransportProfileUri", TransportProfileUri);
            encoder.WriteByte("SecurityLevel", SecurityLevel);

            encoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.Decode(IDecoder)" />
        public virtual void Decode(IDecoder decoder)
        {
            decoder.PushNamespace(Opc.Ua.Namespaces.OpcUaXsd);

            EndpointUrl = decoder.ReadString("EndpointUrl");
            Server = (ApplicationDescription)decoder.ReadEncodeable("Server", typeof(ApplicationDescription));
            ServerCertificate = decoder.ReadByteString("ServerCertificate");
            SecurityMode = (MessageSecurityMode)decoder.ReadEnumerated("SecurityMode", typeof(MessageSecurityMode));
            SecurityPolicyUri = decoder.ReadString("SecurityPolicyUri");
            UserIdentityTokens = (UserTokenPolicyCollection)decoder.ReadEncodeableArray("UserIdentityTokens", typeof(UserTokenPolicy));
            TransportProfileUri = decoder.ReadString("TransportProfileUri");
            SecurityLevel = decoder.ReadByte("SecurityLevel");

            decoder.PopNamespace();
        }

        /// <summary cref="IEncodeable.IsEqual(IEncodeable)" />
        public virtual bool IsEqual(IEncodeable encodeable)
        {
            if (Object.ReferenceEquals(this, encodeable))
            {
                return true;
            }

            EndpointDescription value = encodeable as EndpointDescription;

            if (value == null)
            {
                return false;
            }

            if (!Utils.IsEqual(m_endpointUrl, value.m_endpointUrl)) return false;
            if (!Utils.IsEqual(m_server, value.m_server)) return false;
            if (!Utils.IsEqual(m_serverCertificate, value.m_serverCertificate)) return false;
            if (!Utils.IsEqual(m_securityMode, value.m_securityMode)) return false;
            if (!Utils.IsEqual(m_securityPolicyUri, value.m_securityPolicyUri)) return false;
            if (!Utils.IsEqual(m_userIdentityTokens, value.m_userIdentityTokens)) return false;
            if (!Utils.IsEqual(m_transportProfileUri, value.m_transportProfileUri)) return false;
            if (!Utils.IsEqual(m_securityLevel, value.m_securityLevel)) return false;

            return true;
        }

        #if !NET_STANDARD
        /// <summary cref="ICloneable.Clone" />
        public virtual object Clone()
        {
            return (EndpointDescription)this.MemberwiseClone();
        }
        #endif

        /// <summary cref="Object.MemberwiseClone" />
        public new object MemberwiseClone()
        {
            EndpointDescription clone = (EndpointDescription)base.MemberwiseClone();

            clone.m_endpointUrl = (string)Utils.Clone(this.m_endpointUrl);
            clone.m_server = (ApplicationDescription)Utils.Clone(this.m_server);
            clone.m_serverCertificate = (byte[])Utils.Clone(this.m_serverCertificate);
            clone.m_securityMode = (MessageSecurityMode)Utils.Clone(this.m_securityMode);
            clone.m_securityPolicyUri = (string)Utils.Clone(this.m_securityPolicyUri);
            clone.m_userIdentityTokens = (UserTokenPolicyCollection)Utils.Clone(this.m_userIdentityTokens);
            clone.m_transportProfileUri = (string)Utils.Clone(this.m_transportProfileUri);
            clone.m_securityLevel = (byte)Utils.Clone(this.m_securityLevel);

            return clone;
        }
        #endregion

        #region Private Fields
        private string m_endpointUrl;
        private ApplicationDescription m_server;
        private byte[] m_serverCertificate;
        private MessageSecurityMode m_securityMode;
        private string m_securityPolicyUri;
        private UserTokenPolicyCollection m_userIdentityTokens;
        private string m_transportProfileU