/* BEGIN software license
 *
 * msXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright (C) 2009--2020 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the msXpertSuite project.
 *
 * The msXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


/////////////////////// StdLib includes
#include <cstdio>

/////////////////////// Qt includes
#include <QTimer>
#include <QIcon>
#include <QDebug>
#include <QDir>
#include <QStandardPaths>
#include <QLocale>


/////////////////////// pappsomspp includes
#include <pappsomspp/core/msfile/msfilereader.h>
#include <pappsomspp/core/msfile/msfileaccessor.h>
#include <pappsomspp/core/msrun/msrunid.h>
#include <pappsomspp/core/msrun/msrunreader.h>
#include <pappsomspp/core/utils.h>
#include <pappsomspp/core/processing/combiners/tracepluscombiner.h>
#include <pappsomspp/core/processing/combiners/massspectrumpluscombiner.h>
#include <pappsomspp/core/trace/trace.h>
#include <pappsomspp/core/trace/maptrace.h>


/////////////////////// Local includes
#include "config.h"
#include "Application.hpp"
#include "ProgramWindow.hpp"
#include <pappsomspp/core/processing/combiners/mzintegrationparams.h>
#include "gui/MsRunDataSetTableView.hpp"
#include "gui/MsRunDataSetTableViewItem.hpp"
#include "gui/MsRunDataSetTableViewModel.hpp"
#include "gui/MsRunDataSetTableViewProxyModel.hpp"
#include "../nongui/globals.hpp"

namespace MsXpS
{
namespace MineXpert
{


Application::Application(int &argc, char **argv, const QString &moduleName)
  : QApplication{argc, argv}, m_applicationName{moduleName}
{
  // We are dealing with massive amounts of numerical data and the best locale
  // for that is "C".

  QLocale::setDefault(QLocale::C);

  QPixmap splash_pixmap(":/images/splashscreen.png");

  mpa_splash = new QSplashScreen(splash_pixmap, Qt::WindowStaysOnTopHint);
  mpa_splash->show();

  QTimer::singleShot(2000, this, SLOT(destroySplash()));

  // We now have to perform a number of initialization steps.

  QCoreApplication::setOrganizationName(m_applicationName);
  QCoreApplication::setOrganizationDomain("msxpertsuite.org");
  QCoreApplication::setApplicationName(m_applicationName);

  // The default window icon.
  QString pixmap_file_name = ":/images/icons/32x32/MineXpert3.png";
  QPixmap icon_pixmap(pixmap_file_name);
  QIcon icon(icon_pixmap);
  setWindowIcon(icon);


  // The configuration directory for all the settings of the program

  m_userConfigSettingsDirPath =
    QStandardPaths::writableLocation(QStandardPaths::ConfigLocation);

  if(m_userConfigSettingsDirPath.isEmpty())
    m_userConfigSettingsDirPath =
      QStandardPaths::writableLocation(QStandardPaths::HomeLocation);

  m_userConfigSettingsDirPath =
    m_userConfigSettingsDirPath + QDir::separator() + m_applicationName;

  m_userConfigSettingsFilePath =
    m_userConfigSettingsDirPath + QDir::separator() + "configSettings.ini";

  qDebug() << "m_userConfigSettingsDirPath:" << m_userConfigSettingsDirPath;
  qDebug() << "m_userConfigSettingsFilePath:" << m_userConfigSettingsFilePath;
}

Application::~Application()
{
}

void
Application::setDescription(QString &desc)
{
  m_description = desc;
}

QString
Application::description()
{
  return m_description;
}

void
Application::destroySplash()
{
  delete mpa_splash;
  mpa_splash = 0;
}

QString
Application::getUserConfigSettingsDirPath()
{
  return m_userConfigSettingsDirPath;
}

QString
Application::getUserConfigSettingsFilePath()
{
  return m_userConfigSettingsFilePath;
}

void
Application::saveMzIntegrationParamsToSettings(
  const pappso::MzIntegrationParams &mz_integration_params)
{

  QSettings settings(static_cast<Application *>(QCoreApplication::instance())
                       ->getUserConfigSettingsFilePath(),
                     QSettings::IniFormat);

  settings.beginGroup("MzIntegrationParams");

  settings.setValue("MzIntegrationParams", mz_integration_params.toString());

  settings.endGroup();
}

void
Application::eraseMzIntegrationParamsFromSettings()
{

  QSettings settings(static_cast<Application *>(QCoreApplication::instance())
                       ->getUserConfigSettingsFilePath(),
                     QSettings::IniFormat);

  settings.beginGroup("MzIntegrationParams");

  settings.remove("MzIntegrationParams");

  settings.endGroup();
}

// These settings are crucial throughout the whole program.
pappso::MzIntegrationParams::InitializationResult
Application::getMzIntegrationParamsFromSavedSettings(
  pappso::MzIntegrationParams &mz_integration_params)
{
  // qDebug();

  // Try to get the m/z integration parameters if they were stored as settings
  // by the user. Otherwise, return empty ones.

  QSettings settings(static_cast<Application *>(QCoreApplication::instance())
                       ->getUserConfigSettingsFilePath(),
                     QSettings::IniFormat);

  settings.beginGroup("MzIntegrationParams");

  QString mz_integration_params_settings_string =
    settings.value("MzIntegrationParams").toString();

  // qDebug() << "The text used to initialize the MzIntegrationParams is:"
  //          << mz_integration_params_settings_string;

  settings.endGroup();

  // Even if the string is empty, call the function that will set default
  // values.

  pappso::MzIntegrationParams::InitializationResult settings_results =
    mz_integration_params.initialize(mz_integration_params_settings_string);

  return settings_results;
}

////////////////////////////// Scripting HUB /////////////////////////////
////////////////////////////// Scripting HUB /////////////////////////////
bool
Application::publishQObject(QObject *object,
                            QObject *parent,
                            QJSValue &js_value,
                            const QString &name,
                            const QString &alias,
                            const QString &comment,
                            const QString &help,
                            const QString &tool_tip,
                            const QColor &color,
                            QJSEngine::ObjectOwnership ownership)
{
  return mp_programWindow->getScriptingWnd()->publishQObject(object,
                                                             parent,
                                                             js_value,
                                                             name,
                                                             alias,
                                                             comment,
                                                             help,
                                                             tool_tip,
                                                             color,
                                                             ownership);
}

void
Application::setProgramWindow(ProgramWindow *program_window_p)
{
  mp_programWindow = program_window_p;
}

} // namespace MineXpert

} // namespace MsXpS
