#!/usr/bin/env python3
# Copyright (c) Meta Platforms, Inc. and affiliates.
# SPDX-License-Identifier: LGPL-2.1-or-later

import argparse
import re
import sys

if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description="Generate PageFlag() helpers from include/linux/page-flags.h"
    )
    args = parser.parse_args()

    flags = {
        # PageUptodate() isn't defined with PAGEFLAG because it needs
        # additional memory barriers, but other than that it's the same.
        "Uptodate": "uptodate",
    }
    for match in re.finditer(
        r"\b(?:|__|TEST)PAGEFLAG\s*\(\s*(\w+)(?<!uname)\s*,\s*(\w+)\s*,\s*\w+\s*\)",
        sys.stdin.read(),
    ):
        if flags.setdefault(match.group(1), match.group(2)) != match.group(2):
            sys.exit(f"{match.group('uname')} has multiple lowercase names?")

    print("    # Generated by scripts/generate_page_flag_getters.py.")
    for uname, lname in sorted(flags.items()):
        print(f'    "Page{uname}",')
    print(")")
    print()
    for uname, lname in sorted(flags.items()):
        print(
            f'''
def Page{uname}(page: Object) -> bool:
    """
    Return whether the ``PG_{lname}`` flag is set on a page.

    :param page: ``struct page *``
    """
    try:
        flag = page.prog_["PG_{lname}"]
    except KeyError:
        return False
    return bool(page.flags & (1 << flag))
'''
        )
    print()
    print("# End generated by scripts/generate_page_flag_getters.py.")
