/*
 * Copyright (c) 2022-2024 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef ARM_COMPUTE_ENABLE_SVE

#include "arm_gemm.hpp"
#include "../../utils.hpp"

#include <cassert>
#include <limits>

namespace arm_gemm {

void sve_ffhybrid_fp16_mla_6x4VL_a64fx (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<__fp16> A_arg,
    size_t M, size_t N, const __fp16 *B_ptr, size_t B_stride, IndirectOutputArg<__fp16> output_arg,
    const __fp16 *bias, Activation act, bool accumulate
)
{
    struct KernelArgs {
        __fp16 maxval = static_cast<__fp16>(std::numeric_limits<float>::infinity());
        __fp16 minval = - static_cast<__fp16>(std::numeric_limits<float>::infinity());
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const __fp16 *B_ptr = {};
        const __fp16 *cur_B_ptr = {};
        size_t B_stride = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
        void *output_ptr = {};
        const __fp16 *bias = {};
    } ka;

    unsigned long flags=0;
    void *input_ptr;

    if (output_arg.is_indirect) {
        ka.output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        ka.output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    if (accumulate) {
        flags |= 0x1;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    ka.bias = bias;
    ka.B_stride = B_stride;
    switch(act.type) {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            ka.maxval = static_cast<__fp16>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            ka.minval = 0;
            flags |= 0x2;
            break;
    }
    __asm__ __volatile__(
      "ptrue p4.b\n"
      "1:"  // Row loop
      "cmp %x[M], #0x6\n"
      "bge 66f\n"
      "cmp %x[M], #0x4\n"
      "bgt 53f\n"
      "beq 40f\n"
      "cmp %x[M], #0x2\n"
      "bgt 27f\n"
      "beq 14f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x15, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x14, [%x[args_ptr], %[offsetof_N]]\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "2:"  // Height 1: Column loop
      "ldr x12, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_stride]]\n"
      "cnth x21, ALL, MUL #3\n"
      "add x11, x12, x20, LSL #1\n"
      "add x10, x11, x20, LSL #1\n"
      "add x9, x10, x20, LSL #1\n"
      "add x20, x9, x20, LSL #1\n"
      "cmp x14, x21\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "bgt 3f\n"
      "dech x21\n"
      "mov x9, x12\n"
      "cmp x14, x21\n"
      "bgt 3f\n"
      "dech x21\n"
      "mov x10, x12\n"
      "cmp x14, x21\n"
      "bgt 3f\n"
      "mov x11, x12\n"
      "3:"  // Height 1: B setup done
      "mov x20, #0x0\n"
      "whilelt p3.h, x20, x14\n"
      "inch x20\n"
      "whilelt p2.h, x20, x14\n"
      "inch x20\n"
      "whilelt p1.h, x20, x14\n"
      "inch x20\n"
      "whilelt p0.h, x20, x14\n"
      "cbz x15, 4f\n"
      "ld1h { z8.h }, p4/Z, [x15]\n"
      "ld1h { z9.h }, p4/Z, [x15, #1, MUL VL]\n"
      "ld1h { z10.h }, p4/Z, [x15, #2, MUL VL]\n"
      "ld1h { z11.h }, p4/Z, [x15, #3, MUL VL]\n"
      "addvl x15, x15, #4\n"
      "b 6f\n"
      "4:"  // Height 1: no bias
      "tbz %x[flags], #0, 5f\n"
      "ld1h { z8.h }, p3/Z, [x13]\n"
      "ld1h { z9.h }, p2/Z, [x13, #1, MUL VL]\n"
      "ld1h { z10.h }, p1/Z, [x13, #2, MUL VL]\n"
      "ld1h { z11.h }, p0/Z, [x13, #3, MUL VL]\n"
      "b 6f\n"
      "5:"  // Height 1: no accumulate
      "mov z8.b, #0x0\n"
      "mov z9.b, #0x0\n"
      "mov z10.b, #0x0\n"
      "mov z11.b, #0x0\n"
      "6:"  // Height 1: setup done
      "mov x28, #0x0\n"
      "7:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 8f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "cbnz x28, 9f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "b 9f\n"
      "8:"  // Height 1: setup direct input
      "mov x26, %x[input_ptr]\n"
      "9:"  // Height 1: input setup done
      "subs x27, x27, #0x1\n"
      "ld1rh { z0.h }, p4/Z, [x26]\n"
      "ld1h { z6.h }, p4/Z, [x12]\n"
      "ld1h { z7.h }, p4/Z, [x11]\n"
      "ble 11f\n"
      "10:"  // Height 1: Multiply loop: Main loop
      "addvl x12, x12, #1\n"
      "fmla z8.h, p4/M, z6.h, z0.h\n"
      "ld1h { z17.h }, p4/Z, [x10]\n"
      "addvl x11, x11, #1\n"
      "fmla z9.h, p4/M, z7.h, z0.h\n"
      "ld1h { z16.h }, p4/Z, [x9]\n"
      "add x26, x26, #0x2\n"
      "subs x27, x27, #0x1\n"
      "addvl x10, x10, #1\n"
      "addvl x9, x9, #1\n"
      "fmla z10.h, p4/M, z17.h, z0.h\n"
      "ld1h { z6.h }, p4/Z, [x12]\n"
      "fmla z11.h, p4/M, z16.h, z0.h\n"
      "ld1rh { z0.h }, p4/Z, [x26]\n"
      "ld1h { z7.h }, p4/Z, [x11]\n"
      "bgt 10b\n"
      "11:"  // Height 1: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "fmla z8.h, p4/M, z6.h, z0.h\n"
      "ld1h { z17.h }, p4/Z, [x10]\n"
      "fmla z9.h, p4/M, z7.h, z0.h\n"
      "ld1h { z16.h }, p4/Z, [x9]\n"
      "add x28, x28, #0x1\n"
      "addvl x12, x12, #1\n"
      "addvl x11, x11, #1\n"
      "addvl x10, x10, #1\n"
      "cmp x28, x20\n"
      "addvl x9, x9, #1\n"
      "fmla z10.h, p4/M, z17.h, z0.h\n"
      "fmla z11.h, p4/M, z16.h, z0.h\n"
      "bne 7b\n"
      "tbz %x[flags], #1, 12f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rh { z17.h }, p4/Z, [x21]\n"
      "ld1rh { z16.h }, p4/Z, [x20]\n"
      "fmin z8.h, p4/M, z8.h, z17.h\n"
      "fmin z9.h, p4/M, z9.h, z17.h\n"
      "fmin z10.h, p4/M, z10.h, z17.h\n"
      "fmin z11.h, p4/M, z11.h, z17.h\n"
      "fmax z8.h, p4/M, z8.h, z16.h\n"
      "fmax z9.h, p4/M, z9.h, z16.h\n"
      "fmax z10.h, p4/M, z10.h, z16.h\n"
      "fmax z11.h, p4/M, z11.h, z16.h\n"
      "12:"  // Height 1: No activation
      "st1h { z8.h }, p3, [x13]\n"
      "st1h { z9.h }, p2, [x13, #1, MUL VL]\n"
      "st1h { z10.h }, p1, [x13, #2, MUL VL]\n"
      "st1h { z11.h }, p0, [x13, #3, MUL VL]\n"
      "addvl x13, x13, #4\n"
      "13:"  // Height 1: Writeback done
      "dech x14, ALL, MUL #4\n"
      "cmp x14, XZR\n"
      "bgt 2b\n"
      "b 80f\n"
      "14:"  // Height 2
      "ldr x20, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x15, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x14, [%x[args_ptr], %[offsetof_N]]\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "15:"  // Height 2: Column loop
      "ldr x12, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_stride]]\n"
      "cnth x21, ALL, MUL #3\n"
      "add x11, x12, x20, LSL #1\n"
      "add x10, x11, x20, LSL #1\n"
      "add x9, x10, x20, LSL #1\n"
      "add x20, x9, x20, LSL #1\n"
      "cmp x14, x21\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "bgt 16f\n"
      "dech x21\n"
      "mov x9, x12\n"
      "cmp x14, x21\n"
      "bgt 16f\n"
      "dech x21\n"
      "mov x10, x12\n"
      "cmp x14, x21\n"
      "bgt 16f\n"
      "mov x11, x12\n"
      "16:"  // Height 2: B setup done
      "mov x20, #0x0\n"
      "whilelt p3.h, x20, x14\n"
      "inch x20\n"
      "whilelt p2.h, x20, x14\n"
      "inch x20\n"
      "whilelt p1.h, x20, x14\n"
      "inch x20\n"
      "whilelt p0.h, x20, x14\n"
      "cbz x15, 17f\n"
      "ld1h { z8.h }, p4/Z, [x15]\n"
      "ld1h { z9.h }, p4/Z, [x15, #1, MUL VL]\n"
      "ld1h { z10.h }, p4/Z, [x15, #2, MUL VL]\n"
      "ld1h { z11.h }, p4/Z, [x15, #3, MUL VL]\n"
      "addvl x15, x15, #4\n"
      "mov z12.d, z8.d\n"
      "mov z13.d, z9.d\n"
      "mov z14.d, z10.d\n"
      "mov z15.d, z11.d\n"
      "b 19f\n"
      "17:"  // Height 2: no bias
      "tbz %x[flags], #0, 18f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1h { z8.h }, p3/Z, [x13]\n"
      "ld1h { z9.h }, p2/Z, [x13, #1, MUL VL]\n"
      "ld1h { z10.h }, p1/Z, [x13, #2, MUL VL]\n"
      "ld1h { z11.h }, p0/Z, [x13, #3, MUL VL]\n"
      "add x20, x13, x20, LSL #1\n"
      "ld1h { z12.h }, p3/Z, [x20]\n"
      "ld1h { z13.h }, p2/Z, [x20, #1, MUL VL]\n"
      "ld1h { z14.h }, p1/Z, [x20, #2, MUL VL]\n"
      "ld1h { z15.h }, p0/Z, [x20, #3, MUL VL]\n"
      "b 19f\n"
      "18:"  // Height 2: no accumulate
      "mov z8.b, #0x0\n"
      "mov z9.b, #0x0\n"
      "mov z10.b, #0x0\n"
      "mov z11.b, #0x0\n"
      "mov z12.b, #0x0\n"
      "mov z13.b, #0x0\n"
      "mov z14.b, #0x0\n"
      "mov z15.b, #0x0\n"
      "19:"  // Height 2: setup done
      "mov x28, #0x0\n"
      "20:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 21f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "cbnz x28, 22f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "b 22f\n"
      "21:"  // Height 2: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "22:"  // Height 2: input setup done
      "subs x27, x27, #0x1\n"
      "ld1rh { z0.h }, p4/Z, [x26]\n"
      "ld1rh { z1.h }, p4/Z, [x25]\n"
      "ld1h { z6.h }, p4/Z, [x12]\n"
      "ld1h { z7.h }, p4/Z, [x11]\n"
      "ble 24f\n"
      "23:"  // Height 2: Multiply loop: Main loop
      "addvl x12, x12, #1\n"
      "fmla z8.h, p4/M, z6.h, z0.h\n"
      "fmla z12.h, p4/M, z6.h, z1.h\n"
      "ld1h { z17.h }, p4/Z, [x10]\n"
      "addvl x11, x11, #1\n"
      "fmla z9.h, p4/M, z7.h, z0.h\n"
      "fmla z13.h, p4/M, z7.h, z1.h\n"
      "ld1h { z16.h }, p4/Z, [x9]\n"
      "add x26, x26, #0x2\n"
      "subs x27, x27, #0x1\n"
      "add x25, x25, #0x2\n"
      "addvl x10, x10, #1\n"
      "fmla z10.h, p4/M, z17.h, z0.h\n"
      "fmla z14.h, p4/M, z17.h, z1.h\n"
      "addvl x9, x9, #1\n"
      "ld1h { z6.h }, p4/Z, [x12]\n"
      "fmla z11.h, p4/M, z16.h, z0.h\n"
      "ld1rh { z0.h }, p4/Z, [x26]\n"
      "fmla z15.h, p4/M, z16.h, z1.h\n"
      "ld1rh { z1.h }, p4/Z, [x25]\n"
      "ld1h { z7.h }, p4/Z, [x11]\n"
      "bgt 23b\n"
      "24:"  // Height 2: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "fmla z8.h, p4/M, z6.h, z0.h\n"
      "fmla z12.h, p4/M, z6.h, z1.h\n"
      "ld1h { z17.h }, p4/Z, [x10]\n"
      "fmla z9.h, p4/M, z7.h, z0.h\n"
      "fmla z13.h, p4/M, z7.h, z1.h\n"
      "ld1h { z16.h }, p4/Z, [x9]\n"
      "add x28, x28, #0x1\n"
      "addvl x12, x12, #1\n"
      "addvl x11, x11, #1\n"
      "cmp x28, x20\n"
      "addvl x10, x10, #1\n"
      "fmla z10.h, p4/M, z17.h, z0.h\n"
      "fmla z14.h, p4/M, z17.h, z1.h\n"
      "addvl x9, x9, #1\n"
      "fmla z11.h, p4/M, z16.h, z0.h\n"
      "fmla z15.h, p4/M, z16.h, z1.h\n"
      "bne 20b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x26, x13, x20, LSL #1\n"
      "tbz %x[flags], #1, 25f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rh { z17.h }, p4/Z, [x21]\n"
      "ld1rh { z16.h }, p4/Z, [x20]\n"
      "fmin z8.h, p4/M, z8.h, z17.h\n"
      "fmin z9.h, p4/M, z9.h, z17.h\n"
      "fmin z10.h, p4/M, z10.h, z17.h\n"
      "fmin z11.h, p4/M, z11.h, z17.h\n"
      "fmin z12.h, p4/M, z12.h, z17.h\n"
      "fmin z13.h, p4/M, z13.h, z17.h\n"
      "fmin z14.h, p4/M, z14.h, z17.h\n"
      "fmin z15.h, p4/M, z15.h, z17.h\n"
      "fmax z8.h, p4/M, z8.h, z16.h\n"
      "fmax z9.h, p4/M, z9.h, z16.h\n"
      "fmax z10.h, p4/M, z10.h, z16.h\n"
      "fmax z11.h, p4/M, z11.h, z16.h\n"
      "fmax z12.h, p4/M, z12.h, z16.h\n"
      "fmax z13.h, p4/M, z13.h, z16.h\n"
      "fmax z14.h, p4/M, z14.h, z16.h\n"
      "fmax z15.h, p4/M, z15.h, z16.h\n"
      "25:"  // Height 2: No activation
      "st1h { z8.h }, p3, [x13]\n"
      "st1h { z9.h }, p2, [x13, #1, MUL VL]\n"
      "st1h { z10.h }, p1, [x13, #2, MUL VL]\n"
      "st1h { z11.h }, p0, [x13, #3, MUL VL]\n"
      "addvl x13, x13, #4\n"
      "st1h { z12.h }, p3, [x26]\n"
      "st1h { z13.h }, p2, [x26, #1, MUL VL]\n"
      "st1h { z14.h }, p1, [x26, #2, MUL VL]\n"
      "st1h { z15.h }, p0, [x26, #3, MUL VL]\n"
      "26:"  // Height 2: Writeback done
      "dech x14, ALL, MUL #4\n"
      "cmp x14, XZR\n"
      "bgt 15b\n"
      "b 80f\n"
      "27:"  // Height 3
      "ldr x20, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x15, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x14, [%x[args_ptr], %[offsetof_N]]\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "28:"  // Height 3: Column loop
      "ldr x12, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_stride]]\n"
      "cnth x21, ALL, MUL #3\n"
      "add x11, x12, x20, LSL #1\n"
      "add x10, x11, x20, LSL #1\n"
      "add x9, x10, x20, LSL #1\n"
      "add x20, x9, x20, LSL #1\n"
      "cmp x14, x21\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "bgt 29f\n"
      "dech x21\n"
      "mov x9, x12\n"
      "cmp x14, x21\n"
      "bgt 29f\n"
      "dech x21\n"
      "mov x10, x12\n"
      "cmp x14, x21\n"
      "bgt 29f\n"
      "mov x11, x12\n"
      "29:"  // Height 3: B setup done
      "mov x20, #0x0\n"
      "whilelt p3.h, x20, x14\n"
      "inch x20\n"
      "whilelt p2.h, x20, x14\n"
      "inch x20\n"
      "whilelt p1.h, x20, x14\n"
      "inch x20\n"
      "whilelt p0.h, x20, x14\n"
      "cbz x15, 30f\n"
      "ld1h { z8.h }, p4/Z, [x15]\n"
      "ld1h { z9.h }, p4/Z, [x15, #1, MUL VL]\n"
      "ld1h { z10.h }, p4/Z, [x15, #2, MUL VL]\n"
      "ld1h { z11.h }, p4/Z, [x15, #3, MUL VL]\n"
      "addvl x15, x15, #4\n"
      "mov z12.d, z8.d\n"
      "mov z13.d, z9.d\n"
      "mov z16.d, z8.d\n"
      "mov z17.d, z9.d\n"
      "mov z14.d, z10.d\n"
      "mov z15.d, z11.d\n"
      "mov z18.d, z10.d\n"
      "mov z19.d, z11.d\n"
      "b 32f\n"
      "30:"  // Height 3: no bias
      "tbz %x[flags], #0, 31f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1h { z8.h }, p3/Z, [x13]\n"
      "ld1h { z9.h }, p2/Z, [x13, #1, MUL VL]\n"
      "ld1h { z10.h }, p1/Z, [x13, #2, MUL VL]\n"
      "ld1h { z11.h }, p0/Z, [x13, #3, MUL VL]\n"
      "add x21, x13, x20, LSL #1\n"
      "add x20, x21, x20, LSL #1\n"
      "ld1h { z12.h }, p3/Z, [x21]\n"
      "ld1h { z13.h }, p2/Z, [x21, #1, MUL VL]\n"
      "ld1h { z14.h }, p1/Z, [x21, #2, MUL VL]\n"
      "ld1h { z15.h }, p0/Z, [x21, #3, MUL VL]\n"
      "ld1h { z16.h }, p3/Z, [x20]\n"
      "ld1h { z17.h }, p2/Z, [x20, #1, MUL VL]\n"
      "ld1h { z18.h }, p1/Z, [x20, #2, MUL VL]\n"
      "ld1h { z19.h }, p0/Z, [x20, #3, MUL VL]\n"
      "b 32f\n"
      "31:"  // Height 3: no accumulate
      "mov z8.b, #0x0\n"
      "mov z9.b, #0x0\n"
      "mov z10.b, #0x0\n"
      "mov z11.b, #0x0\n"
      "mov z12.b, #0x0\n"
      "mov z13.b, #0x0\n"
      "mov z14.b, #0x0\n"
      "mov z15.b, #0x0\n"
      "mov z16.b, #0x0\n"
      "mov z17.b, #0x0\n"
      "mov z18.b, #0x0\n"
      "mov z19.b, #0x0\n"
      "32:"  // Height 3: setup done
      "mov x28, #0x0\n"
      "33:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 34f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "cbnz x28, 35f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "add x24, x24, x20, LSL #1\n"
      "b 35f\n"
      "34:"  // Height 3: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "add x24, x25, x21, LSL #1\n"
      "35:"  // Height 3: input setup done
      "subs x27, x27, #0x1\n"
      "ld1rh { z0.h }, p4/Z, [x26]\n"
      "ld1rh { z1.h }, p4/Z, [x25]\n"
      "ld1rh { z2.h }, p4/Z, [x24]\n"
      "ld1h { z6.h }, p4/Z, [x12]\n"
      "ld1h { z7.h }, p4/Z, [x11]\n"
      "ble 37f\n"
      "36:"  // Height 3: Multiply loop: Main loop
      "addvl x12, x12, #1\n"
      "fmla z8.h, p4/M, z6.h, z0.h\n"
      "fmla z12.h, p4/M, z6.h, z1.h\n"
      "addvl x11, x11, #1\n"
      "fmla z16.h, p4/M, z6.h, z2.h\n"
      "ld1h { z21.h }, p4/Z, [x10]\n"
      "fmla z9.h, p4/M, z7.h, z0.h\n"
      "add x26, x26, #0x2\n"
      "fmla z13.h, p4/M, z7.h, z1.h\n"
      "fmla z17.h, p4/M, z7.h, z2.h\n"
      "ld1h { z20.h }, p4/Z, [x9]\n"
      "subs x27, x27, #0x1\n"
      "add x25, x25, #0x2\n"
      "add x24, x24, #0x2\n"
      "addvl x10, x10, #1\n"
      "addvl x9, x9, #1\n"
      "fmla z10.h, p4/M, z21.h, z0.h\n"
      "fmla z14.h, p4/M, z21.h, z1.h\n"
      "fmla z18.h, p4/M, z21.h, z2.h\n"
      "ld1h { z6.h }, p4/Z, [x12]\n"
      "fmla z11.h, p4/M, z20.h, z0.h\n"
      "ld1rh { z0.h }, p4/Z, [x26]\n"
      "fmla z15.h, p4/M, z20.h, z1.h\n"
      "fmla z19.h, p4/M, z20.h, z2.h\n"
      "ld1rh { z1.h }, p4/Z, [x25]\n"
      "ld1rh { z2.h }, p4/Z, [x24]\n"
      "ld1h { z7.h }, p4/Z, [x11]\n"
      "bgt 36b\n"
      "37:"  // Height 3: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "fmla z8.h, p4/M, z6.h, z0.h\n"
      "fmla z12.h, p4/M, z6.h, z1.h\n"
      "add x28, x28, #0x1\n"
      "fmla z16.h, p4/M, z6.h, z2.h\n"
      "ld1h { z21.h }, p4/Z, [x10]\n"
      "fmla z9.h, p4/M, z7.h, z0.h\n"
      "addvl x12, x12, #1\n"
      "fmla z13.h, p4/M, z7.h, z1.h\n"
      "fmla z17.h, p4/M, z7.h, z2.h\n"
      "ld1h { z20.h }, p4/Z, [x9]\n"
      "addvl x11, x11, #1\n"
      "cmp x28, x20\n"
      "addvl x10, x10, #1\n"
      "addvl x9, x9, #1\n"
      "fmla z10.h, p4/M, z21.h, z0.h\n"
      "fmla z14.h, p4/M, z21.h, z1.h\n"
      "fmla z18.h, p4/M, z21.h, z2.h\n"
      "fmla z11.h, p4/M, z20.h, z0.h\n"
      "fmla z15.h, p4/M, z20.h, z1.h\n"
      "fmla z19.h, p4/M, z20.h, z2.h\n"
      "bne 33b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x26, x13, x20, LSL #1\n"
      "add x25, x26, x20, LSL #1\n"
      "tbz %x[flags], #1, 38f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rh { z21.h }, p4/Z, [x21]\n"
      "ld1rh { z20.h }, p4/Z, [x20]\n"
      "fmin z8.h, p4/M, z8.h, z21.h\n"
      "fmin z9.h, p4/M, z9.h, z21.h\n"
      "fmin z10.h, p4/M, z10.h, z21.h\n"
      "fmin z11.h, p4/M, z11.h, z21.h\n"
      "fmin z12.h, p4/M, z12.h, z21.h\n"
      "fmin z13.h, p4/M, z13.h, z21.h\n"
      "fmin z14.h, p4/M, z14.h, z21.h\n"
      "fmin z15.h, p4/M, z15.h, z21.h\n"
      "fmin z16.h, p4/M, z16.h, z21.h\n"
      "fmin z17.h, p4/M, z17.h, z21.h\n"
      "fmin z18.h, p4/M, z18.h, z21.h\n"
      "fmin z19.h, p4/M, z19.h, z21.h\n"
      "fmax z8.h, p4/M, z8.h, z20.h\n"
      "fmax z9.h, p4/M, z9.h, z20.h\n"
      "fmax z10.h, p4/M, z10.h, z20.h\n"
      "fmax z11.h, p4/M, z11.h, z20.h\n"
      "fmax z12.h, p4/M, z12.h, z20.h\n"
      "fmax z13.h, p4/M, z13.h, z20.h\n"
      "fmax z14.h, p4/M, z14.h, z20.h\n"
      "fmax z15.h, p4/M, z15.h, z20.h\n"
      "fmax z16.h, p4/M, z16.h, z20.h\n"
      "fmax z17.h, p4/M, z17.h, z20.h\n"
      "fmax z18.h, p4/M, z18.h, z20.h\n"
      "fmax z19.h, p4/M, z19.h, z20.h\n"
      "38:"  // Height 3: No activation
      "st1h { z8.h }, p3, [x13]\n"
      "st1h { z9.h }, p2, [x13, #1, MUL VL]\n"
      "st1h { z10.h }, p1, [x13, #2, MUL VL]\n"
      "st1h { z11.h }, p0, [x13, #3, MUL VL]\n"
      "addvl x13, x13, #4\n"
      "st1h { z12.h }, p3, [x26]\n"
      "st1h { z13.h }, p2, [x26, #1, MUL VL]\n"
      "st1h { z14.h }, p1, [x26, #2, MUL VL]\n"
      "st1h { z15.h }, p0, [x26, #3, MUL VL]\n"
      "st1h { z16.h }, p3, [x25]\n"
      "st1h { z17.h }, p2, [x25, #1, MUL VL]\n"
      "st1h { z18.h }, p1, [x25, #2, MUL VL]\n"
      "st1h { z19.h }, p0, [x25, #3, MUL VL]\n"
      "39:"  // Height 3: Writeback done
      "dech x14, ALL, MUL #4\n"
      "cmp x14, XZR\n"
      "bgt 28b\n"
      "b 80f\n"
      "40:"  // Height 4
      "ldr x20, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x15, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x14, [%x[args_ptr], %[offsetof_N]]\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "41:"  // Height 4: Column loop
      "ldr x12, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_stride]]\n"
      "cnth x21, ALL, MUL #3\n"
      "add x11, x12, x20, LSL #1\n"
      "add x10, x11, x20, LSL #1\n"
      "add x9, x10, x20, LSL #1\n"
      "add x20, x9, x20, LSL #1\n"
      "cmp x14, x21\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "bgt 42f\n"
      "dech x21\n"
      "mov x9, x12\n"
      "cmp x14, x21\n"
      "bgt 42f\n"
      "dech x21\n"
      "mov x10, x12\n"
      "cmp x14, x21\n"
      "bgt 42f\n"
      "mov x11, x12\n"
      "42:"  // Height 4: B setup done
      "mov x20, #0x0\n"
      "whilelt p3.h, x20, x14\n"
      "inch x20\n"
      "whilelt p2.h, x20, x14\n"
      "inch x20\n"
      "whilelt p1.h, x20, x14\n"
      "inch x20\n"
      "whilelt p0.h, x20, x14\n"
      "cbz x15, 43f\n"
      "ld1h { z8.h }, p4/Z, [x15]\n"
      "ld1h { z9.h }, p4/Z, [x15, #1, MUL VL]\n"
      "ld1h { z10.h }, p4/Z, [x15, #2, MUL VL]\n"
      "ld1h { z11.h }, p4/Z, [x15, #3, MUL VL]\n"
      "addvl x15, x15, #4\n"
      "mov z12.d, z8.d\n"
      "mov z13.d, z9.d\n"
      "mov z16.d, z8.d\n"
      "mov z17.d, z9.d\n"
      "mov z20.d, z8.d\n"
      "mov z14.d, z10.d\n"
      "mov z15.d, z11.d\n"
      "mov z18.d, z10.d\n"
      "mov z19.d, z11.d\n"
      "mov z21.d, z9.d\n"
      "mov z22.d, z10.d\n"
      "mov z23.d, z11.d\n"
      "b 45f\n"
      "43:"  // Height 4: no bias
      "tbz %x[flags], #0, 44f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1h { z8.h }, p3/Z, [x13]\n"
      "ld1h { z9.h }, p2/Z, [x13, #1, MUL VL]\n"
      "ld1h { z10.h }, p1/Z, [x13, #2, MUL VL]\n"
      "ld1h { z11.h }, p0/Z, [x13, #3, MUL VL]\n"
      "add x22, x13, x20, LSL #1\n"
      "add x21, x22, x20, LSL #1\n"
      "add x20, x21, x20, LSL #1\n"
      "ld1h { z12.h }, p3/Z, [x22]\n"
      "ld1h { z13.h }, p2/Z, [x22, #1, MUL VL]\n"
      "ld1h { z14.h }, p1/Z, [x22, #2, MUL VL]\n"
      "ld1h { z15.h }, p0/Z, [x22, #3, MUL VL]\n"
      "ld1h { z16.h }, p3/Z, [x21]\n"
      "ld1h { z17.h }, p2/Z, [x21, #1, MUL VL]\n"
      "ld1h { z18.h }, p1/Z, [x21, #2, MUL VL]\n"
      "ld1h { z19.h }, p0/Z, [x21, #3, MUL VL]\n"
      "ld1h { z20.h }, p3/Z, [x20]\n"
      "ld1h { z21.h }, p2/Z, [x20, #1, MUL VL]\n"
      "ld1h { z22.h }, p1/Z, [x20, #2, MUL VL]\n"
      "ld1h { z23.h }, p0/Z, [x20, #3, MUL VL]\n"
      "b 45f\n"
      "44:"  // Height 4: no accumulate
      "mov z8.b, #0x0\n"
      "mov z9.b, #0x0\n"
      "mov z10.b, #0x0\n"
      "mov z11.b, #0x0\n"
      "mov z12.b, #0x0\n"
      "mov z13.b, #0x0\n"
      "mov z14.b, #0x0\n"
      "mov z15.b, #0x0\n"
      "mov z16.b, #0x0\n"
      "mov z17.b, #0x0\n"
      "mov z18.b, #0x0\n"
      "mov z19.b, #0x0\n"
      "mov z20.b, #0x0\n"
      "mov z21.b, #0x0\n"
      "mov z22.b, #0x0\n"
      "mov z23.b, #0x0\n"
      "45:"  // Height 4: setup done
      "mov x28, #0x0\n"
      "46:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 47f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "cbnz x28, 48f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "add x24, x24, x20, LSL #1\n"
      "add x23, x23, x20, LSL #1\n"
      "b 48f\n"
      "47:"  // Height 4: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "add x24, x25, x21, LSL #1\n"
      "add x23, x24, x21, LSL #1\n"
      "48:"  // Height 4: input setup done
      "subs x27, x27, #0x1\n"
      "ld1rh { z0.h }, p4/Z, [x26]\n"
      "ld1rh { z1.h }, p4/Z, [x25]\n"
      "ld1rh { z2.h }, p4/Z, [x24]\n"
      "ld1rh { z3.h }, p4/Z, [x23]\n"
      "ld1h { z6.h }, p4/Z, [x12]\n"
      "ld1h { z7.h }, p4/Z, [x11]\n"
      "ble 50f\n"
      "49:"  // Height 4: Multiply loop: Main loop
      "addvl x12, x12, #1\n"
      "fmla z8.h, p4/M, z6.h, z0.h\n"
      "fmla z12.h, p4/M, z6.h, z1.h\n"
      "addvl x11, x11, #1\n"
      "fmla z16.h, p4/M, z6.h, z2.h\n"
      "fmla z20.h, p4/M, z6.h, z3.h\n"
      "ld1h { z25.h }, p4/Z, [x10]\n"
      "add x26, x26, #0x2\n"
      "fmla z9.h, p4/M, z7.h, z0.h\n"
      "fmla z13.h, p4/M, z7.h, z1.h\n"
      "subs x27, x27, #0x1\n"
      "add x25, x25, #0x2\n"
      "fmla z17.h, p4/M, z7.h, z2.h\n"
      "fmla z21.h, p4/M, z7.h, z3.h\n"
      "ld1h { z24.h }, p4/Z, [x9]\n"
      "add x24, x24, #0x2\n"
      "add x23, x23, #0x2\n"
      "fmla z10.h, p4/M, z25.h, z0.h\n"
      "fmla z14.h, p4/M, z25.h, z1.h\n"
      "addvl x10, x10, #1\n"
      "addvl x9, x9, #1\n"
      "fmla z18.h, p4/M, z25.h, z2.h\n"
      "fmla z22.h, p4/M, z25.h, z3.h\n"
      "ld1h { z6.h }, p4/Z, [x12]\n"
      "fmla z11.h, p4/M, z24.h, z0.h\n"
      "fmla z15.h, p4/M, z24.h, z1.h\n"
      "ld1rh { z0.h }, p4/Z, [x26]\n"
      "ld1rh { z1.h }, p4/Z, [x25]\n"
      "fmla z19.h, p4/M, z24.h, z2.h\n"
      "fmla z23.h, p4/M, z24.h, z3.h\n"
      "ld1rh { z2.h }, p4/Z, [x24]\n"
      "ld1rh { z3.h }, p4/Z, [x23]\n"
      "ld1h { z7.h }, p4/Z, [x11]\n"
      "bgt 49b\n"
      "50:"  // Height 4: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "fmla z8.h, p4/M, z6.h, z0.h\n"
      "fmla z12.h, p4/M, z6.h, z1.h\n"
      "add x28, x28, #0x1\n"
      "fmla z16.h, p4/M, z6.h, z2.h\n"
      "fmla z20.h, p4/M, z6.h, z3.h\n"
      "ld1h { z25.h }, p4/Z, [x10]\n"
      "addvl x12, x12, #1\n"
      "fmla z9.h, p4/M, z7.h, z0.h\n"
      "fmla z13.h, p4/M, z7.h, z1.h\n"
      "addvl x11, x11, #1\n"
      "addvl x10, x10, #1\n"
      "fmla z17.h, p4/M, z7.h, z2.h\n"
      "fmla z21.h, p4/M, z7.h, z3.h\n"
      "ld1h { z24.h }, p4/Z, [x9]\n"
      "cmp x28, x20\n"
      "addvl x9, x9, #1\n"
      "fmla z10.h, p4/M, z25.h, z0.h\n"
      "fmla z14.h, p4/M, z25.h, z1.h\n"
      "fmla z18.h, p4/M, z25.h, z2.h\n"
      "fmla z22.h, p4/M, z25.h, z3.h\n"
      "fmla z11.h, p4/M, z24.h, z0.h\n"
      "fmla z15.h, p4/M, z24.h, z1.h\n"
      "fmla z19.h, p4/M, z24.h, z2.h\n"
      "fmla z23.h, p4/M, z24.h, z3.h\n"
      "bne 46b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x26, x13, x20, LSL #1\n"
      "add x25, x26, x20, LSL #1\n"
      "add x24, x25, x20, LSL #1\n"
      "tbz %x[flags], #1, 51f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rh { z25.h }, p4/Z, [x21]\n"
      "ld1rh { z24.h }, p4/Z, [x20]\n"
      "fmin z8.h, p4/M, z8.h, z25.h\n"
      "fmin z9.h, p4/M, z9.h, z25.h\n"
      "fmin z10.h, p4/M, z10.h, z25.h\n"
      "fmin z11.h, p4/M, z11.h, z25.h\n"
      "fmin z12.h, p4/M, z12.h, z25.h\n"
      "fmin z13.h, p4/M, z13.h, z25.h\n"
      "fmin z14.h, p4/M, z14.h, z25.h\n"
      "fmin z15.h, p4/M, z15.h, z25.h\n"
      "fmin z16.h, p4/M, z16.h, z25.h\n"
      "fmin z17.h, p4/M, z17.h, z25.h\n"
      "fmin z18.h, p4/M, z18.h, z25.h\n"
      "fmin z19.h, p4/M, z19.h, z25.h\n"
      "fmin z20.h, p4/M, z20.h, z25.h\n"
      "fmin z21.h, p4/M, z21.h, z25.h\n"
      "fmin z22.h, p4/M, z22.h, z25.h\n"
      "fmin z23.h, p4/M, z23.h, z25.h\n"
      "fmax z8.h, p4/M, z8.h, z24.h\n"
      "fmax z9.h, p4/M, z9.h, z24.h\n"
      "fmax z10.h, p4/M, z10.h, z24.h\n"
      "fmax z11.h, p4/M, z11.h, z24.h\n"
      "fmax z12.h, p4/M, z12.h, z24.h\n"
      "fmax z13.h, p4/M, z13.h, z24.h\n"
      "fmax z14.h, p4/M, z14.h, z24.h\n"
      "fmax z15.h, p4/M, z15.h, z24.h\n"
      "fmax z16.h, p4/M, z16.h, z24.h\n"
      "fmax z17.h, p4/M, z17.h, z24.h\n"
      "fmax z18.h, p4/M, z18.h, z24.h\n"
      "fmax z19.h, p4/M, z19.h, z24.h\n"
      "fmax z20.h, p4/M, z20.h, z24.h\n"
      "fmax z21.h, p4/M, z21.h, z24.h\n"
      "fmax z22.h, p4/M, z22.h, z24.h\n"
      "fmax z23.h, p4/M, z23.h, z24.h\n"
      "51:"  // Height 4: No activation
      "st1h { z8.h }, p3, [x13]\n"
      "st1h { z9.h }, p2, [x13, #1, MUL VL]\n"
      "st1h { z10.h }, p1, [x13, #2, MUL VL]\n"
      "st1h { z11.h }, p0, [x13, #3, MUL VL]\n"
      "addvl x13, x13, #4\n"
      "st1h { z12.h }, p3, [x26]\n"
      "st1h { z13.h }, p2, [x26, #1, MUL VL]\n"
      "st1h { z14.h }, p1, [x26, #2, MUL VL]\n"
      "st1h { z15.h }, p0, [x26, #3, MUL VL]\n"
      "st1h { z16.h }, p3, [x25]\n"
      "st1h { z17.h }, p2, [x25, #1, MUL VL]\n"
      "st1h { z18.h }, p1, [x25, #2, MUL VL]\n"
      "st1h { z19.h }, p0, [x25, #3, MUL VL]\n"
      "st1h { z20.h }, p3, [x24]\n"
      "st1h { z21.h }, p2, [x24, #1, MUL VL]\n"
      "st1h { z22.h }, p1, [x24, #2, MUL VL]\n"
      "st1h { z23.h }, p0, [x24, #3, MUL VL]\n"
      "52:"  // Height 4: Writeback done
      "dech x14, ALL, MUL #4\n"
      "cmp x14, XZR\n"
      "bgt 41b\n"
      "b 80f\n"
      "53:"  // Height 5
      "ldr x20, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x15, [%x[args_ptr], %[offsetof_bias]]\n"
      "ldr x14, [%x[args_ptr], %[offsetof_N]]\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "54:"  // Height 5: Column loop
      "ldr x12, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_stride]]\n"
      "cnth x21, ALL, MUL #3\n"
      "add x11, x12, x20, LSL #1\n"
      "add x10, x11, x20, LSL #1\n"
      "add x9, x10, x20, LSL #1\n"
      "add x20, x9, x20, LSL #1\n"
      "cmp x14, x21\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "bgt 55f\n"
      "dech x21\n"
      "mov x9, x12\n"
      "cmp x14, x21\n"
      "bgt 55f\n"
      "dech x21\n"
      "mov x10, x12\n"
      "cmp x14, x21\n"
      "bgt 55f\n"
      "mov x11, x12\n"
      "55:"  // Height 5: B setup done
      "mov x20, #0x0\n"
      "whilelt p3.h, x20, x14\n"
      "inch x20\n"
      "whilelt p2.h, x20, x14\n"
      "inch x20\n"
      "whilelt p1.h, x20, x14\n"
      "inch x20\n"
      "whilelt p0.h, x20, x14\n"
      "cbz x15, 56f\n"
      "ld1h { z8.h }, p4/Z, [x15]\n"
      "ld1h { z9.h }, p4/Z, [x15, #1, MUL VL]\n"
      "ld1h { z10.h }, p4/Z, [x15, #2, MUL VL]\n"
      "ld1h { z11.h }, p4/Z, [x15, #3, MUL VL]\n"
      "addvl x15, x15, #4\n"
      "mov z12.d, z8.d\n"
      "mov z13.d, z9.d\n"
      "mov z16.d, z8.d\n"
      "mov z17.d, z9.d\n"
      "mov z20.d, z8.d\n"
      "mov z14.d, z10.d\n"
      "mov z15.d, z11.d\n"
      "mov z18.d, z10.d\n"
      "mov z19.d, z11.d\n"
      "mov z21.d, z9.d\n"
      "mov z22.d, z10.d\n"
      "mov z23.d, z11.d\n"
      "mov z24.d, z8.d\n"
      "mov z25.d, z9.d\n"
      "mov z26.d, z10.d\n"
      "mov z27.d, z11.d\n"
      "b 58f\n"
      "56:"  // Height 5: no bias
      "tbz %x[flags], #0, 57f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1h { z8.h }, p3/Z, [x13]\n"
      "ld1h { z9.h }, p2/Z, [x13, #1, MUL VL]\n"
      "ld1h { z10.h }, p1/Z, [x13, #2, MUL VL]\n"
      "ld1h { z11.h }, p0/Z, [x13, #3, MUL VL]\n"
      "add x23, x13, x20, LSL #1\n"
      "add x22, x23, x20, LSL #1\n"
      "add x21, x22, x20, LSL #1\n"
      "ld1h { z12.h }, p3/Z, [x23]\n"
      "ld1h { z13.h }, p2/Z, [x23, #1, MUL VL]\n"
      "add x20, x21, x20, LSL #1\n"
      "ld1h { z14.h }, p1/Z, [x23, #2, MUL VL]\n"
      "ld1h { z15.h }, p0/Z, [x23, #3, MUL VL]\n"
      "ld1h { z16.h }, p3/Z, [x22]\n"
      "ld1h { z17.h }, p2/Z, [x22, #1, MUL VL]\n"
      "ld1h { z18.h }, p1/Z, [x22, #2, MUL VL]\n"
      "ld1h { z19.h }, p0/Z, [x22, #3, MUL VL]\n"
      "ld1h { z20.h }, p3/Z, [x21]\n"
      "ld1h { z21.h }, p2/Z, [x21, #1, MUL VL]\n"
      "ld1h { z22.h }, p1/Z, [x21, #2, MUL VL]\n"
      "ld1h { z23.h }, p0/Z, [x21, #3, MUL VL]\n"
      "ld1h { z24.h }, p3/Z, [x20]\n"
      "ld1h { z25.h }, p2/Z, [x20, #1, MUL VL]\n"
      "ld1h { z26.h }, p1/Z, [x20, #2, MUL VL]\n"
      "ld1h { z27.h }, p0/Z, [x20, #3, MUL VL]\n"
      "b 58f\n"
      "57:"  // Height 5: no accumulate
      "mov z8.b, #0x0\n"
      "mov z9.b, #0x0\n"
      "mov z10.b, #0x0\n"
      "mov z11.b, #0x0\n"
      "mov z12.b, #0x0\n"
      "mov z13.b, #0x0\n"
      "mov z14.b, #0x0\n"
      "mov z15.b, #0x0\n"
      "mov z16.b, #0x0\n"
      "mov z17.b, #0x0\n"
      "mov z18.b, #0x0\n"
      "mov z19.b, #0x0\n"
      "mov z20.b, #0x0\n"
      "mov z21.b, #0x0\n"
      "mov z22.b, #0x0\n"
      "mov z23.b, #0x0\n"
      "mov z24.b, #0x0\n"
      "mov z25.b, #0x0\n"
      "mov z26.b, #0x0\n"
      "mov z27.b, #0x0\n"
      "58:"  // Height 5: setup done
      "mov x28, #0x0\n"
      "59:"  // Height 5: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 60f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "cbnz x28, 61f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "add x24, x24, x20, LSL #1\n"
      "add x23, x23, x20, LSL #1\n"
      "add x22, x22, x20, LSL #1\n"
      "b 61f\n"
      "60:"  // Height 5: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "add x24, x25, x21, LSL #1\n"
      "add x23, x24, x21, LSL #1\n"
      "add x22, x23, x21, LSL #1\n"
      "61:"  // Height 5: input setup done
      "subs x27, x27, #0x1\n"
      "ld1rh { z0.h }, p4/Z, [x26]\n"
      "ld1rh { z1.h }, p4/Z, [x25]\n"
      "ld1rh { z2.h }, p4/Z, [x24]\n"
      "ld1rh { z3.h }, p4/Z, [x23]\n"
      "ld1rh { z4.h }, p4/Z, [x22]\n"
      "ld1h { z6.h }, p4/Z, [x12]\n"
      "ld1h { z7.h }, p4/Z, [x11]\n"
      "ble 63f\n"
      "62:"  // Height 5: Multiply loop: Main loop
      "fmla z8.h, p4/M, z6.h, z0.h\n"
      "fmla z12.h, p4/M, z6.h, z1.h\n"
      "addvl x12, x12, #1\n"
      "addvl x11, x11, #1\n"
      "fmla z16.h, p4/M, z6.h, z2.h\n"
      "fmla z20.h, p4/M, z6.h, z3.h\n"
      "add x26, x26, #0x2\n"
      "subs x27, x27, #0x1\n"
      "fmla z24.h, p4/M, z6.h, z4.h\n"
      "ld1h { z29.h }, p4/Z, [x10]\n"
      "fmla z9.h, p4/M, z7.h, z0.h\n"
      "add x25, x25, #0x2\n"
      "fmla z13.h, p4/M, z7.h, z1.h\n"
      "fmla z17.h, p4/M, z7.h, z2.h\n"
      "add x24, x24, #0x2\n"
      "add x23, x23, #0x2\n"
      "fmla z21.h, p4/M, z7.h, z3.h\n"
      "fmla z25.h, p4/M, z7.h, z4.h\n"
      "ld1h { z28.h }, p4/Z, [x9]\n"
      "add x22, x22, #0x2\n"
      "addvl x10, x10, #1\n"
      "addvl x9, x9, #1\n"
      "fmla z10.h, p4/M, z29.h, z0.h\n"
      "fmla z14.h, p4/M, z29.h, z1.h\n"
      "fmla z18.h, p4/M, z29.h, z2.h\n"
      "fmla z22.h, p4/M, z29.h, z3.h\n"
      "fmla z26.h, p4/M, z29.h, z4.h\n"
      "ld1h { z6.h }, p4/Z, [x12]\n"
      "fmla z11.h, p4/M, z28.h, z0.h\n"
      "ld1rh { z0.h }, p4/Z, [x26]\n"
      "fmla z15.h, p4/M, z28.h, z1.h\n"
      "ld1rh { z1.h }, p4/Z, [x25]\n"
      "fmla z19.h, p4/M, z28.h, z2.h\n"
      "ld1rh { z2.h }, p4/Z, [x24]\n"
      "fmla z23.h, p4/M, z28.h, z3.h\n"
      "ld1rh { z3.h }, p4/Z, [x23]\n"
      "fmla z27.h, p4/M, z28.h, z4.h\n"
      "ld1rh { z4.h }, p4/Z, [x22]\n"
      "ld1h { z7.h }, p4/Z, [x11]\n"
      "bgt 62b\n"
      "63:"  // Height 5: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "fmla z8.h, p4/M, z6.h, z0.h\n"
      "fmla z12.h, p4/M, z6.h, z1.h\n"
      "add x28, x28, #0x1\n"
      "fmla z16.h, p4/M, z6.h, z2.h\n"
      "fmla z20.h, p4/M, z6.h, z3.h\n"
      "addvl x12, x12, #1\n"
      "addvl x11, x11, #1\n"
      "fmla z24.h, p4/M, z6.h, z4.h\n"
      "ld1h { z29.h }, p4/Z, [x10]\n"
      "fmla z9.h, p4/M, z7.h, z0.h\n"
      "addvl x10, x10, #1\n"
      "fmla z13.h, p4/M, z7.h, z1.h\n"
      "fmla z17.h, p4/M, z7.h, z2.h\n"
      "cmp x28, x20\n"
      "fmla z21.h, p4/M, z7.h, z3.h\n"
      "fmla z25.h, p4/M, z7.h, z4.h\n"
      "ld1h { z28.h }, p4/Z, [x9]\n"
      "addvl x9, x9, #1\n"
      "fmla z10.h, p4/M, z29.h, z0.h\n"
      "fmla z14.h, p4/M, z29.h, z1.h\n"
      "fmla z18.h, p4/M, z29.h, z2.h\n"
      "fmla z22.h, p4/M, z29.h, z3.h\n"
      "fmla z26.h, p4/M, z29.h, z4.h\n"
      "fmla z11.h, p4/M, z28.h, z0.h\n"
      "fmla z15.h, p4/M, z28.h, z1.h\n"
      "fmla z19.h, p4/M, z28.h, z2.h\n"
      "fmla z23.h, p4/M, z28.h, z3.h\n"
      "fmla z27.h, p4/M, z28.h, z4.h\n"
      "bne 59b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x26, x13, x20, LSL #1\n"
      "add x25, x26, x20, LSL #1\n"
      "add x24, x25, x20, LSL #1\n"
      "add x23, x24, x20, LSL #1\n"
      "tbz %x[flags], #1, 64f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rh { z29.h }, p4/Z, [x21]\n"
      "ld1rh { z28.h }, p4/Z, [x20]\n"
      "fmin z8.h, p4/M, z8.h, z29.h\n"
      "fmin z9.h, p4/M, z9.h, z29.h\n"
      "fmin z10.h, p4/M, z10.h, z29.h\n"
      "fmin z11.h, p4/M, z11.h, z29.h\n"
      "fmin z12.h, p4/M, z12.h, z29.h\n"
      "fmin z13.h, p4/M, z13.h, z29.h\n"
      "fmin z14.h, p4/M, z14.h, z29.h\n"
      "fmin z15.h, p4/M, z15.h, z29.h\n"
      "fmin z16.h, p4/M, z16.h, z29.h\n"
      "fmin z17.h, p4/M, z17.h, z29.h\n"
      "fmin z18.h, p4/M, z18.h, z29.h\n"
      "fmin z19.h, p4/M, z19.h, z29.h\n"
      "fmin z20.h, p4/M, z20.h, z29.h\n"
      "fmin z21.h, p4/M, z21.h, z29.h\n"
      "fmin z22.h, p4/M, z22.h, z29.h\n"
      "fmin z23.h, p4/M, z23.h, z29.h\n"
      "fmin z24.h, p4/M, z24.h, z29.h\n"
      "fmin z25.h, p4/M, z25.h, z29.h\n"
      "fmin z26.h, p4/M, z26.h, z29.h\n"
      "fmin z27.h, p4/M, z27.h, z29.h\n"
      "fmax z8.h, p4/M, z8.h, z28.h\n"
      "fmax z9.h, p4/M, z9.h, z28.h\n"
      "fmax z10.h, p4/M, z10.h, z28.h\n"
      "fmax z11.h, p4/M, z11.h, z28.h\n"
      "fmax z12.h, p4/M, z12.h, z28.h\n"
      "fmax z13.h, p4/M, z13.h, z28.h\n"
      "fmax z14.h, p4/M, z14.h, z28.h\n"
      "fmax z15.h, p4/M, z15.h, z28.h\n"
      "fmax z16.h, p4/M, z16.h, z28.h\n"
      "fmax z17.h, p4/M, z17.h, z28.h\n"
      "fmax z18.h, p4/M, z18.h, z28.h\n"
      "fmax z19.h, p4/M, z19.h, z28.h\n"
      "fmax z20.h, p4/M, z20.h, z28.h\n"
      "fmax z21.h, p4/M, z21.h, z28.h\n"
      "fmax z22.h, p4/M, z22.h, z28.h\n"
      "fmax z23.h, p4/M, z23.h, z28.h\n"
      "fmax z24.h, p4/M, z24.h, z28.h\n"
      "fmax z25.h, p4/M, z25.h, z28.h\n"
      "fmax z26.h, p4/M, z26.h, z28.h\n"
      "fmax z27.h, p4/M, z27.h, z28.h\n"
      "64:"  // Height 5: No activation
      "st1h { z8.h }, p3, [x13]\n"
      "st1h { z9.h }, p2, [x13, #1, MUL VL]\n"
      "st1h { z10.h }, p1, [x13, #2, MUL VL]\n"
      "st1h { z11.h }, p0, [x13, #3, MUL VL]\n"
      "addvl x13, x13, #4\n"
      "st1h { z12.h }, p3, [x26]\n"
      "st1h { z13.h }, p2, [x26, #1, MUL VL]\n"
      "st1h { z14.h }, p1, [x26, #2, MUL VL]\n"
      "st1h { z15.h }, p0, [x26, #3, MUL VL]\n"
      "st1h { z16.h }, p3, [x25]\n"
      "st1h { z17.h }, p2, [x25, #1, MUL VL]\n"
      "st1h { z18.h }, p1, [x25, #2, MUL VL]\n"
      "st1h { z19.h }, p0, [x25, #3, MUL VL]\n"
      "st1h { z20.h }, p3, [x24]\n"
      "st1h { z21.h }, p2, [x24, #1, MUL VL]\n"
      "st1h { z22.h }, p1, [x24, #2, MUL VL]\n"
      "st1h { z23.h }, p0, [x24, #3, MUL VL]\n"
      "st1h { z24.h }, p3, [x23]\n"
      "st1h { z25.h }, p2, [x23, #1, MUL VL]\n"
      "st1h { z26.h }, p1, [x23, #2, MUL VL]\n"
      "st1h { z27.h }, p0, [x23, #3, MUL VL]\n"
      "65:"  // Height 5: Writeback done
      "dech x14, ALL, MUL #4\n"
      "cmp x14, XZR\n"
      "bgt 54b\n"
      "b 80f\n"
      "66:"  // Height 6
      "ldr x20, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x15, [%x[args_ptr], %[offsetof_bias]]\n"
      "mov x21, #0xc\n"
      "ldr x14, [%x[args_ptr], %[offsetof_N]]\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "madd x21, x20, x21, x13\n"
      "str x21, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "67:"  // Height 6: Column loop
      "ldr x12, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "ldr x20, [%x[args_ptr], %[offsetof_B_stride]]\n"
      "cnth x21, ALL, MUL #3\n"
      "add x11, x12, x20, LSL #1\n"
      "add x10, x11, x20, LSL #1\n"
      "add x9, x10, x20, LSL #1\n"
      "add x20, x9, x20, LSL #1\n"
      "cmp x14, x21\n"
      "str x20, [%x[args_ptr], %[offsetof_cur_B_ptr]]\n"
      "bgt 68f\n"
      "dech x21\n"
      "mov x9, x12\n"
      "cmp x14, x21\n"
      "bgt 68f\n"
      "dech x21\n"
      "mov x10, x12\n"
      "cmp x14, x21\n"
      "bgt 68f\n"
      "mov x11, x12\n"
      "68:"  // Height 6: B setup done
      "mov x20, #0x0\n"
      "whilelt p3.h, x20, x14\n"
      "inch x20\n"
      "whilelt p2.h, x20, x14\n"
      "inch x20\n"
      "whilelt p1.h, x20, x14\n"
      "inch x20\n"
      "whilelt p0.h, x20, x14\n"
      "cbz x15, 69f\n"
      "ld1h { z8.h }, p4/Z, [x15]\n"
      "ld1h { z9.h }, p4/Z, [x15, #1, MUL VL]\n"
      "ld1h { z10.h }, p4/Z, [x15, #2, MUL VL]\n"
      "ld1h { z11.h }, p4/Z, [x15, #3, MUL VL]\n"
      "addvl x15, x15, #4\n"
      "mov z12.d, z8.d\n"
      "mov z13.d, z9.d\n"
      "mov z16.d, z8.d\n"
      "mov z17.d, z9.d\n"
      "mov z20.d, z8.d\n"
      "mov z14.d, z10.d\n"
      "mov z15.d, z11.d\n"
      "mov z18.d, z10.d\n"
      "mov z19.d, z11.d\n"
      "mov z21.d, z9.d\n"
      "mov z22.d, z10.d\n"
      "mov z23.d, z11.d\n"
      "mov z24.d, z8.d\n"
      "mov z25.d, z9.d\n"
      "mov z26.d, z10.d\n"
      "mov z27.d, z11.d\n"
      "mov z28.d, z8.d\n"
      "mov z29.d, z9.d\n"
      "mov z30.d, z10.d\n"
      "mov z31.d, z11.d\n"
      "b 71f\n"
      "69:"  // Height 6: no bias
      "tbz %x[flags], #0, 70f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1h { z8.h }, p3/Z, [x13]\n"
      "ld1h { z9.h }, p2/Z, [x13, #1, MUL VL]\n"
      "ld1h { z10.h }, p1/Z, [x13, #2, MUL VL]\n"
      "ld1h { z11.h }, p0/Z, [x13, #3, MUL VL]\n"
      "add x24, x13, x20, LSL #1\n"
      "add x23, x24, x20, LSL #1\n"
      "add x22, x23, x20, LSL #1\n"
      "ld1h { z12.h }, p3/Z, [x24]\n"
      "ld1h { z13.h }, p2/Z, [x24, #1, MUL VL]\n"
      "add x21, x22, x20, LSL #1\n"
      "add x20, x21, x20, LSL #1\n"
      "ld1h { z14.h }, p1/Z, [x24, #2, MUL VL]\n"
      "ld1h { z15.h }, p0/Z, [x24, #3, MUL VL]\n"
      "ld1h { z16.h }, p3/Z, [x23]\n"
      "ld1h { z17.h }, p2/Z, [x23, #1, MUL VL]\n"
      "ld1h { z18.h }, p1/Z, [x23, #2, MUL VL]\n"
      "ld1h { z19.h }, p0/Z, [x23, #3, MUL VL]\n"
      "ld1h { z20.h }, p3/Z, [x22]\n"
      "ld1h { z21.h }, p2/Z, [x22, #1, MUL VL]\n"
      "ld1h { z22.h }, p1/Z, [x22, #2, MUL VL]\n"
      "ld1h { z23.h }, p0/Z, [x22, #3, MUL VL]\n"
      "ld1h { z24.h }, p3/Z, [x21]\n"
      "ld1h { z25.h }, p2/Z, [x21, #1, MUL VL]\n"
      "ld1h { z26.h }, p1/Z, [x21, #2, MUL VL]\n"
      "ld1h { z27.h }, p0/Z, [x21, #3, MUL VL]\n"
      "ld1h { z28.h }, p3/Z, [x20]\n"
      "ld1h { z29.h }, p2/Z, [x20, #1, MUL VL]\n"
      "ld1h { z30.h }, p1/Z, [x20, #2, MUL VL]\n"
      "ld1h { z31.h }, p0/Z, [x20, #3, MUL VL]\n"
      "b 71f\n"
      "70:"  // Height 6: no accumulate
      "mov z8.b, #0x0\n"
      "mov z9.b, #0x0\n"
      "mov z10.b, #0x0\n"
      "mov z11.b, #0x0\n"
      "mov z12.b, #0x0\n"
      "mov z13.b, #0x0\n"
      "mov z14.b, #0x0\n"
      "mov z15.b, #0x0\n"
      "mov z16.b, #0x0\n"
      "mov z17.b, #0x0\n"
      "mov z18.b, #0x0\n"
      "mov z19.b, #0x0\n"
      "mov z20.b, #0x0\n"
      "mov z21.b, #0x0\n"
      "mov z22.b, #0x0\n"
      "mov z23.b, #0x0\n"
      "mov z24.b, #0x0\n"
      "mov z25.b, #0x0\n"
      "mov z26.b, #0x0\n"
      "mov z27.b, #0x0\n"
      "mov z28.b, #0x0\n"
      "mov z29.b, #0x0\n"
      "mov z30.b, #0x0\n"
      "mov z31.b, #0x0\n"
      "71:"  // Height 6: setup done
      "mov x28, #0x0\n"
      "72:"  // Height 6: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 73f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "ldr x21, [x20, #0x28]\n"
      "cbnz x28, 74f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20, LSL #1\n"
      "add x25, x25, x20, LSL #1\n"
      "add x24, x24, x20, LSL #1\n"
      "add x23, x23, x20, LSL #1\n"
      "add x22, x22, x20, LSL #1\n"
      "add x21, x21, x20, LSL #1\n"
      "b 74f\n"
      "73:"  // Height 6: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21, LSL #1\n"
      "add x24, x25, x21, LSL #1\n"
      "add x23, x24, x21, LSL #1\n"
      "add x22, x23, x21, LSL #1\n"
      "add x21, x22, x21, LSL #1\n"
      "74:"  // Height 6: input setup done
      "subs x27, x27, #0x1\n"
      "ld1rh { z0.h }, p4/Z, [x26]\n"
      "ld1rh { z1.h }, p4/Z, [x25]\n"
      "ld1rh { z2.h }, p4/Z, [x24]\n"
      "ld1rh { z3.h }, p4/Z, [x23]\n"
      "ld1rh { z4.h }, p4/Z, [x22]\n"
      "ld1rh { z5.h }, p4/Z, [x21]\n"
      "ld1h { z6.h }, p4/Z, [x12]\n"
      "ld1h { z7.h }, p4/Z, [x11]\n"
      "ble 76f\n"
      "75:"  // Height 6: Multiply loop: Main loop
      "fmla z8.h, p4/M, z6.h, z0.h\n"
      "fmla z12.h, p4/M, z6.h, z1.h\n"
      "addvl x12, x12, #1\n"
      "addvl x11, x11, #1\n"
      "fmla z16.h, p4/M, z6.h, z2.h\n"
      "fmla z20.h, p4/M, z6.h, z3.h\n"
      "add x26, x26, #0x2\n"
      "subs x27, x27, #0x1\n"
      "fmla z24.h, p4/M, z6.h, z4.h\n"
      "fmla z28.h, p4/M, z6.h, z5.h\n"
      "ld1h { z6.h }, p4/Z, [x10]\n"
      "add x25, x25, #0x2\n"
      "fmla z9.h, p4/M, z7.h, z0.h\n"
      "fmla z13.h, p4/M, z7.h, z1.h\n"
      "add x24, x24, #0x2\n"
      "add x23, x23, #0x2\n"
      "fmla z17.h, p4/M, z7.h, z2.h\n"
      "fmla z21.h, p4/M, z7.h, z3.h\n"
      "add x22, x22, #0x2\n"
      "add x21, x21, #0x2\n"
      "fmla z25.h, p4/M, z7.h, z4.h\n"
      "fmla z29.h, p4/M, z7.h, z5.h\n"
      "ld1h { z7.h }, p4/Z, [x9]\n"
      "addvl x10, x10, #1\n"
      "addvl x9, x9, #1\n"
      "fmla z10.h, p4/M, z6.h, z0.h\n"
      "fmla z14.h, p4/M, z6.h, z1.h\n"
      "fmla z18.h, p4/M, z6.h, z2.h\n"
      "fmla z22.h, p4/M, z6.h, z3.h\n"
      "fmla z26.h, p4/M, z6.h, z4.h\n"
      "fmla z30.h, p4/M, z6.h, z5.h\n"
      "ld1h { z6.h }, p4/Z, [x12]\n"
      "fmla z11.h, p4/M, z7.h, z0.h\n"
      "fmla z15.h, p4/M, z7.h, z1.h\n"
      "ld1rh { z0.h }, p4/Z, [x26]\n"
      "ld1rh { z1.h }, p4/Z, [x25]\n"
      "fmla z19.h, p4/M, z7.h, z2.h\n"
      "fmla z23.h, p4/M, z7.h, z3.h\n"
      "ld1rh { z2.h }, p4/Z, [x24]\n"
      "ld1rh { z3.h }, p4/Z, [x23]\n"
      "fmla z27.h, p4/M, z7.h, z4.h\n"
      "fmla z31.h, p4/M, z7.h, z5.h\n"
      "ld1rh { z4.h }, p4/Z, [x22]\n"
      "ld1rh { z5.h }, p4/Z, [x21]\n"
      "ld1h { z7.h }, p4/Z, [x11]\n"
      "bgt 75b\n"
      "76:"  // Height 6: Multiply loop: Main loop skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "fmla z8.h, p4/M, z6.h, z0.h\n"
      "fmla z12.h, p4/M, z6.h, z1.h\n"
      "add x28, x28, #0x1\n"
      "fmla z16.h, p4/M, z6.h, z2.h\n"
      "fmla z20.h, p4/M, z6.h, z3.h\n"
      "addvl x12, x12, #1\n"
      "addvl x11, x11, #1\n"
      "fmla z24.h, p4/M, z6.h, z4.h\n"
      "fmla z28.h, p4/M, z6.h, z5.h\n"
      "ld1h { z6.h }, p4/Z, [x10]\n"
      "addvl x10, x10, #1\n"
      "fmla z9.h, p4/M, z7.h, z0.h\n"
      "fmla z13.h, p4/M, z7.h, z1.h\n"
      "cmp x28, x20\n"
      "fmla z17.h, p4/M, z7.h, z2.h\n"
      "fmla z21.h, p4/M, z7.h, z3.h\n"
      "fmla z25.h, p4/M, z7.h, z4.h\n"
      "fmla z29.h, p4/M, z7.h, z5.h\n"
      "ld1h { z7.h }, p4/Z, [x9]\n"
      "addvl x9, x9, #1\n"
      "fmla z10.h, p4/M, z6.h, z0.h\n"
      "fmla z14.h, p4/M, z6.h, z1.h\n"
      "fmla z18.h, p4/M, z6.h, z2.h\n"
      "fmla z22.h, p4/M, z6.h, z3.h\n"
      "fmla z26.h, p4/M, z6.h, z4.h\n"
      "fmla z30.h, p4/M, z6.h, z5.h\n"
      "fmla z11.h, p4/M, z7.h, z0.h\n"
      "fmla z15.h, p4/M, z7.h, z1.h\n"
      "fmla z19.h, p4/M, z7.h, z2.h\n"
      "fmla z23.h, p4/M, z7.h, z3.h\n"
      "fmla z27.h, p4/M, z7.h, z4.h\n"
      "fmla z31.h, p4/M, z7.h, z5.h\n"
      "bne 72b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "add x26, x13, x20, LSL #1\n"
      "add x25, x26, x20, LSL #1\n"
      "add x24, x25, x20, LSL #1\n"
      "add x23, x24, x20, LSL #1\n"
      "add x22, x23, x20, LSL #1\n"
      "tbz %x[flags], #1, 77f\n"
      "add x21, %x[args_ptr], %[offset_max]\n"
      "add x20, %x[args_ptr], %[offset_min]\n"
      "ld1rh { z1.h }, p4/Z, [x21]\n"
      "ld1rh { z0.h }, p4/Z, [x20]\n"
      "fmin z8.h, p4/M, z8.h, z1.h\n"
      "fmin z9.h, p4/M, z9.h, z1.h\n"
      "fmin z10.h, p4/M, z10.h, z1.h\n"
      "fmin z11.h, p4/M, z11.h, z1.h\n"
      "fmin z12.h, p4/M, z12.h, z1.h\n"
      "fmin z13.h, p4/M, z13.h, z1.h\n"
      "fmin z14.h, p4/M, z14.h, z1.h\n"
      "fmin z15.h, p4/M, z15.h, z1.h\n"
      "fmin z16.h, p4/M, z16.h, z1.h\n"
      "fmin z17.h, p4/M, z17.h, z1.h\n"
      "fmin z18.h, p4/M, z18.h, z1.h\n"
      "fmin z19.h, p4/M, z19.h, z1.h\n"
      "fmin z20.h, p4/M, z20.h, z1.h\n"
      "fmin z21.h, p4/M, z21.h, z1.h\n"
      "fmin z22.h, p4/M, z22.h, z1.h\n"
      "fmin z23.h, p4/M, z23.h, z1.h\n"
      "fmin z24.h, p4/M, z24.h, z1.h\n"
      "fmin z25.h, p4/M, z25.h, z1.h\n"
      "fmin z26.h, p4/M, z26.h, z1.h\n"
      "fmin z27.h, p4/M, z27.h, z1.h\n"
      "fmin z28.h, p4/M, z28.h, z1.h\n"
      "fmin z29.h, p4/M, z29.h, z1.h\n"
      "fmin z30.h, p4/M, z30.h, z1.h\n"
      "fmin z31.h, p4/M, z31.h, z1.h\n"
      "fmax z8.h, p4/M, z8.h, z0.h\n"
      "fmax z9.h, p4/M, z9.h, z0.h\n"
      "fmax z10.h, p4/M, z10.h, z0.h\n"
      "fmax z11.h, p4/M, z11.h, z0.h\n"
      "fmax z12.h, p4/M, z12.h, z0.h\n"
      "fmax z13.h, p4/M, z13.h, z0.h\n"
      "fmax z14.h, p4/M, z14.h, z0.h\n"
      "fmax z15.h, p4/M, z15.h, z0.h\n"
      "fmax z16.h, p4/M, z16.h, z0.h\n"
      "fmax z17.h, p4/M, z17.h, z0.h\n"
      "fmax z18.h, p4/M, z18.h, z0.h\n"
      "fmax z19.h, p4/M, z19.h, z0.h\n"
      "fmax z20.h, p4/M, z20.h, z0.h\n"
      "fmax z21.h, p4/M, z21.h, z0.h\n"
      "fmax z22.h, p4/M, z22.h, z0.h\n"
      "fmax z23.h, p4/M, z23.h, z0.h\n"
      "fmax z24.h, p4/M, z24.h, z0.h\n"
      "fmax z25.h, p4/M, z25.h, z0.h\n"
      "fmax z26.h, p4/M, z26.h, z0.h\n"
      "fmax z27.h, p4/M, z27.h, z0.h\n"
      "fmax z28.h, p4/M, z28.h, z0.h\n"
      "fmax z29.h, p4/M, z29.h, z0.h\n"
      "fmax z30.h, p4/M, z30.h, z0.h\n"
      "fmax z31.h, p4/M, z31.h, z0.h\n"
      "77:"  // Height 6: No activation
      "st1h { z8.h }, p3, [x13]\n"
      "st1h { z9.h }, p2, [x13, #1, MUL VL]\n"
      "st1h { z10.h }, p1, [x13, #2, MUL VL]\n"
      "st1h { z11.h }, p0, [x13, #3, MUL VL]\n"
      "addvl x13, x13, #4\n"
      "st1h { z12.h }, p3, [x26]\n"
      "st1h { z13.h }, p2, [x26, #1, MUL VL]\n"
      "st1h { z14.h }, p1, [x26, #2, MUL VL]\n"
      "st1h { z15.h }, p0, [x26, #3, MUL VL]\n"
      "st1h { z16.h }, p3, [x25]\n"
      "st1h { z17.h }, p2, [x25, #1, MUL VL]\n"
      "st1h { z18.h }, p1, [x25, #2, MUL VL]\n"
      "st1h { z19.h }, p0, [x25, #3, MUL VL]\n"
      "st1h { z20.h }, p3, [x24]\n"
      "st1h { z21.h }, p2, [x24, #1, MUL VL]\n"
      "st1h { z22.h }, p1, [x24, #2, MUL VL]\n"
      "st1h { z23.h }, p0, [x24, #3, MUL VL]\n"
      "st1h { z24.h }, p3, [x23]\n"
      "st1h { z25.h }, p2, [x23, #1, MUL VL]\n"
      "st1h { z26.h }, p1, [x23, #2, MUL VL]\n"
      "st1h { z27.h }, p0, [x23, #3, MUL VL]\n"
      "st1h { z28.h }, p3, [x22]\n"
      "st1h { z29.h }, p2, [x22, #1, MUL VL]\n"
      "st1h { z30.h }, p1, [x22, #2, MUL VL]\n"
      "st1h { z31.h }, p0, [x22, #3, MUL VL]\n"
      "78:"  // Height 6: Writeback done
      "dech x14, ALL, MUL #4\n"
      "cmp x14, XZR\n"
      "bgt 67b\n"
      "subs %x[M], %x[M], #0x6\n"
      "beq 80f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 79f\n"
      "add x21, x21, #0x6\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "79:"  // Update direct input
      "mov x20, #0xc\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "80:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr)
      : [args_ptr] "r" (&ka), [flags] "r" (flags), [offset_max] "I" (offsetof(KernelArgs, maxval)), [offset_min] "I" (offsetof(KernelArgs, minval)), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_B_stride] "I" (offsetof(KernelArgs, B_stride)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_bias] "I" (offsetof(KernelArgs, bias)), [offsetof_cur_B_ptr] "I" (offsetof(KernelArgs, cur_B_ptr)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_output_ptr] "I" (offsetof(KernelArgs, output_ptr)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths))
      : "cc", "memory", "p0", "p1", "p2", "p3", "p4", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
    );
}

} // namespace arm_gemm
#endif  // ARM_COMPUTE_ENABLE_SVE
